/* Copyright (C) 2003 Reliable Software Group
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: smart_profile.cpp,v 1.6 2004/03/12 20:44:21 dhm Exp $ */

#include <string>
#include <math.h>

#include <anomaly.h>

#include "profile_int.h"
#include "bayes.h"

#define SUSPICIOUS 0.9

int map_to_level(double val)
{
  /* map a double in the interval (0,1) into 5 levels [0-4] */
  if ((val < 0.0) || (val > 1.0))
    throw ModelConsistencyException("SmartProfile: mapping failed: value not in interval [0,1]");

  int val_level;
  
  if (val < 0.2)
    val_level = 0;
  else if (val < 0.4)
    val_level = 1;
  else if (val < 0.6)
    val_level = 2;
  else if (val < 0.8)
    val_level = 3;
  else 
    val_level = 4;

  return val_level;
}

GBUSmartProfile::GBUSmartProfile(BayesianNetwork *bn)
{
  _good = BayesianNetworkModel::instance(bn, "Classification", "Regular");
  _bad = BayesianNetworkModel::instance(bn, "Classification", "Attack");
  _ugly = BayesianNetworkModel::instance(bn, "Classification", "Abnormal");

  _debug = false;
}

double GBUSmartProfile::check_item(ListCollection *features)
{
  ListCollection::iterator i, j;
  FeatureVector::iterator feature;

  // no models in the profile, return non-anomaly
  if (_modelFeatureMappings.size() == 0) 
    return 1;

  // copy FeatureVector contents to a vector for random access
  // FIX ... very ugly
  vector<Item *> fvCopy;
  for (i = features->begin(); i != features->end(); i++) 
    fvCopy.push_back(*i);

  /* list to store evidence for the Bayes net */
  ListCollection *evidence = new ListCollection();

  /* cycle through all individual features and models */
  for (i = _modelFeatureMappings.begin(); i != _modelFeatureMappings.end(); 
       i++) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    assert(mfm);
    unsigned int numFeaturesRequested = mfm->getFeatures()->size();

    Model *model = mfm->getModel();
    Item *to_check;
    ListCollection *featureList = new ListCollection();

    // compute the input for this model
    if (numFeaturesRequested == 0) {
      // no features requested, so skip this model
      continue;
    }
    else if (numFeaturesRequested == 1) {
      // single feature, don't build a list
      IntegerItem *index = 
	dynamic_cast<IntegerItem *>(mfm->getFeatures()->front());

      assert(index);
      assert(index->value() <= fvCopy.size());
      to_check = fvCopy[index->value()];
    }
    else {
      // build a list of the requested features
      for (j = mfm->getFeatures()->begin(); j != mfm->getFeatures()->end(); 
	   j++) {
	IntegerItem *index = dynamic_cast<IntegerItem *>(*j);
	assert(index);
	assert(index->value() <= fvCopy.size());
	featureList->push_back(fvCopy[index->value()]);
      }

      to_check = featureList;
    }

    double result = model->check_item(to_check);
    double epsilon = 1E-6;
	
    // make sure evaluationResult is in the open interval (0,1)
    if (fabs(result - 0.0) < epsilon) {
      result = (0.0 + epsilon);
    }
    
    if (fabs(result - 1.0) < epsilon) {
      result = (1.0 - epsilon);
    }

    /* 
     *  prepare new evidence for the Bayes net - beware of hacks! 
     */
    BayesianEvidence *e;

    /*
     * first, get the level of suspicion and name of corresponding Bayes 
     * node
     */    
    string model_name = model->get_name();
    int model_level;

    if ((model_name.find("Token") != string::npos) || 
	(model_name.find("Structure") != string::npos)) {
      if (result < 0.1)
	model_level = 1;
      else
	model_level = 0;
    }
    else
      model_level = map_to_level(1.0 - result);

    e = new BayesianEvidence(model_name, model_level);
    evidence->push_back(e);

    /* second, get the level of the confidence and name of corresponding Bayes node */
    string conf_name = model_name + "Confidence";
    int conf_level;

    if (model_name.find("Token") != string::npos) {
      if (model->get_confidence() < 0.1)
	conf_level = 1;
      else
	conf_level = 0;
    }
    else
      conf_level = map_to_level(1.0 - model->get_confidence());

    e = new BayesianEvidence(conf_name, conf_level);
    evidence->push_back(e);
  
    /* delete list */
    delete featureList;
  }

  /* get the probability of an attack */
  double good = _good->check_item(evidence); 

  if (_debug)
    cout << "Bayesian network returned (good) --->>> " << good << "\n"; 

  /* HERE THE PROBABILITY IS RETURNED IMMEDIATELY */

  evidence->release();
  return good;

  if (good < SUSPICIOUS) {
    double attack = _bad->check_item(evidence);
    double misconfig = _ugly->check_item(evidence);

    if (_debug) {
      cout << "Bayesian network returned (bad) --->>> " << attack << "\n"; 
      cout << "Bayesian network returned (ugly) --->>> " << misconfig << "\n"; 
    }

    /* clean up and return final decision */
    evidence->release();
    if (attack > misconfig)
      return 1.0;
    else
      return 0.0;
  }
  else {
    /* clean up and return final decision */
    evidence->release();
    return 0.0;
  }

}


 
