/* Copyright (C) 2003 Reliable Software Group
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: profile.cpp,v 1.15 2004/10/20 00:35:41 dhm Exp $ */

#include <stdlib.h>
#include <math.h>
#include <assert.h>

#include <anomaly.h>
#include "profile_int.h"


ProfileImpl::ProfileImpl(void) {
  _debug = false;
}

void ProfileImpl::addModelFeatureMapping(Model *m, int feature) {
  assert(feature >= 0);

  ListCollection *l = new ListCollection();
  l->push_back(new IntegerItem(feature));

  this->addModelFeatureMapping(m, l);
}

void 
ProfileImpl::addModelFeatureMapping(Model *m, ListCollection *featureSet) {
  assert(m);
  assert(featureSet);

  _modelFeatureMappings.push_back(new ModelFeatureMapping(m, featureSet));
}


ListCollection *
ProfileImpl::_getEvaluationResults(ListCollection *fv, 
				   ListCollection *evaluationResults) {
  ListCollection::iterator i, j;
  FeatureVector::iterator feature;

  if (evaluationResults == 0x0)
    evaluationResults = new ListCollection();

  evaluationResults->clear();

  if (_modelFeatureMappings.size() == 0) 
    return evaluationResults; // no models in the profile, return empty list

  // copy FeatureVector contents to a vector for random access
  vector<Item *> fvCopy;

  for (i = fv->begin(); i != fv->end(); i++) 
    fvCopy.push_back(*i);

  // for each model in _modelFeatureMappings, apply the features in the
  // feature set and store the anomaly score
  for (i = _modelFeatureMappings.begin(); i != _modelFeatureMappings.end();
       i++) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    assert(mfm);
    unsigned int numFeaturesRequested = mfm->getFeatures()->size();

    if (numFeaturesRequested == 0) {
      // no features requested, so skip this model
      continue;
    }
    else if (numFeaturesRequested == 1) {
      IntegerItem *index = 
	dynamic_cast<IntegerItem *>(mfm->getFeatures()->front());
      assert(index);
      assert(index->value() <= fvCopy.size());

      EvaluationResult *evaluationResult = 
	_getEvaluationResult(mfm->getModel(), 
			     fvCopy[index->value()], 
			     new IntegerItem(index->value()));
      
      evaluationResults->push_back(evaluationResult);
    }
    else {
      // build a list of the requested features
      ListCollection *featureList = new ListCollection();
      ListCollection *featureIndexList = new ListCollection();

      assert(featureList);
      assert(featureIndexList);

      for (j = mfm->getFeatures()->begin(); j != mfm->getFeatures()->end();
	   j++) {
	IntegerItem *index = dynamic_cast<IntegerItem *>(*j);
	assert(index);
	assert(index->value() <= fvCopy.size());
	featureList->push_back(fvCopy[index->value()]);
	featureIndexList->push_back(new IntegerItem(index->value()));
      }

      EvaluationResult *evaluationResult = 
	_getEvaluationResult(mfm->getModel(), 
			     featureList,
			     featureIndexList);

      evaluationResults->push_back(evaluationResult);

      featureList->release();
    }
  }

  return evaluationResults;
}


// suppose check_item has an optional parameter that's a pointer to 
// an allocated (to be cleared out) ListCollection
// if this is provided, then the evaluation results are contained in this
// this upon returning. otherwise, the results of the computation are
// discarded.
// instead! make the evalutaionresults an optional parameter to 
// _getevaluationresults(). and still have check_item() and check_item_deep()
double ProfileImpl::check_item_deep(ListCollection *featureVector,
				    ListCollection *evaluationResults) {
  assert(evaluationResults != 0x0);

  _getEvaluationResults(featureVector, evaluationResults);

  assert(evaluationResults);
  
  if (evaluationResults->size() == 0) {
    return 1; // no anomaly
  }
  else {
    double aggregateScore = _computeAggregateScore(evaluationResults);
    
    if (_debug) {
      cout << "ProfileImpl::check_item_deep: aggregateScore = " 
	   << aggregateScore << endl;
    }

    return aggregateScore;
  }
}


double ProfileImpl::check_item(ListCollection *fv) {
  ListCollection *evaluationResults = _getEvaluationResults(fv);

  assert(evaluationResults);

  if (evaluationResults->size() == 0) {
    return 1; // no anomaly
  }
  else {
    double aggregateScore = _computeAggregateScore(evaluationResults);
    
    if (_debug) {
      cout << "ProfileImpl::check_item: aggregateScore = " 
	   << aggregateScore << endl;
    }

    evaluationResults->release();
    
    return aggregateScore;
  }
}

ListCollection *ProfileImpl::
check_item_individual_model_scores(ListCollection *fv) {
  ListCollection *evaluationResults = _getEvaluationResults(fv);

  assert(evaluationResults);

  ListCollection *returnValues = new ListCollection();

  assert(returnValues);

  for (ListCollection::iterator i = evaluationResults->begin(); 
       i != evaluationResults->end(); i++) {
    EvaluationResult *er = dynamic_cast<EvaluationResult *>(*i);
    assert(er);
    returnValues->push_back(er->getModelScore());
  }

  return returnValues;
}


Profile::EvaluationResult *
ProfileImpl::_getEvaluationResult(Model *model, 
				  Item *arg,
				  Item *featureNumber) {
  double epsilon = 1E-6;
  double anomalyScore = model->check_item(arg);
	
  // make sure evaluationResult is in the open interval (0,1)
  if (fabs(anomalyScore - 0.0) < epsilon) {
    anomalyScore = 0.0 + epsilon;
  }

  if (fabs(anomalyScore - 1.0) < epsilon) {
    anomalyScore = 1.0 - epsilon;
  }

  EvaluationResult *evaluationResult = 
    new EvaluationResult(model->get_name(),
			 new DoubleItem(anomalyScore),
			 new DoubleItem(model->get_confidence()),
			 featureNumber);

  return evaluationResult;
}

void ProfileImpl::switch_mode(Model::ModelMode mode) {
  ListCollection::iterator i;
  unsigned int counter;

  if (_debug) {
    cout << "ProfileImpl::switch_mode:" << endl;
  }

  for (i = _modelFeatureMappings.begin(), counter = 0; 
       i != _modelFeatureMappings.end(); 
       i++, counter++) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    assert(mfm);
    assert(mfm->getModel());

    if (_debug) {
      cout << "  switching mode for model #" << counter << endl;
    }

    mfm->getModel()->switch_mode(mode);
  }
}

void ProfileImpl::insert_item(ListCollection *fv) {
  ListCollection::iterator i, j;
  FeatureVector::iterator feature;
  unsigned int counter;

  if (_debug) {
    cout << "ProfileImpl::insert_item:" << endl;
  }

  if(_modelFeatureMappings.size() == 0 ) {
    if (_debug) {
      cout << "  no models in this profile. returning."
	   << endl;
    }

    return; // no models in this profile, so just return
  }

  // copy FeatureVector contents to a vector for random access
  vector<Item *> fvCopy;

  for (i = fv->begin(); i != fv->end(); i++) 
    fvCopy.push_back(*i);

  if (_debug) {
    cout << "  applying " << fv->size()
	 << " features to " << _modelFeatureMappings.size() << " models."
	 << endl;
  }

  // for each model in _modelFeatureMappings, apply the features in the
  // feature set and store the anomaly score
  for (i = _modelFeatureMappings.begin(), counter = 0; 
       i != _modelFeatureMappings.end();
       i++, counter++) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    assert(mfm);
    unsigned int numFeaturesRequested = mfm->getFeatures()->size();

    if (numFeaturesRequested == 0) {
      if (_debug) {
	cout << "  model #" << counter << " requests " << numFeaturesRequested 
	     << " features. continuing." << endl;
      }

      // no features requested, so skip this model
      continue;
    }
    else if (numFeaturesRequested == 1) {
      if (_debug) {
	cout << "  model #" << counter << " requests " << numFeaturesRequested 
	     << " feature. calling insert_item()." << endl;
      }

      IntegerItem *index = 
	dynamic_cast<IntegerItem *>(mfm->getFeatures()->front());
      assert(index);
      assert(index->value() <= fvCopy.size());

      mfm->getModel()->
	insert_item(fvCopy[index->value()]);
    }
    else {
      // build a list of the requested features
      ListCollection *featureList = new ListCollection();

      if (_debug)
	cout << "ProfileImpl::insert_item: model gets feature(s): ";

      for (j = mfm->getFeatures()->begin(); j != mfm->getFeatures()->end();
	   j++) {
	IntegerItem *index = dynamic_cast<IntegerItem *>(*j);
	assert(index);
	assert(index->value() <= fvCopy.size());
	featureList->push_back(fvCopy[index->value()]);

	if (_debug)
	  cout << index->value() << " ";
      }

      if (_debug)
	cout << endl;

      mfm->getModel()->insert_item(featureList);

      featureList->release();
    }
  }
}

// Compute an aggregate score from the results of invoking this model
// collection on the given feature vector. Each evaluation result is
// weighted against the computed confidence in the model.
double ProfileImpl::_computeAggregateScore(ListCollection *results) {
  ListCollection::iterator i;
  ListCollection::iterator j;
  double sum = 0.0;

  assert(results->size() == _modelFeatureMappings.size());

  // compute score for results in evaluationResults, using model confidences
  for (i = results->begin(); i != results->end(); i++) {
    EvaluationResult *evaluationResult = dynamic_cast<EvaluationResult *>(*i);
    assert(evaluationResult);

    sum += evaluationResult->getModelConfidence()->get_value() * (-1.0) * 
      log(evaluationResult->getModelScore()->get_value());
  }

  double aggregateResult;
  if (results->size() == 0) {
    aggregateResult = 0.0;
  }
  else {
    aggregateResult = sum / ((double) results->size());
  }

  return aggregateResult;
}


void ProfileImpl::toString(void) {
  ListCollection::iterator i, j;

  cout << "ProfileImpl::toString:" << endl;

  cout << "  model/NULL model  | mapped features" << endl;

  for (i = _modelFeatureMappings.begin(); i != _modelFeatureMappings.end(); 
       i++) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    assert(mfm);

    if (mfm->getModel() == 0x0) {
      cout << "  NULL model       <- ";
    }
    else {
      cout << "  model            <- ";
    }

    for (j = mfm->getFeatures()->begin(); j != mfm->getFeatures()->end();
	 j++) {
      IntegerItem *intItem = dynamic_cast<IntegerItem *>(*j);
      assert(intItem);
      cout << intItem->value() << " ";
    }

    cout << endl;
  }
}

Model *ProfileImpl::get_model(unsigned int index) {
  ListCollection::iterator i;
  unsigned int counter;

  for (i = _modelFeatureMappings.begin(), counter = 0; 
       i != _modelFeatureMappings.end();
       i++, counter++) {
    if (counter == index) {
      ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
      assert(mfm);
      return mfm->getModel();
    }
  }

  return 0x0;
}

unsigned int ProfileImpl::get_number_of_models(void) {
  return _modelFeatureMappings.size();
}


Profile *Profile::instance(void) {
  return new ProfileImpl();
}

Profile *Profile::smart_instance(string bayesNetFilename,
				 string smartProfileClassName) {
#ifdef HAVE_SMILE
  /* load the Bayesian network from file with name s */
  BayesianNetwork *bn = BayesianNetwork::instance(bayesNetFilename);

  /* instantiate the smart network */
  if (smartProfileClassName == "GBUSmartProfile") {
    return new GBUSmartProfile(bn);
  }
  else {
    cerr << "Profile::smart_instance: don't know how to handle class name '"
	 << smartProfileClassName << "'. Returning null." << endl;
    return NULL;
  }
#else
  cerr << "Profile::smart_instance: Smile library isn't installed! Returning"
       << " null." << endl;

  return NULL;
#endif
}


ModelFeatureMapping::ModelFeatureMapping(Model *m, ListCollection *features) {
  _model = m;
  _features = features;
}

size_t ModelFeatureMapping::hash_value(void) const {
  ListCollection::iterator i;
  size_t hashVal = (size_t) _model;
  unsigned int counter;

  for (i = _features->begin(), counter = 0; i != _features->end(); 
       i++, counter++) {
    IntegerItem *index = dynamic_cast<IntegerItem *>(*i);
    assert(index);
    
    unsigned int numBitsToShift = 
      (counter % sizeof(size_t)) / sizeof(char) * 8;

    hashVal ^= index->value() << numBitsToShift;
  }
  
  return hashVal;
}

Model *ModelFeatureMapping::getModel(void) {
  return _model;
}

ListCollection *ModelFeatureMapping::getFeatures(void) {
  return _features;
}

void Profile::check()
{
  return;
}

void Profile::set_debug(bool debug) {
  _debug = debug;
}


Profile::EvaluationResult::EvaluationResult(string modelName, 
					    DoubleItem *modelScore,
					    DoubleItem *modelConfidence,
					    Item *featureNumber) {
  _modelName = modelName;
  _modelScore = modelScore;
  _modelConfidence = modelConfidence;
  _featureNumber = featureNumber;
}


#ifdef __undef
void SmartProfile::check()
{
  ListCollection::iterator i;

  cout << "------------ Smart check ----------\n";
  for (i = _modelFeatureMappings.begin(); i !=  _modelFeatureMappings.end(); ++i) {
    ModelFeatureMapping *mfm = dynamic_cast<ModelFeatureMapping *>(*i);
    cout << "Name: " << mfm->getModel()->get_name() << "\n";
  }

  cout << "************* Smart check End **************\n";
}
#endif
