#ifndef MARKOV_H__
#define MARKOV_H__

/* Copyright (C) 2003 Reliable Software Group 
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* $Id: markov.h,v 1.16 2003/03/07 22:58:59 chris Exp $ */

#include <iostream>
#include <string>
#include <map>

#include "anomaly.h"


/* forward declarations */
class ModelTrans;
class ModelState;
class HmmImpl;


typedef __gnu_cxx::hash_map<Item *, unsigned int>::iterator EmissionIter;
typedef __gnu_cxx::hash_map<unsigned int, ModelState *>::iterator StateIter;
typedef __gnu_cxx::hash_map<unsigned int, ModelTrans *>::iterator TransIter;
typedef __gnu_cxx::hash_map<unsigned int, long double>::iterator StepIter;

/* a transition between two states of the automaton */
class ModelTrans {

  /* declare ModelState as friend - this class is a helper for ModelState */
  friend class ModelState;
  friend class HmmImpl;

 private:

  /* pointer to target state */
  ModelState *target;

  /* number of individul transitions to target */
  unsigned int count;

 public:

  ModelTrans(ModelState *t, unsigned int c) {
    target = t;
    count = c;
  }

  /* is this transition pointing to target state */
  bool has_target(ModelState *state);
};



/* a single state of the markov automaton */
class ModelState {

  friend class HmmImpl;

 private:

  /* reference to enclosing model implementation */
  HiddenMarkovModel *model;

  /* unique id to identify state */
  unsigned int id;

  /* total number of emitted letters */
  unsigned int emission_count;

  /* number of emitted items (indexed by items) */
  __gnu_cxx::hash_map<Item *, unsigned int> emissions;

  /* total number of forward (outgoing) transitions */
  unsigned int trans_count;

  /* set of forward (outgoing) transitions indexed by target state id */
  __gnu_cxx::hash_map<unsigned int, ModelTrans *> trans_forward;

  /* set of transitions that point to this state indexed by originating state id */
  __gnu_cxx::hash_map<unsigned int, ModelTrans *> trans_back;

  /* current probabilities */
  long double p_structure, p_likelihood;

  /* probability storage */
  long double last_structure, last_likelihood;

 public:

  /* return unique state id */
  unsigned int get_id();


public:

  /* default constructor */
  ModelState(HiddenMarkovModel *model);

  /* constructor that adds a token to the states emissions */ 
  ModelState(HiddenMarkovModel *model, Item *token);

  /* destructor - delete transitions */
  ~ModelState();

  /* add forward transition */
  void add_forward_edge(ModelState *target, unsigned int count);

  /* subtract forward transition */
  void subtract_forward_edge(ModelState *target, unsigned int count);

  /* add back transition */
  void add_back_edge(ModelState *target, unsigned int count);

  /* subtract back transition */
  void subtract_back_edge(ModelState *target, unsigned int count);

  /* merge the emissions of state 'other' into this one */
  void merge_emissions(ModelState *other);

  /* backtrack (and remove) forward transitions of state 'other' from this one */
  void backtrack_emissions(ModelState *other);
 
  /* merge forward transitions of state 'other' into this one */
  void merge_forward(ModelState *other, set<unsigned int> *);

  /* backtrack (and remove) forward transitions of state 'other' from this one */
  void backtrack_forward(ModelState *other, set<unsigned int> *);

  /* merge back transitions of state 'other' into this one */
  void merge_back(ModelState *other, set<unsigned int> *);

  /* backtrack (and remove) back transitions of state 'other' from this one */
  void backtrack_back(ModelState *other, set<unsigned int> *);

  /* check if a certain input token is emitted in one of the nodes successors */
  ModelState* find_successor(Item *token);

  /* add a token to the states emissions */
  void add_emission(Item *token, unsigned int count);

  /* print a string in dot (graph) format */
  void to_string(ostream & = cout);

  /* get the current structural probability of this state */
  long double get_structural_probability(bool, bool);

  /* get the current likelihood probability of this state */
  long double get_likelihood_probability(bool);

  /* get the last structural probability of this state */
  long double get_last_structural_probability();

  /* get the last likelihood probability of this state */
  long double get_last_likelihood_probability();

  /* reset the current structural probability of this state to its old value */
  long double reset_structural_probability();

  /* reset the current likelihood probability of this state to its old value */
  long double reset_likelihood_probability();

  /* check the consistency of this state */
  void check_consistency();

  void print_state();

};


/* holds the update information of a state merge to undo it later */
class ModelUpdate
{
    /* destination state <-- source state */
    ModelState *s1, *s2;

 public:
    ModelUpdate(ModelState *ms1, ModelState *ms2) : s1(ms1), s2(ms2) { }

    ModelState* get_merge_dest() { return s1; }

    ModelState* get_merge_source() { return s2; }

};


/* hidden markov model - represented as non-deterministic automaton */
class HmmImpl : public HiddenMarkovModel { 

 private:

  /* counter for the Ids of its states */
  int unique;

  /* the single start state of the automaton */
  ModelState *start;

  /* the single terminal state of the automaton */
  ModelState *terminal;
  
  /* the states that are temporary removed during the merging process */
  __gnu_cxx::hash_map<unsigned int, ModelState *> dangling; 

  /* the set of model states - indexed by their id */
  __gnu_cxx::hash_map<unsigned int, ModelState *> states;

  /* the set of letters used in the input alphabet */
  __gnu_cxx::hash_map<Item*, unsigned char> alphabet;
    
  /* the current probability of the markov model */
  long double probability;

  /* a boolean that is set to true when the probability is up-to-date */ 
  bool valid_probability;

  /* a boolean that is set to true when the probability is up-to-date */ 
  bool valid_reduced;

  /* count the number of inserted elements */
  unsigned long _inserted;

  int to_optimize;

 private:

  /* update the probability after a merge w/o re-evaluating the whole model */
  void update_merge_probability(set<unsigned int> *, unsigned int);

  /* update the probability after a backtrack w/o re-evaluating the whole model */
  void update_backtrack_probability(set<unsigned int> *, unsigned int);

  /* evaluate the likelihood of the input given the current structure */
  long double eval_likelihood(bool );

  /* evaluate the probability of the current structure - simpler automatons receive higher probabilities */
  long double eval_structure(bool, bool );

  /* check the consistency of the model */
  void check_consistency();

  /* merges the emissions and transitions of state s2 into state s1 */
  ModelUpdate* merge_states(ModelState *s1, ModelState *s2);

  /* backtracks the emissions and transitions represented by update */
  void backtrack_states(ModelUpdate *update);

  /* finalize a merge - no backtrack is possible as the required data structures are deleted */ 
  void finalize(ModelState *dst, ModelState *src);
  
 public:
  
  /* constructor */
  HmmImpl();
  

  /* destructor */
  ~HmmImpl();

 public:
  
  void print_states();

  /* return the next free id for a model state */
  int get_next_state_id();

  /* get the number of states */
  unsigned int get_model_size();

  /* get the number of letters in input alphabet */
  unsigned int get_alphabet_size();

  /* insert a new sequence of input items into the model */
  void _insert_item(ListCollection &input_sequence);

  /* get the current probability of the model */
  long double get_probability(bool);

  /* get the current probability of the model */
  long double get_probability();

  /* set the probability of the model with a reduced number pf states to effectively calculate updates */
  void set_reduced_probability();

  /* merge states to optimize a-posteriori probability of model with respect to input */
  void optimize();

  /* print a string in dot (graph) format */
  void to_string(ostream & = cout);

  /* insert an item into the model */
  virtual void insert_item(Item *) throw (ModelInputException);
  
  /* switch to different mode */
  virtual void switch_mode(ModelMode);

  /* check an item to accordance with the model */
  virtual double check_item(Item *) throw (ModelInputException);
  
  /* return a confidence value */
  virtual double get_confidence(void);

  /* check an item to accordance with the model */
  virtual long double _check_item(ListCollection *);

  /* module test functions */
  virtual bool module_test(bool);
  
};  


#endif
