/* Copyright (C) 2003 Reliable Software Group 
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: linear_regression.cpp,v 1.1 2003/04/11 23:14:40 chris Exp $ */

#include <stdio.h>

#include <anomaly.h>


LinearRegression::LinearRegression()
{
  _increasing = false;
}

LinearRegression::~LinearRegression(void)
{
}


void LinearRegression::switch_mode(ModelMode mode) throw (ModelException)
{
  if (_mode == Training && mode == Detection) {

    HistogramIter iter;
    HistogramRIter riter;
    
    double mu_x, mu_y, square_x, square_y, dot_xy;
    double sum, N;

    if (_element_count <= 1) {
      _valid = false;
      _mode = mode;
      _left_bound = _right_bound = 0;
      return;
    }

    /* get mean, squares for x values */
    sum = N = square_x = dot_xy = 0.0;
    for (iter = _elements.begin(); iter != _elements.end(); ++iter) {
      sum += (double) iter->first;
      N += 1.0;
      square_x += ((double) iter->first * (double) iter->first);
      dot_xy += ((double) iter->first * (double) iter->second);
    }
    mu_x = sum / N;

    /* get mean, squares for y values */
    sum = N = square_y = 0.0;
    for (iter = _elements.begin(); iter != _elements.end(); ++iter) {
      sum += (double) iter->second;
      N += 1.0;
      square_y += ((double) iter->second * (double) iter->second);
    }
    mu_y = sum / N;

    /* f(x) = a + bx */
    double a, b, zero_point;
    a = ((mu_y * square_x - mu_x * dot_xy) / (square_x - N * mu_x * mu_x));
    b = ((dot_xy - N * mu_x * mu_y) / (square_x - N * mu_x * mu_x));

    if (b != 0) {
      zero_point = -(a/b);

      if (b > 0.0) {
	_increasing = true;
	riter = _elements.rbegin();
	_right_bound = (double) riter->first;
	_left_bound = zero_point;
      }
      else {
	iter = _elements.begin();
	_left_bound = (double) iter->first;
	_right_bound = zero_point;
      }
    }
    else {
	iter = _elements.begin();
	_left_bound = (double) iter->first;
	riter = _elements.rbegin();
	_right_bound = (double) riter->first;
    }      

    if (_left_bound > _right_bound)
      throw ModelException("LinearRegression::switch_mode: bounds constraint violated");

    _mode = mode;
    _valid = true;
  }
  else
    throw ModelException("LinearRegression::switch_mode performs illegal mode transition");
}


double LinearRegression::get_mean(void) throw (ModelException)
{
  return _left_bound + (_right_bound - _left_bound)/2;
}


double LinearRegression::get_variance(void) throw (ModelException)
{
  return (((_right_bound - _left_bound) *  (_right_bound - _left_bound))/ 12);
}


double LinearRegression::f(double x)
{
    if (x > _right_bound)
      return 0.0;
    else if (x < _left_bound)
      return 0.0;

    if ((_right_bound == _left_bound) && (x == _right_bound))
      return 1.0;
    else
      return 1 / (_right_bound - _left_bound);
}


double LinearRegression::F(double x)
{
  double tmp;

  if (x >= _right_bound)
    tmp =  1.0;
  else if (x <= _left_bound)
    tmp = 0.0;
  else 
    tmp = ((x - _left_bound) / (_right_bound - _left_bound));

  if (_increasing)
    return tmp;
  else
    return 1.0 - tmp;
}


bool LinearRegression::test(bool verbose)
{
  LinearRegression lr1;
  ListCollection lc1; 
  lc1.push_back(new ItemCount(new IntegerItem(1), 10));
  lc1.push_back(new ItemCount(new IntegerItem(2), 8));
  lc1.push_back(new ItemCount(new IntegerItem(3), 6));
  lc1.push_back(new ItemCount(new IntegerItem(4), 4));
  lr1.insert_item(&lc1);
  lr1.switch_mode(Detection);
  if (verbose) printf("[%f, %f]\n", lr1._left_bound, lr1._right_bound); 
  if (verbose) printf("%f %f %f %f %f %f\n", lr1.get_mean(), lr1.get_variance(), lr1.f(0.0), lr1.f(2.0), lr1.F(0), lr1.F(5));

  LinearRegression lr2;
  ListCollection lc2; 
  lc2.push_back(new ItemCount(new IntegerItem(1), 4));
  lc2.push_back(new ItemCount(new IntegerItem(2), 6));
  lc2.push_back(new ItemCount(new IntegerItem(3), 8));
  lc2.push_back(new ItemCount(new IntegerItem(4), 10));
  lr2.insert_item(&lc2);
  lr2.switch_mode(Detection);
  if (verbose) printf("[%f, %f]\n", lr2._left_bound, lr2._right_bound); 
  if (verbose) printf("%f %f %f %f %f %f\n", lr2.get_mean(), lr2.get_variance(), lr2.f(0.0), lr2.f(2.0), lr2.F(0), lr2.F(5));

  LinearRegression lr3;
  ListCollection lc3; 
  lc3.push_back(new ItemCount(new IntegerItem(1), 10));
  lc3.push_back(new ItemCount(new IntegerItem(2), 8));
  lc3.push_back(new ItemCount(new IntegerItem(3), 16));
  lc3.push_back(new ItemCount(new IntegerItem(4), 4));
  lc3.push_back(new ItemCount(new IntegerItem(5), 10));
  lc3.push_back(new ItemCount(new IntegerItem(6), 8));
  lc3.push_back(new ItemCount(new IntegerItem(8), 6));
  lc3.push_back(new ItemCount(new IntegerItem(12), 4));
  lc3.push_back(new ItemCount(new IntegerItem(17), 10));
  lc3.push_back(new ItemCount(new IntegerItem(18), 2));
  lc3.push_back(new ItemCount(new IntegerItem(19), 1));
  lc3.push_back(new ItemCount(new IntegerItem(42), 1));
  lr3.insert_item(&lc3);
  lr3.switch_mode(Detection);
  if (verbose) printf("[%f, %f]\n", lr3._left_bound, lr3._right_bound); 
  if (verbose) printf("%f %f %f %f %f %f\n", lr3.get_mean(), lr3.get_variance(), lr3.f(0.0), lr3.f(2.0), lr3.F(2.0), lr3.F(24.0));


  return true;
}



