/* Copyright (C) 2003 Reliable Software Group 
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* $Id: bayes.cpp,v 1.15 2004/03/12 20:44:21 dhm Exp $ */

#include <iostream>

#include "anomaly.h" 
#include "bayes.h"

BayesianNetworkModelImpl::BayesianNetworkModelImpl(BayesianNetwork *bn, 
						   string queryVariable, 
						   string stateName) {
  // FIX: hack!
  // This can't be fixed (without a cast) unless we make DSL_network
  // visible in bayes.h. But that type is only visible if Smile is
  // installed on the user's system. So, for now we cast to the only
  // type bn can be. -dhm
  _bn = (BayesianNetworkImpl *) bn;
  assert(_bn);
  _queryVariable = strdup((char *) queryVariable.c_str());
  _stateName = strdup((char *) stateName.c_str());

  _net = _bn->getNetworkReference();

  _debug = false;
}


void BayesianNetworkModelImpl::insert_item(Item *item)
  throw (ModelInputException) {
  // do nothing
  return;
}

void BayesianNetworkModelImpl::switch_mode(ModelMode mode) {
  // do nothing
  return;
}

double BayesianNetworkModelImpl::check_item(Item *item) 
  throw (ModelInputException) {

  int status;
  unsigned int nodeIndex;

  if (_debug)
    cout << "BayesianNetworkModelImpl::check_item:" << endl;

  assert(_net);

  // enter evidence
  ListCollection *evidence = dynamic_cast<ListCollection *>(item);
  if (!evidence) {
    throw ModelInputException("BayesianNetworkModelImpl::check_item: argument"
			      " item should be of type ListCollection");
  }

  assert(evidence);

  if (_debug) {
    cout << "  Entering " << evidence->size() << " pieces of evidence:" << endl;
  }

  ListCollection::iterator i;
  for (i = evidence->begin(); i != evidence->end(); i++) {
    BayesianEvidence *be = dynamic_cast<BayesianEvidence *>(*i);
    
    if (!be) {
      throw ModelInputException("BayesianNetworkModelImpl::check_item: "
				"argument item should contain instances of "
				"BayesianEvidence");
    }

    assert(be);

    if (_debug) {
      cout << "  variable (" << be->getVariable() << ") receiving evidence ("
	   << be->getState() << ")" << endl;
    }

    int evidenceVariableHandle = _net->FindNode((char*)be->getVariable());
    
    if (evidenceVariableHandle < 0) {
      sprintf(_debugString, 
	      "BayesianNetworkModelImpl::check_item: "
	      "couldn't find node named '%s' in DSL_Network",
	      be->getVariable());
      throw ModelInputException(_debugString);
    }

    _net->GetNode(evidenceVariableHandle)->Value()->SetEvidence(be->getState());
  }

  // do propagation
  _net->UpdateBeliefs();

  if (_debug) {
    cout << "  UpdateBeliefs() finished" << endl;
  }

  // call IsValueValid on the node in question

  // get value of 'state' for query variable _queryVariable
  int queryVariableHandle = _net->FindNode(_queryVariable);

  if (queryVariableHandle < 0) {
    sprintf(_debugString, 
	    "BayesianNetworkModelImpl::check_item: "
	    "couldn't find variable '%s' in _net",
	    _queryVariable);
    throw ModelInputException(_debugString);
  }

  DSL_sysCoordinates 
    theCoordinates(*_net->GetNode(queryVariableHandle)->Value());

  DSL_idArray *theNames;

  theNames = 
    _net->GetNode(queryVariableHandle)->Definition()->GetOutcomesNames();

  int stateNameIndex = theNames->FindPosition(_stateName);

  if (stateNameIndex < 0) {
    sprintf(_debugString,
	    "BayesianNetworkModelImpl::check_item: couldn't find state '%s'" 
	    " in query variable '%s'",
	    _stateName, _queryVariable);
    throw ModelInputException(_debugString);
  }

  if (_debug) {
    cout << "bayesian check === " << _stateName << " == " << stateNameIndex 
	 << endl;
  }

  theCoordinates[0] = stateNameIndex;
  theCoordinates.GoToCurrentPosition();
  double result = theCoordinates.UncheckedValue();

  // clear evidence
  _net->ClearAllEvidence();

  return result;
}

double BayesianNetworkModelImpl::get_confidence(void) {
  // BayesNets don't change confidence, so return max
  return 1.0;
}

bool BayesianNetworkModel::test(bool verbose) {
  if (verbose) {
    cout << "BayesianNetworkModel::test:" << endl;
  }

  string filename = "simple-ids.dsl";

  BayesianNetwork *bn = BayesianNetwork::instance(filename);

  if (!bn) {
    throw ModelException("BayesianNetworkModel::test: "
			 "couldn't instance BayesianNetwork\n");
  }

  BayesianNetworkModel *bnm = BayesianNetworkModel::instance(bn,
							     "compromised",
							     "successful");

  if (!bnm) {
    throw ModelException("BayesianNetworkModel::test: "
			 "couldn't instance BayesianNetworkModel\n");
  }

  BayesianEvidence *be1 = new BayesianEvidence("ris", 0);
  BayesianEvidence *be2 = new BayesianEvidence("ris", 1);

  BayesianEvidence *be3 = new BayesianEvidence("Node2", 0);
  BayesianEvidence *be4 = new BayesianEvidence("Node2", 1);
  
  ListCollection *evidence = new ListCollection();
  
  evidence->push_back(be1);

  double probSuccessfulCompromise = bnm->check_item(evidence);

  if (verbose) {
    cout << "  probSuccessfulCompromise = " << probSuccessfulCompromise 
	 << endl;
  }

  return true;
}



BayesianNetwork *BayesianNetwork::instance(string filename) {
  return new BayesianNetworkImpl(filename);
}

BayesianNetworkImpl::BayesianNetworkImpl(string filename) {
  _filename = filename;

  int status;
  _net = new DSL_network;

  status = _net->ReadFile((char *) _filename.c_str(), DSL_DSL_FORMAT);
  
  if (status < 0) {
    cout << "BayesianNetworkImpl::BayesianNetworkImpl: "
	 << "couldn't initialize bayes net from file '"
	 << _filename << "'. (operation returned " << status << ")" << endl;
    exit(1);
  }
  else {
      cout << "_net->ReadFile succeeded. Loaded network has " 
	   << _net->GetNumberOfNodes() << " nodes." << endl;
  }

  _net->SetDefaultBNAlgorithm(DSL_ALG_BN_LAURITZEN);
}

// add constructor args later
BayesianNetworkModel *BayesianNetworkModel::instance(BayesianNetwork *bn,
						     string queryVariable,
						     string stateName) {
  return new BayesianNetworkModelImpl(bn,
				      queryVariable,
				      stateName);
}

BayesianEvidence::BayesianEvidence(string variable, int state) {
  _variable = variable;
  _state = state;
}

size_t BayesianEvidence::hash_value(void) const {
  size_t hashVal = (size_t) _state;
  char *variableName = (char *) _variable.c_str();
  unsigned int i;

  for (i = 0; i < strlen(variableName); i++) {
    unsigned int numBitsToShift = 
      (i % sizeof(size_t)) / sizeof(char) * 8;
    
    hashVal ^= variableName[i] << numBitsToShift;
  }
  
  return hashVal;
}

