#ifndef PROFILE_H__
#define PROFILE_H__

/* Profile declaration - used to map input elements to appropriate models
 * Copyright (C) 2003 Reliable Software Group
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: profile.h,v 1.11 2004/10/20 00:33:07 dhm Exp $ */

/** The profile class manages which models receive 
 * features (parameter and return values) from a particular training
 * example. 
 *
 * A profile combines several models and manages the distribution of
 * input events to the appropriate models. A profile is organized as a
 * list of list of models. This allows one to assign multiple models to
 * each feature; and to assign different model lists to multiple
 * features (hence a list of model lists).
 * Subclasses of the ModellingProfile class (this class) implement 
 * policies that dictate how examples (FeatureVectors) are used as training
 * examples by and evaluated using Models. Some elements of the collection 
 * may be ignored completely. In general, one modeling profile will be
 * applied to all monitored applications in the system.
 */
class Profile {
 public:
  /** Add a single model instance to the profile which will receive
   * feature number 'feature' upon invocation of ::insert_item and
   * ::check_item. This actually inserts a list containing the single
   * element 'feature'.
   *
   * @param m: the model instance to be added to the profile 
   *
   * @param feature: the index of the feature in FeatureVector's
   * provided to the profile that will be passed to the model as input
   */
  virtual void addModelFeatureMapping(Model *m, int feature) = 0;

  /** Add a single model instance to the profile which will receive
   * the features specified in featureSet upon invocation of
   * ::insert_item and ::check_item.
   *
   * @param m: the model instance to be added to the profile 
   *
   * @param featureSet: a collection of IntegerItem's that are indexes
   * of features in FeatureVector's provided to the profile that will
   * be passed to the model as a Collection of Items
   */
  virtual void addModelFeatureMapping(Model *m, ListCollection *featureSet) = 0;

  /** Print a string representation of this Profile to the standard
   * output stream. This call traverses the list of model lists and
   * prints each element.
   */
  virtual void toString(void) = 0;

  /** Insert a list of input elements into the profile. Note that the
   * length of the list collection input_vector has to be equal to the number of
   * elements (lists) in the list of lists of this profile. Each
   * individual item of the input element list (input_vector) is then added to
   * corresponding the model list such that the first element of
   * input_vector gets added to the first list of the profile and so on.
   * @param input_vector: list of input elements that are added to the
   * respective model lists of this Profile
   */
  virtual void insert_item(ListCollection *input_vector) = 0;
  
  /** Switch the mode of all models of this Profile (each element in the list of
   * model lists).
   * @param mode: the new mode of the models
   */
  virtual void switch_mode(Model::ModelMode mode) = 0;
  
  /** Check a list of input elements against the models in the profile. Note 
   * that the length of the list collection input_vector must be compatible
   * with the model-feature mappings as defined by calls to 
   * @method addModelFeatureMapping. Features are provided to models 
   * according to these mappings and the resulting individual anomaly scores
   * are combined in an aggregate final score returned by this method.
   * @param input_vector: list of input elements that are checked by the
   * respective model lists of this Profile
   * @return the aggregate anomaly score of the input_vector as
   * determined by the models of this profile
   */
  virtual double check_item(ListCollection *input_vector) = 0;

  /** Check a list of input elements against the models in the profile. Note 
   * that the length of the list collection input_vector must be compatible
   * with the model-feature mappings as defined by calls to 
   * @method addModelFeatureMapping. Features are provided to models 
   * according to these mappings and the resulting individual anomaly scores
   * are combined in an aggregate final score returned by this method.
   * @param input_vector: list of input elements that are checked by the
   * respective model lists of this Profile
   * @return a list of individual model anomaly scores for the models contained
   * in the profile
   */
  virtual ListCollection *
    check_item_individual_model_scores(ListCollection *input_vector) = 0;


  /** Check a list of input elements against the models in the
   * profile. Note that the length of the list collection input_vector
   * must be compatible with the model-feature mappings as defined by
   * calls to @method addModelFeatureMapping. Features are provided to
   * models according to these mappings and the resulting individual
   * anomaly scores are combined in an aggregate final score returned
   * by this method.  Details of the evaluation are returned via
   * @param evaluationResults.
   *
   * @param input_vector: list of input elements that are checked by
   * the respective model lists of this Profile
   *
   * @param evaluationResults: A list of class
   * EvaluationResult containing details of the per-model
   * evaluation.
   *
   * @return the aggregate anomaly score of the input_vector as
   * determined by the models of this profile
   */
  virtual double check_item_deep(ListCollection *input_vector,
				 ListCollection *evaluationResults) = 0;
  
  virtual Model *get_model(unsigned int index) = 0;
  virtual unsigned int get_number_of_models(void) = 0;

  /** Return a pointer to an implementation of the basic Profile class.
   * @return a pointer to an implementation of the basic Profile class
   */
  static Profile *instance(void);

  /** Return a pointer to an implementation of the smart Profile class
   * that supports Bayes net score aggregation.
   * @param path_to_net: the file name of the Bayesnet that should be loaded
   * @param smart_profile_class_name: the class name of the subclass of SmartProfile
   *                                  that handles the model variables in the specified
   *                                  bayes net file
   * @return a pointer to an implementation of the smart Profile class
   */
  static Profile *smart_instance(string path_to_net,
				 string smart_profile_class_name);

  virtual void check();

  /** Turn on/off debugging for this profile.
   *  @param debug: new value for debug flag
   */
  virtual void set_debug(bool debug);

  // check_item_deep should eventually return a ListCollection of these
  class EvaluationResult : public Item {
  private:
    string          _modelName;
    DoubleItem     *_modelScore;
    DoubleItem     *_modelConfidence;
    Item           *_featureNumber;

  public:
    EvaluationResult(string modelName, 
		     DoubleItem *modelScore,
		     DoubleItem *modelConfidence,
		     Item *featureNumber);

    DoubleItem *getModelScore(void) { return _modelScore; }
    DoubleItem *getModelConfidence(void) { return _modelConfidence; }
    string getModelName(void) { return _modelName; }

    size_t hash_value(void) const {
      __gnu_cxx::hash<const char*> H;
      size_t hashVal = 
	H(_modelName.c_str()) ^ 
	_modelScore->hash_value() ^
	_modelConfidence->hash_value() ^
	_featureNumber->hash_value();

      return hashVal;
    }
  };

 protected:
  /** The current debug mode of the model (on/off).
   */
  bool     _debug;
};

#endif
