#ifndef ITEM_H__
#define ITEM_H__

/* Declares the items (data objects) that the anomaly detection
 * library can operate on
 * Copyright (C) 2003 Reliable Software Group 
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: item.h,v 1.8 2003/03/25 05:44:14 dhm Exp $ */




/** \brief Root object of the libAnomaly framework.
 *
 *  The generic, root class for data objects in the libAnomaly
 * framework. All other data objects and their containers must be
 * (directly or indirectly) derived from Item
 */
class Item {
 protected:
  /** Memeory management variable. This reference counter stores the
   * number of external references to this Item. An Item can only be
   * deleted when the reference count has reached 0.
   */
  unsigned int _refc;

  /** Memory management variabel. Stores the total number of allocated
   * Item objects.
   */
  static unsigned int _total_alloc;

 public:
  /** Base class constructor. Initialize the reference count to
   *  zero. That is, no other class holds a reference to it.
   */
  Item(void);

  /** Virtual destructor.
   */
  virtual ~Item(void);

  /** Memory management function. Register interest in that object
   *  (Item) and increase the reference counter (private Item::_refc)
   *  by one. The * corresponding release functions cannot delete the
   *  object until * the reference count has reached zero.
   */
  virtual void lock();

  /** Memory management function. Attempt to delete the object. First,
   *  decrease the reference counter (private Item::_refc) by one. If
   *  it is zero, then the memeory is actually freed (delete is
   *  called). Otherwise, the reference counter is decreased by one.
   */
  virtual void release();

  /** Memory management function. Returns the total number of
   * currently allocated Item objects.
   *  @return  Returns the total number of currently allocated Item
   * objects.
   */
  static unsigned int get_allocated();

  /** Compare this Item object with another Item. 
   * @param item: second item used for comparison
   * @return true, if items are equal, false otherwise 
   */
  virtual bool operator==(const Item & item) const;

  /** Return a hash value for this Item.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const = 0;

  /** Print a string representation of this Item to an output stream.
   * Use the standard output stream as default.
   * @param o: output stream where the Item should write itself to
   */
  virtual void to_string(ostream &o = cout) const;
};

/** \relates Item
 * Printout function for Item.
 * @param os: reference to output stream that item is written to
 * @param i: reference to item that is printed
 * @return reference to output stream after object is written out 
*/
ostream& operator<< ( ostream& os, Item& i);

/** \relates Item
 * Printout function for Item.
 * @param os: reference to output stream that item is written to
 * @param i: pointer to item that is printed
 * @return reference to output stream after object is written out 
*/
ostream& operator<< ( ostream& os, Item* i);




namespace std {
  /** \brief Function Object (Template) for Item used in STL Hashtable
   *
   * This structure is used as a function object to encapsulate the
   * equals operation for the STL hashtable
   */
  template<> struct equal_to<Item*>  
  {
    /** The function operator invokes a comparison between two items 
     * @param left: first operand in equals operation
     * @param right: second operand in equals operation
     * @return true, if the left operand is equal to the right operand, false otherwise
     */
    bool operator()(const Item *left, const Item *right) const
    {
      return *left == *right;
    }
  }; 
}




#if defined(__GNUC__) && (__GNUC__ == 3 || __GNUC__ == 4)
namespace __gnu_cxx {
#else
namespace std {
#endif
  /** \brief Function Object (Template) for Item used in STL Hashtable
   *
   * This structure is used as a function object to encapsulate the
   * hash value operation for the STL hashtable
   */
  template<> struct hash<Item*>
  { 
    /** The function operator invokes the calculation of the hash
     * value of an Item
     * @param item: item whose hash value is determined
     * @return hash value of type size_t
     */
    size_t operator()(const Item *item) const
    {
      return item->hash_value();
    }
  }; 
#if 1
}
#endif




/** \brief Class to store data objects of type string.
 *
 */
class StringItem : public Item {
 public:

  /** Default constructor.
   * Set stored string to the empty string
   */
  StringItem(void);

  /** Constructor.
   * Set stored string to the argument string
   * @param s: string to store in StringItem object
   */
  StringItem(string s);

  /** Virtual destructor.
   */
  virtual ~StringItem(void);

  /** Compare this StringItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is a StringItem and they are equal, false otherwise 
   */
  virtual bool operator==(const Item &item) const;

  /** Return a hash value for this Item.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Print a string representation of this StringItem to an output stream.
   * Use the standard output stream as default.
   * @param o: output stream where the StringItem should write itself to
   */
  virtual void to_string(ostream &o = cout) const;

  /** Get method.
   * @return string value that this object is storing
   */
  string getValue(void);

 private:
  /** String object variable 
   */
  string _s;
};




/** \brief Class to store data objects of type double.
 *
 */
class DoubleItem : public Item {
 public:

  /** Default constructor.
   * Set stored double to zero
   */
  DoubleItem(void);

  /** Constructor.
   * Set stored double to the double argument 
   * @param x: double to store in this DoubleItem
   */
  DoubleItem(double x);

  /** Virtual destructor.
   */
  virtual ~DoubleItem(void);

  /** Compare this DoubleItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is a DoubleItem and they are equal, false otherwise 
   */
  bool operator==(const Item &item) const;

  /** Return a hash value for this Item.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Print a string representation of this DoubleItem to an output stream. 
   * Use the standard output stream as default.
   * @param o: output stream where the DoubleItem should write itself to
   */
  void to_string(ostream &o = cout) const;

  /** Get a floating point representation of the value of this item.
   * @return double precision representation of this item.
   */
  double get_value(void) const;


 private:
  /** Double object variable
   */
  double _x;
};




/** \brief An abstract class to represent data objects than can be
 * mapped to a numerical value
 *
 * Numerical items have to implement the less-than relationship and
 * thus, can be sorted
 */ 
class NumericalItem : public Item {
 public:
  /** Virtual destructor.
   */
  virtual ~NumericalItem(void);

  /** Return the numerical representation of this NumericalItem.
   * @return the numerical representation as a value of type size_t
   */
  virtual size_t value(void) const = 0;

  /** Compare this NumericalItem with an Item according to the
   * less-than relationship.
   * @param item: second item used for comparison
   * @returns true, if item is a NumericalItem and less than this.
   * false if item is a numerical item and greater than or equal to
   * this
   * @throws NumericalException: when item is not of type NumericalItem (or derived)
   */
  virtual bool operator<(const Item &item) const throw (NumericalException) = 0;
};




namespace std {
  /** \brief Function Object (Template) for NumericalItem used in STL Map
   *
   * This structure is used as a function object to encapsulate the
   * less-than operation for the STL hashtable
   */
  template<> struct greater<NumericalItem*>
    {
    /** The function operator invokes a comparison between two items 
     * @param left: first operand in less-than operation
     * @param right: second operand in less-than operation
     * @return true, if the left operand is less than the right operand, false otherwise
     */
      bool operator()(const NumericalItem* left, const NumericalItem* right) const
      {   
	      return left->value() < right->value();
      }
    };
}




/** \brief Class to store data objects of type integer.
 *
 */
class IntegerItem : public NumericalItem {
 public:

  /** Default constructor.
   * Set stored integer to zero
   */
  IntegerItem(void);

  /** Constructor.
   * Set stored integer to the integer argument 
   * @param i: integer to store in this IntegerItem
   */
  IntegerItem(int i);

  /** Virtual destructor.
   */
   virtual ~IntegerItem(void);

  /** Compare this IntegerItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is an IntegerItem and they are equal, false otherwise 
   */
  virtual bool operator==(const Item &item) const;

  /** Return a hash value for this Item.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Compare this IntegerItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is an IntegerItem and this object is
   * less than the second one, false otherwise 
   * @throws NumericalException: when item is not of type NumericalItem (or derived)
   */
  virtual bool operator<(const Item &item) const throw (NumericalException);

  /** Returns the numerical representation of the stored integer as
   * value of type size_t
   * @return numerical representation of the stored integer as
   * value of type size_t
   */
  virtual size_t value(void) const;

  /** Print a string representation of this IntegerItem to an output stream. 
   * Use the standard output stream as default.
   * @param o: output stream where the IntegerItem should write itself to
   */
  virtual void to_string(ostream &o = cout) const;

 private:
  /** Double object variable
   */
  int _i;
};





/** \brief Class to store data objects of type unsigned character.
 *
 */
class CharItem : public NumericalItem {
public:
  
  /** Default constructor.
   * Set stored character to zero
   */
  CharItem(void);

  /** Constructor.
   * Set stored character to the character argument 
   * @param c: Character to store in this CharItem
   */
  CharItem(unsigned char c);

  /** Virtual destructor.
   */
   virtual ~CharItem(void);

  /** Compare this CharItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is a CharItem and they are equal, false otherwise 
   */
  virtual bool operator==(const Item &item) const;

  /** Return a hash value for this Item.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Compare this CharItem with another Item.
   * @param item: second item used for comparison
   * @return true, if second item is an CharItem and this object is
   * less than the second one, false otherwise 
   * @throws NumericalException: when item is not of type NumericalItem (or derived)
   */
  virtual bool operator<(const Item &item) const throw (NumericalException);

  /** Returns the numerical representation of the stored character as
   * value of type size_t
   * @return numerical representation of the stored character as
   * value of type size_t
   */
  virtual size_t value(void) const;

   /** Print a string representation of this CharItem to an output stream. 
   * Use the standard output stream as default.
   * @param o: output stream where the CharItem should write itself to
   */
  virtual void to_string(ostream &o = cout) const;

 private:
  /** Character object variable
   */
  unsigned char _c;
};

#endif
