#ifndef COLLECTION_H__
#define COLLECTION_H__

/* Declares containers to handle data objects
 * Copyright (C) 2003 Reliable Software Group
 *                    - University of California, Santa Barbara
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/* CVS $Id: collection.h,v 1.7 2003/02/28 22:00:58 chris Exp $ */



/** \brief The generic, root container class.
 */ 
class Collection : public Item 
{
 public:
  /** Virtual destructor.
   */
  virtual ~Collection(void);
};




/** \brief A container that holds objects in a certain order.
 */
class OrderedCollection : public Collection 
{
 public:
  /** Virtual destructor.
   */
  virtual ~OrderedCollection(void);
};




/** \brief A container that stores objects as sequence.
 *
 * The sequence is realized as a list by subclassing the corresponding
 * STL class. 
 */
class ListCollection : public OrderedCollection, public list<Item *> {
 public:
  /** Virtual destructor.
   */
  virtual ~ListCollection(void);

  /** Print a string representation of this container to an output
   * stream. This is done by printing every object in the ordered
   * sequence. Use the standard output stream as default.
   * @param o: output stream where the ListCollection should write itself to
   */
  virtual void to_string(ostream &o = cout) const;

  /** Return a hash value for this Item. Returns a hash of the hash
   * values of its list elements.
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Memory management function. Lock the list and its elements. First, invoke lock
   * for each element, the increase the reference counter (Item::_refc) for this list.
   */
  virtual void lock(void);

  /** Memory management function. Attempt to release the list of objects. First,
   * attempt to release all list elements, then handle the list itself.
   */
  virtual void release(void);
};

/** \typedef typedef ListCollection::iterator ListCollectionIterator;
 * Simplifies the declaration of an iterator for the ListCollection.
 */
typedef ListCollection::iterator ListCollectionIterator;




/** \brief A container that stores an object (item) together with its number
 * of occurrences (count).
 *
 * ItemCount can be used to combine multiple occurrences of the same
 * object by keeping a counter value.
 */ 
class ItemCount : public OrderedCollection { 
 public:

  /** Default constructor.
   * Set item to NULL and counter to zero
   */
  ItemCount(void); 

  /** Constructor.
   * Set item but set count to zero
   * @param i: item that the container should store
   */
  ItemCount(Item* i);

  /** Constructor.
   * Set item and its count
   * @param i: item that the container should store
   * @param c: number of occurrences of item i
   */
  ItemCount(Item* i, size_t c);

  /** Virtual destructor.
   */
  virtual ~ItemCount(void);

  /** Memory management function. Lock the ItemCount its stored
   * object. First, invoke lock for the stored object, the increase
   * the reference counter (Item::_refc) for this object.
   */
  virtual void lock(void);

  /** Memory management function. Attempt to release the ItemCount and
   * its stored object. First, attempt to release the stored object,
   * then handle the ItemCount itsel.
   */
  virtual void release(void);

  /** Return a hash value for this Item. Returns a hash of
   * the item that it is pointing to
   * @return opaque hash_value of type size_t 
   */
  virtual size_t hash_value(void) const;

  /** Compare this ItemCount with another ItemCount.
   * @param right: second object used for comparison
   * @return true, if second ItemCount has occurred more often, false otherwise 
   */
  virtual bool operator<(ItemCount& right);
  
  /** Print a string representation of this ItemCount to an output stream. 
   * Use the standard output stream as default.
   * @param o: output stream where the ItemCount should write itself to
   */
   virtual void to_string(ostream &o = cout) const;

   /** Get method.
    * @result a pointer to the stored object (item)
    */
   Item* getValue(void);

   /** Set method.  
    * @param item: the pointer to the new item that this
    * object holds (make sure to free the old item pointer manually)
    */
   void setValue(Item *item);

   /** Get method.
    * @result the number of occurrences as value of type size_t
    */
   size_t getCount(void) const;

 private:
  /** Reference to stored object (item)
   */
  Item* _item;

  /** Number of occurrences of item
   */
  size_t _count;
};

#endif

