#!perl -wW

package CyberArmy::ZebServ;

$|++;

$__PACKAGE__::VERSION = '1.0rc1';

use strict;
use Getopt::Long;
use Time::HiRes ();

use Proc::Daemon;
use Proc::PID::File;

use CyberArmy;
use CyberArmy::User;
use CyberArmy::Database;
use CyberArmy::Groupware;

Getopt::Long::GetOptions (\my %args, qw(host=s port=i name=s
		pass=s nick=s desc=s chan=s oper=s tick=i flag=s debug));

## defaults args
$args{'name'} ||= 'zebserv.cyberarmy.net';
$args{'desc'} ||= "http://$CyberArmy::Config{SERVER_NAME}/ Service";
$args{'nick'} ||= 'CyberArmy';

$args{'tick'} ||= 10;
$args{'flag'} = 'irc' unless defined $args{'flag'};
$args{'nickserv'} = 'NickServ';

foreach (qw(host port pass oper)) {
	die "required argument $_ missing\n" unless $args{$_}
}

Proc::Daemon::Init unless $args{'debug'};

die "Already running!\n" if Proc::PID::File->running( { name => 'zebserv' } );

$0 = __PACKAGE__ . '/' . $__PACKAGE__::VERSION;

if ($args{'flag'}) { ## group in which we will put online users
	if (CyberArmy::Groupware->get($args{'flag'})) {
		## wipe out any stale entries from the defined flag group
		CyberArmy::Groupware->wipe($args{'flag'}) if $args{'flag'};
	} else {
		## simply make sure it exists before proceeding
		die "Group $args{flag} doesn't exist!\n";
	}
}

$SIG{'INT'} = \&quit;

my $irc = CyberArmy::ZebServ::Connection->instance(\%args);

CyberArmy::Database->instance->{'HandleError'} =
	sub { ## install a dbi error handler
		$irc->log('DB ERROR: '. shift @_);
	};

#CyberArmy::Database->instance->trace(1);

## These were singletons: from now on objects returned
## by instance() will just reference these initial ones

$irc->sock("PASS $irc->{pass}");
$irc->sock("PROTOCTL NICKv2 SUSER");
$irc->sock("SERVER $irc->{name} 1 :CyberArmy IRC Services");
$irc->sock(":$irc->{name} 1 :U0-*-1 $irc->{desc}");
$irc->sock("NICK $irc->{nick} 0 1 service $irc->{name}".
			" $irc->{name} 0 +iowgWqS * * :$irc->{desc}");
$irc->send("MODE $irc->{nick} +oqS");

## first, join main channel, then, all other registered channels
$irc->send("JOIN $irc->{chan}") if $irc->{'chan'};
CyberArmy::ZebServ::Channel->load; ## load them up into memory
if (my $registered = CyberArmy::ZebServ::Channel->getChanneList()) {
	my (@channel) = (values %{$registered});

	foreach (@channel) {
		my $mode = defined $_->{'acl_join'} ? 'i' : '';
		$irc->send('JOIN '.$_->{'name'},
			"MODE $_->{name} +${mode}qo $irc->{nick} $irc->{nick}");
	}

	if (my $join = join (',', map {$_->{'name'}} @channel)) {
		$irc->log('Joined: '. $join);
	}
}

my (%ignore); ## internal ignore list
$irc->{'backlog'} = 1; ## start in backlog mode
my $tick = time(); ## timestamp of last tick

while () {
	if ($irc->get_line()) { &handler }
	elsif (!$irc->{'backlog'} &&
			($tick + $irc->{'tick'}) <= (my $now = time)) {
		$irc->debug("tick... ($tick)");
		my $last = $tick; $tick = $now;

		CyberArmy::ZebServ::Event::User->handler($last);
		CyberArmy::ZebServ::Event::Forum->handler($last);
		CyberArmy::ZebServ::Event::Message->handler($last);
	} else { ## no events are pending (temporary ugly hack)
		Time::HiRes::sleep(0.2); ## limit CPU usage by sleeping
	}
}

sub handler {
	my($event,@event) = $irc->get_event();

	if ($irc->{'special'}) {
		## first handle events with exotic formatting
		## as in those lines not beginning with a ":"

		if ($event eq 'PING') { ## that's server ping
			$irc->sock(":$irc->{name} PONG $irc->{name}");

		} elsif ($event eq 'NICK') { ## user joining
			## NICK <nick> <hopcount> <timestamp> <username> <hostname> *
			## <server> <servicestamp> <modes> <fakehost> :<realname
			my ($nick,$hostmask,$stamp,$modes) = (@event[0,4,6,7]);
			if (index($modes,'r') >= 0) { ## if he is a registered user
				if (my $new = CyberArmy::ZebServ::User->new($nick)) {
					$new->setAsOnline(); ## just pre-login for now
					## we aren't sure if the +r is geniuine or not, yet
					$irc->nickStatusQuery($nick) unless $irc->{'backlog'};
					## nickStatusQuery() will verify that with NickServ
				}
			} elsif (index($modes,'S') >= 0) { ## service
				$ignore{$nick} = 1; ## ignore him
				## ignore for most ACTIONs to avoid service war
				## ZebServ VS ChanServ being the most notorious
				if ($nick eq $irc->{'nickserv'}) {

					unless ($irc->{'backlog'}) {
						## NickServ just came online back after
						## an outage, sort out any queued login
						my $users = CyberArmy::ZebServ::User->getLoginQueue();
						my @nicknames = map { $_->{'nickname'} } (values %$users);
						$irc->nickStatusQuery(@nicknames); ## check NickServ status
					}

					## track NickServ's availability
					## as we cannot operate without it
					$irc->{'services'} = 1 ;
					$irc->log("Services are up!");
				}

			}
		} elsif ($event eq 'NETINFO') { ## backlog is over
			$irc->{'backlog'} = 0;
			## after joining, we delays login to the end of the backlog
			## which is now. there is 2 reasons for that, first, we need
			## to make sure services are actually working, and we can't
			## until the end of the backlog, also if services are up, we
			## don't want users to login and join as NICK happens before
			## JOIN, which means autoJoin() won't know what channels is
			## the user already in, & will send bogus (harmless) SVSJOIN
			my $users = CyberArmy::ZebServ::User->getLoginQueue();

			if ($irc->{'services'}) { ## services are up
				my @nicknames = map { $_->{'nickname'} } (values %$users);
				$irc->nickStatusQuery(@nicknames); ## check NickServ status
			} ## if services aren't up, we will do the above with it does come

			$irc->log("backlog is over...");
		}

	} else { ## now handle standard format events
		my $from = $event;
		$event = uc shift @event;
		my $to = shift @event;

		## see if we are dealing with registered elements or not
		my $user = CyberArmy::ZebServ::User->isLoggedIn($from)
				|| CyberArmy::ZebServ::User->inLoginQueue($from);

		if ($event eq 'SVSMODE' || $event eq 'SVS2MODE') {
			my ($modes) = shift @event;
			if ($modes =~ m/\+\w*r/ and not defined($user)) {
				## a new user just identified (registered)
				if (my $new = CyberArmy::ZebServ::User->new($to)) {
					unless ($new->isLoggedIn()) {
						## no actual login here, we wait for
						## whois to see what channs he is in
						## get his hostmask etc...
						$irc->whois($to);
					}
				}

			} elsif ($modes =~ m/\-\w*r/) {
				if (my $user = CyberArmy::ZebServ::User->isLoggedIn($to)) {
					## user is loosing his registered status
					if ($to eq $user->irc_nickname) {
						## it is droping his registered nick
						$user->drop(); ## drop him out too
						$irc->notice('Your nickname has been dropped',$to);
					}
				}
			}

		} elsif ($event eq '307') { ## WHOIS: is a registered nick
			## this is a self triggered event, when someone issues a login
			## command we need to know if he is identified before allowing
			## him in, so we fire a WHOIS, and wait confirmation from this
			my $nick = shift @event;
			if (my $new = CyberArmy::ZebServ::User->new($nick)) {
				$new->setAsOnline(); ## pre-login step
				## also see callbacks 378, 319 and 318
			}

		} elsif (($event eq 'NOTICE')
					and ($to eq $irc->{'nick'})
					and ($from eq $irc->{'nickserv'})) {
			my ($cmd,$nick,$status) = $irc->get_command();
			if ($cmd eq 'STATUS') {
				## this is a reply to our /msg NickServ STATUS <nicks>
				## see /msg NickServ help STATUS for more informations
				if (my $new = CyberArmy::ZebServ::User->inLoginQueue($nick)) {
					if ($status > 1) {
						$new->logIn(); ## THIS IS THE LOGIN POINT ##
						$new->autoJoin(); ## auto join him to his channels
					} else { ## this means the nickname is not registered
						$new->setAsOffline();
						$new->drop(); ## reset nick association
						$irc->notice(
							'You must be identified with NickServ',$nick);
					}
				}
			}

		} elsif ($event eq '319') { ## WHOIS: channel list
			## this is linked to event 307
			## get the list of channels user is already in
			my $nick = shift @event;
			if (my $new = CyberArmy::ZebServ::User->inLoginQueue($nick)) {
				foreach my $name ($irc->get_command()) {
					my ($symbols,$chan) = split(/#/,$name,2);
					my $channel = "#$chan";
					if (CyberArmy::ZebServ::Channel->isRegistred($channel)) {
						$new->joined($channel); ## take notice
						foreach (split//,$symbols) {
							if (my ($cmode) = {qw(+ v % h @ o ^ a)}->{$_}) {
								$new->chanAccess($channel,'+',$cmode);
							}
						}
					}
				}
			}

		} elsif ($event eq '318') { ## End of /WHOIS list.

			if ($irc->{'services'})  {
				## verify nickserv actually recognizes our user
				$irc->nickStatusQuery(shift @event); ## final callback
				## see the NOTICE event for more informations
			} else {
				## make sure services are up before allowing logins:
				## services go down, both tr_elite & tr_lame are +r,
				## tr_elite is logged in to ZS, but tr_lame isn't...
				## if tr_elite quits, tr_lame can chnick to tr_elite
				## while still retaining his tr_lame +r, and ZebServ
				## will gladly accept a password-less login from him
				## as tr_elite. not good... says the +wise Enstyne
				$irc->notice('services are currently not available',$from);
				$irc->notice('you will be logged in as soon as they are back',$from);
			}

		} elsif ($event eq '352') { ## WHO list
			## this is sent when a channel has just been registered
			my ($name,$nick,$modes) = @event[1,5,6];
			if (my $channel = CyberArmy::ZebServ::Channel->isRegistred($name)) {
				if (my $user = CyberArmy::ZebServ::User->isLoggedIn($nick)) {
					$user->joined($name);
					foreach (split//, $modes) {
						if (my ($cmode) = {qw(+ v % h @ o ^ a)}->{$_}) {
							$user->chanAccess($name,'+',$cmode);
						}
					}
					$user->autoAccess(); ## give him any extra modes he has
				}
			}

		} elsif ($event eq 'QUIT') { ## leaving...
			if (defined ($user)) {
				$user->logOut(); ## take notice
			} elsif (exists $ignore{$from}) {
				delete $ignore{$from};
				if ($from eq $irc->{'nickserv'}) {
					## services just went down
					$irc->{'services'} = 0;
					my ($quit) = join(' ',$irc->get_command());
					$irc->log("Services are down ($quit)")
				}
			}

		} elsif ($event eq 'KILL' || $event eq 'SVSKILL') { ## roasted!
			if (my $user = CyberArmy::ZebServ::User->isLoggedIn($to)) {
				$user->logOut(); ## take notice
			} elsif ($to eq $irc->{'nick'}) {
				my ($msg) = join(' ',$irc->get_command());
				$irc->log("Killed! ($msg)"); &quit;
			}

		} elsif (($event eq 'NICK') and defined($user)) { ## nick change
			$user->nickChange($to)

		} elsif (($event eq 'AWAY') and defined($user)) { ## away status
			if ($irc->get_command()) {
				$user->notificationsOff();
				$irc->notice('Notifications are off because you are away',$from);
			} else {
				$user->notificationsOn();
				$irc->notice('Notifications are now on',$from);
			}
		} elsif ($event eq 'JOIN' and !$ignore{$from}) {
			## it's okay if we aren't fully loggedin yet
			## when we will be we will get kicked out if
			## we weren't supposed/allowed to be here...

			foreach my $c (split(/,/, $to)) { ## foreach joined channel
				my $channel = CyberArmy::ZebServ::Channel->isRegistred($c)
					or next; ## skip channels that aren't registered

				if (defined($user)) { ## track user joins
					if ($c eq '0') { ## watch out for /join 0
						$user->parted(); next; ## parting all channels
					} else { $user->joined($c) }
				}

				if ( defined($channel->{'acl_join'}) ) { ## restricted channel
					my $acl = join(';', map { ## build the complete access list
						defined $channel->{'acl_'.$_} &&  $channel->{'acl_'.$_}
					} qw(join voice hop op admin));

					if ($channel->{'strict_access'} eq 'y' &&
							!(defined($user) && $user->CheckGroupList($acl))) {
						$irc->send("KICK $c $from :Access denied");
						$user->parted($c) if defined($user); ## force parting
					}
				}

				$user->autoAccess($c)
					## but skip if we are in backlog mode to prevent
					## giving a user a mode he already has (JOIN precedes MODE)
					unless ($irc->{'backlog'} || !defined($user))
			}
		} elsif ((($event eq 'PART') or ($event eq 'KICK')) and defined($user)){
			$user->parted($_) foreach (split(/,/, $to));

		} elsif ($event eq 'PRIVMSG' && lc($to) eq lc($irc->{'nick'})) {
			## check PRIVMSG sent to ZebServ only for commands ##
			my @command = $irc->get_command();
			my $command = uc(shift(@command));

			if ($command eq "\x01PING") {
				my $pong = shift(@command);
				$irc->notice("\x01PONG $pong",$from);

			} elsif ($command eq "\x01VERSION\x01") {
				chomp (my $uname = `/usr/bin/uname -ps`);
				my $version = join(' ',(
					__PACKAGE__,$__PACKAGE__::VERSION,
					$uname,$CyberArmy::Config{'DINAH_ADMIN'}
				));
				$irc->notice("\x01VERSION $version\x01", $from);

			} elsif ($command eq 'LOGIN') { ## login/mapping gateway
				my($nick) = shift(@command);
				my($pass) = shift(@command);

				if (defined($user) and $user->isLoggedIn()) {
					$irc->notice('Logout first, please', $from)
				} elsif (defined($user) and $user->inLoginQueue()) {
					$irc->notice('yes... wait...', $from);
				} elsif ($nick && $pass) {
					if (my $new = CyberArmy::User->new(nickname => $nick)) {
						if (my $ban = $new->IsBanned()) {
							$irc->notice(
								'Sorry, you are banned till '.
								gmtime($new->bantime),$new->{'nickname'});
						} elsif (not $irc->{'services'}) {
							$irc->notice(
								'sorry, services down, please try again later...'
							,$from);
						} elsif ($new->Passhash($pass)) { ## check password
							## reference his nick to allow future autologin
							$new->Update(irc_nickname => $from);
							## Check if this nick is registered to NickServ
							$irc->whois($from); ## WHOIS the user, and wait
							## for event 307, handled earlier to log him in
							$irc->notice('Logging you in, please wait...', $from);
						} else { $irc->notice('Invalid password', $from) }
					} else { $irc->notice('Invalid username', $from) }
				} else {
					if (my $new = CyberArmy::ZebServ::User->new($nick||$from)) {
						## Check if this nick is registered to NickServ
						$irc->whois($from); ## WHOIS the user, and wait
						## for event 307, handled earlier to log him in
						$irc->notice('Logging you in, please wait...', $from);
					} else {
						$irc->notice('Usage: /msg '.$irc->{'nick'}
							.' LOGIN <site_nick> <site_pass>', $from);
					}
				}

			} elsif (!defined($user) || !$user->isLoggedIn()) {
				$irc->notice('login first, please...',$from); ## STOP ##
				## below this point $user is presumed to be valid and logged in

			} elsif ($command eq 'WHOAMI') { ## am i logged in?
				$irc->notice('You are currently logged in as '.
					$user->{'sitename'},$from);

			} elsif ($command eq 'INVITE' and defined($user)) { ## invite :)
				if (my $name = lc shift (@command)) {
					my $chan = CyberArmy::ZebServ::Channel->isRegistred($name);
					if (defined $chan){
						my $acl = join(';', map { ## build the complete access list
							defined $chan->{'acl_'.$_} &&  $chan->{'acl_'.$_}
						} qw(join voice hop op admin));
						if ($user->CheckGroupList($acl)) {
							$irc->send("INVITE $from $name");
							$irc->sock(":$irc->{name} SVSJOIN $from $name");
						} else { $irc->notice("Access denied!",$from) }

					} else { $irc->notice("$name isn't registered",$from) }
				} else { $user->autoJoin() }

			} elsif ($command eq 'LOGOUT') { ## logout
				$user->logOut();

			} elsif ($command eq 'DROP') { ## drop nick association
				$user->drop();

			} elsif ($command eq 'NOTIFY') { ## notifications?
				my $switch = lc shift (@command);

				if ($switch eq 'on') {
					$user->notificationsOn();
					$irc->notice('Notifications turned on',$from);
				} elsif ($switch eq 'off') {
					$user->notificationsOff();
					$irc->notice('Notifications turned off',$from);
				} else {
					$irc->notice(
						"Usage: /msg $irc->{nick} notify [ON|OFF]",$from)
				}

			} elsif ($command eq 'ADMIN') { ## Administration

				if ($user->IsInGroup($args{'oper'})) {
					my $adm_cmd = lc shift(@command);
					if ($adm_cmd eq 'quit') {
						my $quit = join(' ',@command) || 'bbl';
						$irc->error($quit);
						$irc->send("QUIT :Quit: $quit");
						&quit; ## suicide...

					#} elsif ($adm_cmd eq 'rawr') {
					#	$irc->sock(join(' ',@command));

					} elsif ($adm_cmd eq 'login') {
						my ($irc_nick) = shift(@command);
						my ($nick) = lc shift(@command);

						if (!$irc_nick || !$nick) {
							$irc->notice(
								"Usage: /msg $irc->{nick} login <ircn> <siten>",
							$from);
						} elsif (
							my $xst = CyberArmy::ZebServ::User->new($irc_nick)){
							$irc->notice( $irc_nick.' already mapped ('.
									$xst->nickname .'), drop it, first', $from);

						} elsif (
							my $new = CyberArmy::User->new(nickname => $nick)) {
							$new->Update(irc_nickname => $irc_nick);
							$irc->whois($irc_nick);
							$irc->notice("mapped $nick<->$irc_nick",$from);

						} else { $irc->notice("Unknown nickname",$from) }

					} elsif ($adm_cmd eq 'drop') {
						my ($irc_nick) = shift(@command);

						if (not $irc_nick) {
							$irc->notice(
								"Usage: /msg $irc->{nick} login <irc_nick>",
							$from);
						} elsif (
							my $xst = CyberArmy::ZebServ::User->new($irc_nick)){
							$xst->logOut(); $xst->drop();
						} else { $irc->notice("Unknown nickname",$from) }

					} elsif ($adm_cmd eq 'register') {
						my $name = lc shift(@command);
						if (CyberArmy::ZebServ::Channel->isRegistred($name)) {
							$irc->notice("$name already registered...",$from)

						} else {
							my $admin = shift(@command) || '~'.$user->nickname;
							my $channel = CyberArmy::ZebServ::Channel
								->new({ name => $name, acl_admin => $admin });
							if ($channel) {
								$irc->notice('registered '.$name,$from);
								$irc->send("JOIN $name", "WHO $name",
									"MODE $name +qo $irc->{nick} $irc->{nick}");

							} else { $irc->notice("Can't register $name",$from) }
						}

					} else { $irc->notice('Invalid admin command...',$from) }
				} else { $irc->notice('Access denied!',$from) }

			} elsif ($command eq 'CHANNEL') { ## Administration
				my $name = shift (@command);
				my $command = shift(@command) || '';
				my $oper = $user->IsInGroup($args{'oper'});
				my $channel = CyberArmy::ZebServ::Channel->isRegistred($name);

				if (defined ($channel) ){
					if ($oper or $user->CheckGroupList($channel->{'acl_admin'})){
						if ($command eq 'drop') {
							$irc->send("MODE $name -i")
								if defined($channel->{'acl_join'});
							$channel->drop();
							$irc->notice('dropped '.$name,$from);
							$irc->send("PART $name :channel dropped");

						} elsif ($command eq 'access') { ## manage access lists
							my $type = shift @command; ## join or a/o/h/v

							if ($type && exists($channel->{$type}) ) {
								my $new = shift @command;
								my $old = $channel->{$type};
								if (index($type,'auto_') == 0) {
									if ($oper) {
										if ($channel->update({$type => $new})) {
											$irc->notice($type.'='.$new,$from)
										} else { $irc->notice("Update failed") }
									} else {
										$irc->notice("Access denied!",$from)
									}
								} else {
									if ($channel->update({$type => $new})) {
										$irc->notice($type.'='.$new,$from);
									} else { $irc->notice("Update failed") }
								}

								if ($type eq 'acl_join') { ## join restriction
									if ($new and !$old) { ## added
										$irc->send("MODE $name +i")
									} elsif ($old and !$new) { ## removed
										$irc->send("MODE $name -i")
									}
								}
							} elsif (not $type) { ## listing current accesses
								$irc->notice("Access List of $name:",$from);
								foreach (sort keys %{$channel}) {
									next if $_ eq 'name';
									$irc->notice("$_ = $channel->{$_}",$from)
										if $channel->{$_};
								}
							}
						} elsif ($command eq 'strict') { ## shall be strict ?
							my $type = lc shift @command; ## access/modes
							my $yn = lc shift @command; ## y/n

							if ($type and exists $channel->{'strict_'.$type}) {
								my $strict = 'strict_'.$type;
								($yn eq 'y') || ($yn eq 'n') || ($yn = 'n');

								if ($channel->update({ $strict => $yn })) {
									$irc->notice($strict.'='.$yn,$from)
								} else { $irc->notice("Update failed") }
							} else {
								$irc->notice("strict setting of $name",$from);
								foreach (qw(access mode)) {
									my $strict = $channel->{'strict_'.$_};
									$irc->notice("strict $_ = $strict",$from);
								}
							}

						} else { $irc->notice('Unknown command...',$from) }
					} else { $irc->notice('Access denied!',$from) }
				} else { $irc->notice("$name not registred",$from) }

			} else { $irc->notice('Unknown command...',$from) }

		} elsif ($event eq 'PRIVMSG' && defined($user) && $user->isLoggedIn()) {
			if (my $channel = CyberArmy::ZebServ::Channel->isRegistred($to)) {
				my @cmd = $irc->get_command();
				my $cmd = lc(shift(@cmd));

				if (index($cmd,'!') == 0) {
					if (my $nick = shift(@cmd)) {
						my $cuser = CyberArmy::ZebServ::User->isLoggedIn($nick);

						my $reason = join(' ',@cmd) || $from;
						foreach (qw(hop op admin)) {
							next unless defined ($channel->{'acl_'.$_})
								&& $user->CheckGroupList($channel->{'acl_'.$_});

							if (defined($cuser) &&
									$cuser->hasChanAccess($to,'a')) {
								$irc->notice('sorry, protected user...',$from);
							} else {
								if ($cmd eq '!kick' || $cmd eq '!k') {
									$irc->send("KICK $to $nick :$reason");
								} elsif ($cmd eq '!ban' || $cmd eq '!b') {
								} elsif ($cmd eq '!kickban' || $cmd eq '!kb') {
								}
							}
							last;
						}
					} else {
						my $type = ($cmd =~ s/^!de/!/) ? '-' : '+';
						foreach (qw(voice hop op admin)) {
							if ('!'.$_ eq $cmd) {
								my $mode = substr($cmd,1,1);
								my $has_mode = defined
									$user->{'channels'}->{$to}->{$mode};

								if ($type eq '+' && $has_mode) {
									$irc->notice("you are ${_}ed already",$from);
								} elsif ($type eq '-' && !$has_mode) {
									$irc->notice("you aren't ${_}ed",$from);
								} else {
									my $acl = $channel->{'acl_'.$_};
									if ($acl && $user->CheckGroupList($acl)) {
										$irc->send("MODE $to $type$mode $from");
										$user->chanAccess($to,$type,$mode);
									} else { $irc->notice('Access denied',$from) }
								}
								last;
							}
						}
					}
				}
			}

		} elsif (($event eq 'MODE') and (substr($to,0,1) eq '#')) {
			if (my $channel = CyberArmy::ZebServ::Channel->isRegistred($to)) {
				my @mode = split(//, (shift(@event)||''));
				my %acl = (qw(v voice h hop o op a admin));

				my $type = shift(@mode);
				while ( my $mode = shift(@mode) ) {
					if ($mode eq '+' || $mode eq '-') { $type = $mode; next; }
					next unless $mode =~ m/^[vhoaqbefklL]$/; ## mode takes args
					my $line = shift (@event); ## get the argument off the line
					next unless ($acl{$mode}); ## ...mode if a user access

					my $modu = $irc->{'backlog'} ?
						CyberArmy::ZebServ::User->inLoginQueue($line)
							: CyberArmy::ZebServ::User->isLoggedIn($line);
					## again, note that inLoginQueue() is enough for
					## backlog, logIn() will fixup access issues later

					if ($type eq '+' && $channel->{'strict_mode'} eq 'y') {
						my $acl = $channel->{'acl_'.$acl{$mode}};

						unless( exists($ignore{$from}) ## peace&&love
								&& defined($acl) && defined($modu)
								&& $modu->CheckGroupList($acl) ) {
							## we the revert mode if necessary
							$irc->send("MODE $to -$mode $line");
							next; ## skip chanAccess() below
						}
					}

					## getting here means the mode wasn't reverted
					## take note of the mode if we have a user
					$modu->chanAccess($to,$type,$mode) if defined($modu);
				}
			}
		}
	}
}

sub quit {
	CyberArmy::Groupware->wipe($args{'flag'}) if $args{'flag'};
	#foreach my $user (values %{CyberArmy::ZebServ::User->getOnlineUserList()}) {
	#	$user->logOut(); ## sequentially log out every online user
	#}
	exit;
}

END { &quit }

package CyberArmy::ZebServ::Connection;

use strict;
use IO::Select ();
use IO::Socket::SSL qw(debug1);
use Apache::Singleton;
use base qw( Apache::Singleton );

sub _new_instance {
	my $class = shift;
	my $self = shift;

	bless $self, $class;

	$self->debug('Connecting to '.
		$self->{'host'} .':'. $self->{'port'} . '... ');

	my $sock = IO::Socket::SSL->new(
		PeerAddr => $self->{'host'},
		PeerPort => $self->{'port'},
	);

	if (defined $sock) {
		$self->{'select'} =
			IO::Select->new($sock);
		return $self;
	} else { die IO::Socket::SSL::errstr() }
}

sub get_line {
	my $self = shift;
	if (my ($sock) = $self->{'select'}->can_read(0)) {
		if (my $line = ($self->{'line'} = <$sock>)) {
			$line =~ s/(^\:|\x0a|\x0d|\x00)//g;
			$self->debug(">> $self->{line}");

			($self->{'pre'}, $self->{'tag'}) = split(/ :/, $line, 2);
			my (@event) = split(/\s+/, $self->{'pre'});
			my (@command) = split(/\s+/, $self->{'tag'} || '');

			$self->{'event'} = \@event;
			$self->{'command'} = \@command;

			$self->{'special'} = (substr($self->{'line'},0,1) eq ':') ? 0 : 1;

			return 1;
		} else { close($sock); exit; }
	} else { return 0; }
}

sub get_event {
	return @{$_[0]->{'event'}}
}

sub get_command {
	return @{$_[0]->{'command'}}
}

sub sock {
	my($self, $line) = @_;
	if (my ($sock) = $self->{'select'}->can_write(5)) {
		$self->debug("<< $line");
		print $sock "$line\n";
	} else { die 'Write error: ',IO::Socket::SSL::errstr() }
}

sub send {
	my $self = shift;
	foreach (@_) {
		$self->sock(':'.$self->{'nick'}.' '.$_);
	}
}

sub notice {
	my($self, $notice, @rcpts) = @_;
	foreach (@rcpts) {
		$self->send("NOTICE $_ :$notice");
	}
}

sub privmsg {
	my($self, $message, @rcpts) = @_;
	foreach (@rcpts) {
		$self->send("PRIVMSG $_ :$message");
	}
}

sub whois {
	my($self) = shift;
	foreach (@_) {
		$self->send("WHOIS $self->{host} :$_");
	}
}

sub nickStatusQuery {
	my($self) = shift;

	while (scalar @_) {
		my @nicks = splice(@_,0,16); ## accepts up to 16 nicks in a row
		$self->privmsg("STATUS @nicks",$self->{'nickserv'});
	}
}

sub log {
	my($self, $message) = @_;
	$self->privmsg($message,$self->{'chan'}) if $self->{'chan'};
}

sub error {
	my($self, $message) = @_;
	$message ||= 'bbl';
	$self->sock(":$self->{name} ERROR :".$message);
}

sub debug {
	if (shift->{'debug'}) {
		## for print formatting purposes
		chomp(@_);  ## kill trailing \ns
		print STDERR @_,"\n";
	}
}

package CyberArmy::ZebServ::User;

use CyberArmy::User;
use base qw(CyberArmy::User);

my %isOnline = ();
my %isLoggedIn = ();
my %inLoginQueue = ();

sub getOnlineUserList {
	return \%isOnline;
}

sub getLoginQueue {
	return \%inLoginQueue;
}

sub new {
	my $class = shift;
	my $nickname = shift;

	my $user = CyberArmy::User::new($class,
		key => 'irc_nickname', irc_nickname => $nickname,
		select => 'nickname,irc_nickname,showname,'.
					'passwd,retired,bantime,brigade_chain'
	) or return undef;

	if (exists $isOnline{$user->nickname}) {
		## user is already online, so we will
		## return the already existing object
		return $isOnline{$user->nickname}
	} else {
		my $getAttr = $user->getAttributes('title');
		$user->{'sitename'} = join(' ',@{$getAttr->{'title'}},$user->showname);
		$user->{'sitename'} .= ' ('. $user->nickname .')';
		$user->{'nickname'} = $nickname;
		$user->{'channels'} = {};

		return $user;
	}
}

sub drop {
	my $self = shift;

	## kills the nickname mappings
	$self->Update(irc_nickname => undef);
	#$self->logOut(); ## & force a logout
}

sub setAsOnline {
	## this an intermediary step before the logIn(), this is to say
	## that we know user is online, but we don't want to log him in
	## yet, for security reasons, most probably
	my $self = shift;

	$isOnline{$self->nickname} = $self; ## references his site's nickname
	$inLoginQueue{$self->{'nickname'}} = $self;
}

sub setAsOffline {
	my $self = shift;

	delete $isOnline{$self->nickname};
	delete $inLoginQueue{$self->{'nickname'}}
}

sub isOnline {
	## this will tell us if a given site nickname is currently online
	## For example: my site nickname is wa1800z, & my IRC nickname is waz
	## isOnline("wa1800z") will tell me that "wa1800z" is Online as "waz"
	my $self = shift;

	if (my $nick = shift) { ## called constructor
		## note that we force low-casing
		return $isOnline{lc $nick} ## returns the object
	} elsif (UNIVERSAL::isa($self,__PACKAGE__)) { ## called as method
		return exists $isOnline{$self->nickname}
	} else { return undef }
}

sub inLoginQueue {
	my $self = shift;

	if (my $nick = shift) { ## called as a constructor
		return $inLoginQueue{$nick} ## returns the object
	} elsif (UNIVERSAL::isa($self,__PACKAGE__)) { ## called as method
		return exists $inLoginQueue{$self->{'nickname'}}
	} else { return undef }
}

sub logIn {
	my $self = shift;

	## skip this, if we know he is logged in already
	return 1 if exists $isLoggedIn{$self->{'nickname'}};

	my $irc = CyberArmy::ZebServ::Connection->instance();
	if ($self->IsBanned()) { return undef }
	else {
		$irc->log("$self->{nickname} successfully logged as $self->{sitename}");
		$irc->notice("Welcome $self->{sitename}", $self->{'nickname'});
		$irc->sock("SWHOIS $self->{nickname} :is $self->{sitename}");

		$self->setAsOnline() unless $self->isOnline();
		$isLoggedIn{$self->{'nickname'}} = $self; ## tracks his current IRC nick
		## for more infomations, check the isOnline() and isLoggedIn() methods
		delete $inLoginQueue{$self->{'nickname'}}; ## reset the pre-login status

		## mark him as online on the site
		$self->Update(addtogroup => $irc->{'flag'}) if $irc->{'flag'};

		if (my $mnum = $self->CyberArmy::User::Messages::CheckForUnreadMsg()) {
			$irc->notice( ## warn the user if he has any unread messages
				"You have \x02$mnum\x02 unread messages, see ".
				'http://'.$CyberArmy::Config{'SERVER_NAME'}.'/my/messages/',
			$self->{'nickname'});
		}

		$self->notificationsOn(); ## turn on dynamic notifications by default

		return 1;
	}
}

sub isLoggedIn {
	## this will tell us if the given IRC nickname is currently logged
	## For example: my registered irc nickname on the site is wa1800z,
	## to which i have identified, but if /nick waz, isLoggedIn("waz")
	## will tell me that "waz" is actually wa1800z on the site despite
	## the fact that wa1800z's irc_nickname is not waz.
	my $self = shift;

	if (my $nick = shift) { ## called as a constructor
		return $isLoggedIn{$nick} ## returns the object
	} elsif (UNIVERSAL::isa($self,__PACKAGE__)) { ## called as method
		return exists $isLoggedIn{$self->{'nickname'}}
	} else { return undef }
}

sub logOut {
	my $self = shift;

	$self->autoPart(); ## leaves restricted channels and drop modes
	$self->notificationsOff(); ## remove all references to the user

	my $irc = CyberArmy::ZebServ::Connection->instance();
	$irc->sock("SWHOIS $self->{nickname} :"); ## clear the SWHOIS
	$irc->log("$self->{sitename} has logged out ($self->{nickname})");

	$self->Update(delfromgroup => $irc->{'flag'}) if $irc->{'flag'};

	delete $isLoggedIn{$self->{'nickname'}};
	$self->setAsOffline();
}

sub joined {
	my $self = shift;

	foreach (@_) {
		$self->{'channels'}->{lc $_} = {};
	}
}

sub autoJoin { ## check out what channels user is supposed to join
	my $self = shift;

	my @channel = map { CyberArmy::ZebServ::Channel->isRegistred($_) }
		( $#_ > -1 ? (@_) : (keys %{$self->{'channels'}}) );

	my $channels = CyberArmy::ZebServ::Channel->getChanneList();
	my $irc = CyberArmy::ZebServ::Connection->instance();

	my (@svsjoin);
	foreach my $channel (values %$channels) {
		my $is_in_channel = exists $self->{'channels'}->{$channel->{'name'}};

		if (defined($channel->{'acl_join'})) {
			my @acl; foreach (qw(join voice hop op admin)) {
				push @acl,$channel->{'acl_'.$_}
					if defined($channel->{'acl_'.$_})
			}
			unless ($self->CheckGroupList(join(';',@acl))) {
				if ($is_in_channel) {
					## when the user's info (Event::User) change
					## if he is in, but no longer has access
					$irc->send( ## we kick him out
						"KICK $self->{nickname} $channel->{name} :Access denied"
					); $self->parted($channel->{'name'});
				}
				next; ## skip auto-joining
			}
		}

		next if $is_in_channel; ## skip auto-joining

		foreach (qw(auto_join auto_invite)) {
			next unless $channel->{$_};
			if ($self->CheckGroupList($channel->{$_}) > 0) {
				if ($_ eq 'auto_join') {
					$irc->send("INVITE $self->{nickname} $channel->{name}")
						if defined($channel->{'acl_join'});
					push @svsjoin,$channel->{'name'}; last;
				} else {
					$irc->send("INVITE $self->{nickname} $channel->{name}")
				}
			}
		}
	}

	if (my $svsjoin = join(',',@svsjoin)) {
		$irc->sock(":$irc->{name} SVSJOIN $self->{nickname} $svsjoin");
	}

	$self->autoAccess(); ## give them access
	## modes in channels they were already in
}

sub chanAccess {
	my $self = shift;
	my $chan = lc shift;
	my $type = shift or return undef;
	my $mode = shift or return undef;

	if ($type eq '+') {
		$self->{'channels'}->{$chan}->{$mode} = 1;
	} elsif ($type eq '-') {
		delete $self->{'channels'}->{$chan}->{$mode}
			if exists $self->{'channels'}->{$chan}->{$mode};
	}
}

sub hasChanAccess {
	my $self = shift;
	my $chan = lc shift;

	foreach (@_) {
		return 0 unless
			defined $self->{'channels'}->{$chan}->{$_}
	}

	return 1;
}

sub autoAccess {
	my $self = shift;

	my @channel = map { CyberArmy::ZebServ::Channel->isRegistred($_) }
		( $#_ > -1 ? (@_) : (keys %{$self->{'channels'}}) );

	foreach my $channel (@channel) {
		my $modes = my $add_modes = my $del_modes = '';
		foreach (qw(admin op hop voice)) {
			my $mode = substr($_,0,1);

			my $has_mode = $self->hasChanAccess($channel->{'name'},$mode);
			my $acl = $channel->{'acl_'.$_};
			if ($acl && $self->CheckGroupList($acl)) {
				unless ($has_mode) {
					$add_modes .= $mode; ## give him his access
					$self->chanAccess($channel->{'name'},'+',$mode);
				}
				last unless $_ eq 'admin';
			} elsif ($has_mode && $channel->{'strict_mode'} eq 'y') {
				$del_modes .= $mode; ## drop him his access
				$self->chanAccess($channel->{'name'},'-',$mode);
			}
		}

		$modes .= '+' . $add_modes if $add_modes;
		$modes .= '-' . $del_modes if $del_modes;
		CyberArmy::ZebServ::Connection->instance()->send(
			"MODE $channel->{name} $modes " . "$self->{nickname} "
				x (length($add_modes) + length($del_modes))
		) if ($modes);

	}
}

sub parted {
	my $self = shift;
	if ($#_ > -1) {
		foreach (@_) {
			delete $self->{'channels'}->{lc $_};
		}
	} else { $self->{'channels'} = {} }
}

sub autoPart { ## part from channels requiring access
	my $self = shift;

	my $irc = CyberArmy::ZebServ::Connection->instance();
	foreach (keys %{$self->{'channels'}}) {
		my $channel = CyberArmy::ZebServ::Channel->isRegistred($_) or next;
		if ($channel->{'strict_mode'} eq 'y') { ## strict privileges
			if (my $modes = join('',(keys %{$self->{'channels'}->{$_}}))) {
				$irc->send("MODE $_ -$modes "
					."$self->{nickname} " x length($modes) );
				$self->{'channels'}->{$_} = {};
			}

			if (defined($channel->{'acl_join'})
					&& $channel->{'strict_access'} eq 'y') {
				## kick the user if he no longer has access
				$irc->send("KICK $_ $self->{nickname} :Access denied");
				$self->parted($_);
			}
		}
	}
}

sub notificationsOn {
	my $self = shift;
	## we need to implement some hooking system here
	CyberArmy::ZebServ::Event::Forum->add_user($self);
	CyberArmy::ZebServ::Event::Message->add_user($self);
}

sub notificationsOff {
	my $self = shift;
	## we need to implement some hooking system here
	CyberArmy::ZebServ::Event::Forum->del_user($self);
	CyberArmy::ZebServ::Event::Message->del_user($self);
}

sub nickChange {
	my $self = shift;
	my $nick = shift or return undef;

	if ($self->inLoginQueue()) {
		delete $inLoginQueue{$self->{'nickname'}}; ## delete old reference
		$self->{'nickname'} = $nick; ## update the object to reflect the new nick
		$inLoginQueue{$nick} = $self; ## finally reference up the new nick
	} else {
		delete $isLoggedIn{$self->{'nickname'}}; ## delete old reference
		$self->{'nickname'} = $nick; ## update object to reflect the new nick
		$isLoggedIn{$nick} = $self; ## finally reference up the new nick
	}
}

package CyberArmy::ZebServ::Channel;

my %isRegistred = ();

sub load {
	my $class = shift;

	my $db = CyberArmy::Database->instance;
	my $getChannelList = $db->prepare_cached('SELECT * FROM irc_channel');
	$getChannelList->execute();

	while (my $channel = $getChannelList->fetchrow_hashref()) {
		$isRegistred{$channel->{'name'}} = bless $channel, __PACKAGE__;
	}

	$getChannelList->finish;
}

sub getChanneList {
	return \%isRegistred;
}

sub new {
	my $class = shift;
	my $self = shift;

	## some sanity checks
	return undef unless	$self->{'name'}
		&& substr($self->{'name'},0,1) eq '#';

	$self->{'name'} = lc $self->{'name'};

	if (exists $isRegistred{$self->{'name'}}) {
		return $isRegistred{$self->{'name'}}
	} else {
		my @fields = sort keys%{$self};
		my @values = @{$self}{@fields};

		CyberArmy::Database->instance->do(
			'INSERT INTO irc_channel ('. join(',',@fields) .')
				VALUES ('. join(',', (('?')x@values)) .')'
		,undef,@values) or return undef;

		$isRegistred{$self->{'name'}} = $self;
		bless $self, $class;

		$self->update(); ## init data
		return $self;
	}
}

sub update {
	my $self = shift;
	my $db = CyberArmy::Database->instance;
	if (my $update = shift) {
		## make sure we can't modify the name
		return undef if ($update->{'name'} and
			($self->{'name'} ne $update->{'name'}));

		my @fields = sort keys%{$update};
		my @values = @{$update}{@fields};

		$db->do(
			'UPDATE irc_channel SET '. join('=?,',@fields) .'=? WHERE name = ?'
		,undef,@values,$self->{'name'}) or return undef;

		foreach (keys %$update) {
			$self->{$_} = $update->{$_};
		}

		return 1;
	} else {
		my $updated = $db->selectrow_hashref(
			'SELECT * FROM irc_channel WHERE name = ?'
				,undef,$self->{'name'}
		) or $self->drop();

		foreach (keys %$updated) {
			$self->{$_} = $updated->{$_}
		}

		return 1;
	}
}

sub isRegistred {
	my $self = shift;

	if (my $channel = shift) {
		return $isRegistred{$channel}
	} elsif (ref($self) eq 'HASH') { ## called as method
		return exists $isRegistred{$self->{'name'}};
	} else { return undef }
}

sub drop {
	my $self = shift;

	delete $isRegistred{$self->{'name'}};

	CyberArmy::Database->instance->do(
		'DELETE FROM irc_channel WHERE name = ?'
	,undef,$self->{'name'});
}



package CyberArmy::ZebServ::Event::User;

use strict;
use CyberArmy::Database;

sub handler { ## watches for logged user modifications
	my $self = shift;
	my $time = shift or return undef;

	my $getLogs = CyberArmy::Database->instance->prepare_cached(
		'		(SELECT type,nickname,msg,logby,`date` AS log_date
					FROM log_users
				WHERE `date` >= FROM_UNIXTIME(?) ORDER BY `date`)
			UNION
				(SELECT type,action_on,msg,action_by,adate AS log_date
					FROM log_brigades_users
				WHERE adate >= FROM_UNIXTIME(?) ORDER BY adate)
		');
	$getLogs->execute($time,$time);

	my $irc = CyberArmy::ZebServ::Connection->instance;
	while (my $log = $getLogs->fetchrow_hashref()) {
		my $user = CyberArmy::ZebServ::User->isOnline($log->{'nickname'})
			or next; ## the user must be Online for us to take action
		$user->Update(); ## update the user's infomations from the db
		$irc->log($user->{'nickname'}. ' ('. $user->nickname. ') was modified');

		## we won't bother checking the event type, for now
		if (my $ban = $user->IsBanned) { ## user got banned
			$irc->log( ## explain what is going on to opers
				"$log->{nickname} was banned by $log->{logby} ".
				"($log->{logby}) till ".gmtime($ban));
			$user->logOut(); ## we will *just* log him out
			## note that we could eventually (k|g)line him
			## user can't login anymore, as long as banned

		} else { ## just rehash his accesses
			$user->autoJoin();
			$user->notificationsOff();
			$user->notificationsOn();
		}
	}

	$getLogs->finish;
}

package CyberArmy::ZebServ::Event::Forum;

use strict;
use HTML::Entities ();

use CyberArmy::Database;

my %forum_list = ();
my %forum_user = ();

{ ## initialization
	my $getForumList = CyberArmy::Database->instance->prepare_cached(
		'SELECT forum,access_group,administrator,moderator FROM forum_list'
	);
	$getForumList->execute();

	%forum_list = %{$getForumList->fetchall_hashref('forum')};

	$getForumList->finish;
}

sub handler { ## notifies new forum posts
	my $self = shift;
	my $time = shift or return undef;

	## get a list of all new posts, along with their forum details
	my $getMsgList = CyberArmy::Database->instance->prepare_cached(
		'SELECT
			forum_list.forum,name,access_group,administrator,moderator,
				mid,thread,subject,author,author_rank
			FROM forum_list JOIN forum_replies ON last_post_id = mid
		WHERE rdate >= FROM_UNIXTIME(?) ORDER BY rdate'
	);
	$getMsgList->execute($time);

	my $irc = CyberArmy::ZebServ::Connection->instance;
	while (my $msg = $getMsgList->fetchrow_hashref()) {

		my $modified = 0; ## modified since our last vist
		if (my $forum = $forum_list{$msg->{'forum'}}) {
			foreach (qw(access_group moderator administrator)) {
				## we are only intrested in access groups
				unless (($forum->{$_}||'') eq ($msg->{$_}||'')) {
					$forum->{$_} = $msg->{$_}; ## we save
					$modified++; ## flag the modification
				}
			}

		} else { ## now this forum has been newly created
			$forum_list{$msg->{'forum'}} = { ## we save it
				forum => $msg->{'forum'},
				access_group => $msg->{'access_group'},
				moderator => $msg->{'moderator'},
				administrator => $msg->{'administrator'},
			};
			$modified++; ## new forum, marked as modified
		}

		if ($modified) {
			#delete $forum_user{$msg->{'forum'}}; ## reset its notification list
			foreach (values %{CyberArmy::ZebServ::User->getOnlineUserList()}) {
				## sequentially rebuild the notification list :
				## goal here is to make sure that notifications aren't
				## interrupted while we are syncing notifications list
				__PACKAGE__->del_user($_,$msg->{'forum'});
				__PACKAGE__->add_user($_,$msg->{'forum'});
			}
		}

		$msg->{'subject'} = HTML::Entities::decode($msg->{'subject'});
		$irc->notice( ## and then we can finally send the actual notification
			"Forum: \"$msg->{subject}\" by $msg->{author_rank} $msg->{author} ".
			"on $msg->{forum} - http://$CyberArmy::Config{SERVER_NAME}/".
			"forum/$msg->{forum}/messages/$msg->{mid}.html"
		, (map {$_->{'nickname'}} values %{$forum_user{$msg->{'forum'}}}));
	}

	$getMsgList->finish;
}

sub add_user {
	my $self = shift;
	my $user = shift or return undef;
	my (@forums) = $#_ >=0 ? (@_) : (keys %forum_list);

	foreach (@forums) {
		my $forum = $forum_list{$_};
		if ($forum && $forum->{'access_group'}) {
			next unless $user->CheckGroupList(
				"$forum->{access_group};".
				"$forum->{administrator};".
				"$forum->{moderator}"
			);
		}

		## i would have used an array here, but it would be too painful to
		## eventually remove elements from it when the user logs out...
		## so yeah, i am resorting to using the scalar value of the object
		$forum_user{ $forum->{'forum'} }->{"$user"} = $user;
	}
}

sub del_user {
	my $self = shift;
	my $user = shift or return undef;
	my @forums = @_;

	foreach ((@forums) or (keys %forum_user)) {
		delete $forum_user{$_}->{"$user"}
	}
}

package CyberArmy::ZebServ::Event::Message;

use strict;
use CyberArmy::Database;

my %message_user = ();

sub handler { ## notifies new private messages
	my $self = shift;
	my $time = shift or return undef;

	my $getMsgList = CyberArmy::Database->instance->prepare_cached(
		'SELECT mid,caID,subject,sender_name,sender_rank FROM cMS
			WHERE rdate >= FROM_UNIXTIME(?) AND viewed = "n" ORDER BY rdate');
	$getMsgList->execute($time);

	my $irc = CyberArmy::ZebServ::Connection->instance;
	while (my $msg = $getMsgList->fetchrow_hashref()) {
		my $user = $message_user{$msg->{'caID'}} or next;
		$irc->notice( ## send the notification
			'Message: from '.
				$msg->{'sender_rank'}.' '.$msg->{'sender_name'}.
				': "'.$msg->{'subject'}.'" - http://'.
					$CyberArmy::Config{'SERVER_NAME'}.
				"/my/messages/?action=view&msg=$msg->{mid}"
		,$user->{'nickname'});
	}

	$getMsgList->finish;
}

sub add_user {
	my $self = shift;
	my $user = shift or return undef;

	$message_user{$user->caID} = $user;
}

sub del_user {
	my $self = shift;
	my $user = shift or return undef;

	delete $message_user{$user->caID};
}
