#!perl -wW
package CyberArmy::WWW::Login;

#
# Logins Manager
#

## Apache module, managing logins

use strict;

use Digest::MD5 ();
use Apache::Util qw(escape_html);

use CyberArmy::User;
use CyberArmy::Utils;
use CyberArmy::WWW::Request;

$CyberArmy::WWW::Login::VERSION = '0.5.5';

sub handler {
	my $r = CyberArmy::WWW::Request->instance( shift );
	my (undef,$node) = split /\// , $r->path_info;
	if (not $node) {
		my $cookies = $r->getParams({from=>'cookies'});
		if (not $cookies->{'login_method'} 
			or ($cookies->{'login_method'} eq 'default')) {
				&loginForm ## use the normal login form
		}	else { $r->redirectTo('/login/'.$cookies->{'login_method'}.'/') }
	} elsif ($node eq 'options') { &loginOptions }
	elsif ($node eq 'basic') { &loginBasic }
	elsif ($node eq 'my') { &loginForm }
	else { $r->exit(404) }

	return 0;
}

sub loginOptions {
	my $r = CyberArmy::WWW::Request->instance;
	if ($r->method eq 'POST') {
		my $posted = $r->getParams({from=>'posted'});
		my $mode = $posted->{'mode'} ? 'secured' : 'normal';
		my $method = $posted->{'method'} ? $posted->{'method'} : 'default';
		my $timeout = $posted->{'timeout'} ? $posted->{'timeout'} : 2;

		$r->err_headers_out->add('Set-Cookie' => 'login_mode=' . $mode
			.'; path=/; expires=Tue, 1-Jan-2020 13:37:33 GMT');
		$r->err_headers_out->add('Set-Cookie' => 'login_method=' . $method
			.'; path=/; expires=Tue, 1-Jan-2020 13:37:33 GMT');
		$r->err_headers_out->add('Set-Cookie' => 'login_timeout=' . $timeout
			.'; path=/; expires=Tue, 1-Jan-2020 13:37:33 GMT');
		#if 'section' posted, then this is being used as part of the profile
		#updates in /my/, so don't print a template
		if(!$posted->{'section'})
		{
			$r->printTemplate('login/options_updated.tmpl');
		}
	} else { 
		$r->printTemplate('login/options.tmpl',$r->getParams({from=>'cookies'}));
	}
}

sub _postLoginRedirect {
	my $r = CyberArmy::WWW::Request->instance;
	
	$r->checkReferer() or $r->redirectTo('/');
	foreach ($r->dir_config->get('IGNORE_REFERER')) {
		$r->redirectTo('/') if $r->checkReferer($_)
	} 
	
	$r->redirectTo($r->header_in('Referer') || '/');
}

sub loginForm {
	my $r = CyberArmy::WWW::Request->instance;
	my ($user) = (CyberArmy::WWW::Request::User->instance);
	if ($r->method eq 'POST') {
		my $posted = $r->getParams({from => 'posted', escapehtml => 1});
		exit (412) unless ($posted->{'username'} && $posted->{'password'});
		# call the database cleansing sub, to clean out expired security lockouts
		&cleanSecurityLockouts;
		if (my $user = CyberArmy::User
				->new(nickname => CyberArmy::Utils::nickFromShowName($posted->{'username'}))) {
			if (not $user->Passhash($posted->{'password'})) {
				my @totalloginfailures;
				@totalloginfailures = &checkLoginAttempts($user->nickname);
				$totalloginfailures[0]++; # increase the failures by one because not updated yet
				if($user->IsSecurityFlagged) {
					$r->errorTemplate('errors/locked.tmpl');
				} else {
					&recordLoginFailure($user->nickname, $ENV{'REMOTE_ADDR'}, $totalloginfailures[0]);
					$r->errorTemplate(403,'login/error_wrong.tmpl', { previousfails => $totalloginfailures[0] });
				}
			} elsif ($user->IsBanned) {
				$r->errorTemplate('errors/banned.tmpl',{user => $user });
			} elsif ($user->IsSecurityFlagged) {
				$r->errorTemplate('errors/locked.tmpl');
			}else { ## all checks okay
				CyberArmy::WWW::Request::User->logIn($user);
				$posted->{'location'} ## for checkReferer()
					&& $r->header_in(Referer => $posted->{'location'});
				_postLoginRedirect();
			}
		} else { $r->printTemplate('login/error_wrong.tmpl') }

	} else {
		my $referer = $r->header_in('Referer');
		CyberArmy::WWW::Utils::escapeHtml($referer) if $referer;
		$r->printTemplate('login/form.tmpl',{location =>$referer});
	}
}

sub loginBasic {
	my $r = CyberArmy::WWW::Request->instance;
	$r->auth_name('CyberArmy Login');
	my ($status, $sent_pw) = $r->get_basic_auth_pw;
	my $auth_name = $r->connection->user;
	$r->log->info('login attempt (basic)'.($auth_name ? ': '.$auth_name : ''));
	if ($auth_name and $sent_pw) {
		if (my $user = CyberArmy::User->new(nickname => CyberArmy::Utils::nickFromShowName($auth_name))) {
			if (not $user->Passhash(escape_html($sent_pw))) {
				$r->note_basic_auth_failure;
				$r->errorTemplate(401,'login/error_wrong.tmpl');
			} elsif ($user->IsBanned) {
				$r->note_basic_auth_failure;
				$r->errorTemplate(401,'errors/banned.tmpl',{user => $user });
			} else { ## all checks okay
				CyberArmy::WWW::Request::User->logIn($user);
				_postLoginRedirect();
			}
		} else { 
			$r->note_basic_auth_failure;
			$r->errorTemplate(401,'login/error_wrong.tmpl');
		}
	} else {
		$r->note_basic_auth_failure;
		$r->errorTemplate(401,'login/error_wrong.tmpl');
	}
}

sub cleanSecurityLockouts {
	# select from the database any failed logins that are more than 30 minutes old and locked the user's account
	# (expired failed logins are deleted directly later in this sub, but we need the nicks here to unlock accounts)
	my $cmd = $CyberArmy::Utils::Database::dbh->prepare(q~
	select nickname from `failedlogins` where TIMEDIFF(NOW(), `failuredt`) > '00:30:00.000000' and `failurelock`='1'
	~);
	
	$cmd->execute;
	while(my @sth = $cmd->fetchrow_array) {
		# go through them and unlock the associated accounts by removing their flag...
		my $cmd = $CyberArmy::Utils::Database::dbh->do(qq~
		UPDATE `users` SET `securityflag`='0' WHERE `nickname`=?
		~, undef, $sth[0]);
		# and removing them from the securityflag group
		my $victim = new CyberArmy::User('nickname' => $sth[0]);
		$victim->Update('delfromgroup' =>	'securityflag');
	}
	
	# now clean out all expired failed logins - we don't need them sat there anymore
	$cmd = $CyberArmy::Utils::Database::dbh->do(qq~
	delete from `failedlogins` where TIMEDIFF(NOW(), `failuredt`) > '00:30:00.000000'
	~);
}

sub recordLoginFailure {
	# records a failed login
	my $nickname = $_[0];
	my $failureip = $_[1];
	my $totalflags = $_[2];
	
	if($totalflags < 4) {
		my $cmd = $CyberArmy::Utils::Database::dbh->do(qq~
		INSERT INTO `failedlogins` (`nickname`, `failuredt`, `failureip`) VALUES ( ?, NOW(), ?)
		~, undef, $nickname, $failureip);
	} elsif($totalflags == 4) {
		my $cmd = $CyberArmy::Utils::Database::dbh->do(qq~
		INSERT INTO `failedlogins` (`nickname`, `failuredt`, `failureip`, `failurelock`) VALUES ( ?, NOW(), ?, 1)
		~, undef, $nickname, $failureip);
	} elsif($totalflags > 4) {
		my $cmd = $CyberArmy::Utils::Database::dbh->do(qq~
		UPDATE `users` SET `securityflag`='1' WHERE `nickname`=?
		~, undef, $nickname);
		my $victim = new CyberArmy::User('nickname' => $nickname);
		$victim->Update('addtogroup' =>	'securityflag');
		$victim->Log(
						'type' => 'modified',
						'logby' => $nickname,
                        'logbycaID' => $victim->caID, 
						'msg' => 'security alert from '. $failureip);
						
	# send a cMS to the person's inbox alerting them
	CyberArmy::Message->new(
				caID => [$victim->caID],
				user => $victim,
				subject => 'Security lockout initiated',
				body => qq~ Hello\n This is an automated cMS to alert you to the fact that your account has been locked down due to a series of failed login attempts from $failureip. Please contact a staff member if you were not responsible for the failed attempts. ~
			);
	}
	
}

sub checkLoginAttempts {
	# looks up failed logins, returns the number of failures retrieved (if any)
	my $nickname = $_[0];
	my @sth;
	my $sth;
	
	my $cmd = $CyberArmy::Utils::Database::dbh->prepare(q~
	select count(nickname) from `failedlogins` where `nickname`=?
	~);
	
	$cmd->execute($nickname);
	
	@sth = $cmd->fetchrow_array;
	
	return $sth[0];
}
	
	

1;
