#!/usr/bin/perl -W
package CyberArmy::WWW::Library;

use strict;
use CyberArmy::User;
use CyberArmy::Library;
use CyberArmy::WWW::Request;
use Encode ();

$CyberArmy::WWW::Library::VERSION = '0.1';


sub handler {
	my $r = shift;
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my (undef,$node) = split(/\//,$r->path_info);
	if (not $node) { &showCategories }
	elsif ($node eq 'category.rss') { &showCategoryRSS }
	elsif ($node eq 'lingua.rss') { &showLinguaRSS }
	elsif ($node eq 'library.rss') { &showLibraryRSS }
	elsif ($node eq 'edition.rss') { &showEditionRSS }
	elsif ($node eq 'license' || $node eq 'licence') {
		$r = CyberArmy::WWW::Request->instance();	
		$r->content_type('text/html');
		$r->printTemplate('library/license.tmpl', {});
	
	}
	elsif ($node eq 'search') { &search }
	elsif ($node eq 'guide') { &showGuide }
	elsif ($node eq 'admin') { &showAdmin }
	elsif ($node eq 'adminPublications') { &showAdminPublications }
	elsif ($node eq 'adminEdition') { &showAdminEdition }
	elsif ($node eq 'article') { &showArticle }
	elsif ($node eq 'review') { &showQueued }
	elsif ($node eq 'author') { &showUserArticles}
	elsif ($node eq 'category') { &showCategory }
	elsif ($node eq 'lingua') { &showLingua }
	elsif ($node eq 'pubArticles') { &showPublicationArticles }
	elsif ($node eq 'edition') { &showEdition }
	elsif ($node eq 'logs') { &showLogs }
	else { return -1 } ## drop it to the default handler

}

sub search {
   my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $categoryList = CyberArmy::Library->getSubCategories(0, 0);
	
		##Perform search if necessary
		my $search; 
		
	    my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
		$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 1}} );
		
		
		if ($params->{'terms'}) {
			return 412 unless
				length($params->{'terms'}) > 2;

			$search = CyberArmy::Library::Search->new($params)
				or return 500;
		}
		#Get the current year
		my (undef,undef,undef,undef,undef,$year,undef,undef) = gmtime(time());
		$year += 1900;
		
		
	$r->content_type('text/html');
	$r->printTemplate(
		'library/search.tmpl', {
			title => "CyberArmy Library :: Search",
			categories => $categoryList,
			category => $params->{'category'} ? $params->{'category'} : undef,#search category, if set
			thisyear => $year,
			terms => $params->{'terms'} ? $params->{'terms'} : undef,#search terms, if set
			results => $search,
			match => $params->{'match'} ? $params->{'match'} : undef,#match type, if set
			in_title => (!$params->{'in_title'} && $params->{'terms'}) ? 'OFF' : $params->{'in_title'},#do/don't search in title, if set
			in_summ => (!$params->{'in_summ'} && $params->{'terms'}) ? 'OFF' : $params->{'in_summ'},#do/don't search in summary, if set
			in_body => $params->{'in_body'} ? $params->{'in_body'} : undef,#do/don't search in article body , if set
			in_author => $params->{'in_author'} ? $params->{'in_author'} : undef,#do/don't search in authors, if set
			num => $params->{'num'} ? $params->{'num'} : undef,#num results to display, if set
			date => $params->{'date'} ? $params->{'date'} : undef,
			day_1 => $params->{'day_1'} ? $params->{'day_1'} : undef,
			month_1 => $params->{'month_1'} ? $params->{'month_1'} : undef,
			year_1 => $params->{'year_1'} ? $params->{'year_1'} : undef,
			day_2 => $params->{'day_2'} ? $params->{'day_2'} : undef,
			month_2 => $params->{'month_2'} ? $params->{'month_2'} : undef,
			year_2 => $params->{'year_2'} ? $params->{'year_2'} : undef
		}
	);
	
	return 0;

}


####
# sub showCategories
#
# Displays the main Library index page. This is the default
# page when viewing the library section of the site.
# No access restrictions
##

sub showCategories {
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $categoryList = CyberArmy::Library->getSubCategories(0, 0);
	my $languageList = CyberArmy::Library->getLanguages();
	my $admin = "";
	$admin .=(($user) && CyberArmy::Library->isAdmin($user)) ? qq~ [<a href="/library/admin">Admin</a>] ~ : "";
	$admin .= (($user) && CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isPublicationAdmin($user)) ? qq~ [<a href="/library/adminPublications">Publication Center</a>] ~ : "";
	$admin .=(($user) && (CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isEditor($user))) ? qq~ [<a href="/library/review">Queued Submissions</a>] ~ : "";
	my $user_options = ($user) ? qq~[<a href="/my/articles/?action=create">Create Article</a>]~ : "";
	my $publicationList = CyberArmy::Library->listPublicPublications();
	my $latest_articles = CyberArmy::Library->GetLatest(15);
	
	$r->content_type('text/html');
	$r->printTemplate(
		'library/index.tmpl', {
			title => "CyberArmy Library",
			categories => $categoryList,
			languages => $languageList,
			publications => $publicationList,
			latest_articles => $latest_articles,
			user_options => $user_options,
			admin_options => $admin
		}
	);
	
	return 0;
}


####
# sub showAdminPublications
#
# The 'publication center'. Enables admins to create new publications or edit existing ones.
# Publications can have their name/admin list/description changed, and editions added. Editions
# are listed and links to their admin page.
#
# Access restrictions: Only administrators and publication editors can see this section. Publication editors
# cannot add publications, but can edit and view publications they have access to.
##

sub showAdminPublications {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 1}} );
	$params->{'action'} ||= 'view';
	
	
	return 403 unless $user && 
		(CyberArmy::Library->isAdmin($user) || 
		CyberArmy::Library->isPublicationAdmin($user));

	my $admin = CyberArmy::Library->isAdmin($user);
	if ($params->{'action'} eq 'add_publication'){
		return 403 unless $admin;
		my $message="Couldn't create publication.";
		if($params->{'publication_name'}){
		
			if(CyberArmy::Library->createPublication($params->{'publication_name'}, $params->{'description'} ? $params->{'description'} : "", $user)!='0E0'){
					$message="Publication successfully created.";
			}
		}
		
		my $publicationList = CyberArmy::Library->listPublications();
		my $deletablePublications = CyberArmy::Library->listDeletablePublications();
		foreach my $publication (@$publicationList){	
			next unless (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$publication->{administrator}"));
			$publication->{access} = 'granted';
		}
		$r->printTemplate('library/admin_publications.tmpl',{
			publications => $publicationList,
			deletable => $deletablePublications,
			admin => $admin,
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'del_publication'){
		return 403 unless CyberArmy::Library->isAdmin($user);
		my $message="Couldn't delete publication.";
		if($params->{'publication_id'}){
		
			if(CyberArmy::Library->deletePublication($params->{'publication_id'}, $user)!='0E0'){
					$message="Publication successfully deleted.";
			}
		}
		
		my $publicationList = CyberArmy::Library->listPublications();
		my $deletablePublications = CyberArmy::Library->listDeletablePublications();
		foreach my $publication (@$publicationList){	
			next unless (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$publication->{administrator}"));
			$publication->{access} = 'granted';
		}
		$r->printTemplate('library/admin_publications.tmpl',{
			publications => $publicationList,
			deletable => $deletablePublications,
			admin => $admin,
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'edit_publication'){
		if($params->{'id'}){
		    
			my $deletableEditions= CyberArmy::Library->listDeletableEditions($params->{'id'});
			my $publication = CyberArmy::Library->getPublication($params->{'id'});
				$r->printTemplate('library/edit_publication.tmpl',{
					publication => $publication,
					admin => $admin,
					deletable => $deletableEditions
				});
		
		}
		else {
			my $publicationList = CyberArmy::Library->listPublications();
			my $deletablePublications = CyberArmy::Library->listPublications();
			foreach my $publication (@$publicationList){	
				next unless (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$publication->{administrator}"));
				$publication->{access} = 'granted';
			}

			$r->content_type('text/html');
			$r->printTemplate(
				'library/admin_publications.tmpl', {
					publications => $publicationList,
					deletable => $deletablePublications,
					admin => $admin,
					message => "Please select a publication to edit"
				}
			);		
		}
	} elsif ($params->{'action'} eq 'doedit_publication'){
		my $message="Failed to edit the selected publication.";
		if($params->{'id'} && $params->{'publication_name'}){
			if(!$admin)
			{
				my $publication = CyberArmy::Library->getPublication($params->{'id'});
				$params->{'publication_name'} = $publication->{title};
			}
			my $desc = $params->{'new_desc'} ? $params->{'new_desc'} : "";
			my $admin = $params->{'admin'} ? $params->{'admin'} : "";
			if(CyberArmy::Library->editPublication($params->{'id'},$params->{'publication_name'},$desc, $admin, $user)!='0E0'){
				$message = "Publication successfully edited.";
			}
		}
		my $publication = CyberArmy::Library->getPublication($params->{'id'});
		my $deletableEditions = CyberArmy::Library->listDeletableEditions($params->{'id'});
			
		$r->printTemplate('library/edit_publication.tmpl',{
			publication => $publication,
			deletable => $deletableEditions,
			admin => $admin,
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'add_edition'){
		my $message="Couldn't create edition.";
		if(!$params->{'id'}) {
			$message="Couldn't create edition: publication unknown";
			my $publicationList = CyberArmy::Library->listPublications();
			my $deletablePublications = CyberArmy::Library->listDeletablePublications();
			foreach my $publication (@$publicationList){	
				next unless (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$publication->{administrator}"));
				$publication->{access} = 'granted';
			}

			$r->printTemplate('library/admin_publications.tmpl',{
						publications => $publicationList,
				deletable => $deletablePublications,
				admin => $admin,
				message => $message#success/failure message
			});
		}
		elsif($params->{'edition_name'}){
			if(CyberArmy::Library->createEdition($params->{'id'}, $params->{'edition_name'}, $params->{'description'} ? $params->{'description'} : "", $user)!='0E0'){
					$message="Edition successfully created.";
			}
		}
		my $publication = CyberArmy::Library->getPublication($params->{'id'});
		my $deletableEditions= CyberArmy::Library->listDeletableEditions($params->{'id'});
			
		$r->printTemplate('library/edit_publication.tmpl',{
			publication => $publication,
			deletable => $deletableEditions,
			message => $message
		});
	} elsif ($params->{'action'} eq 'del_edition'){
		my $message="Couldn't delete edition.";
		if($params->{'edition_id'}){
		
			if(CyberArmy::Library->deleteEdition($params->{'edition_id'}, $user)!='0E0'){
					$message="Edition successfully deleted.";
			}
		}
		my $publication = CyberArmy::Library->getPublication($params->{'id'});
		my $deletableEditions= CyberArmy::Library->listDeletableEditions($params->{'id'});
			
		$r->printTemplate('library/edit_publication.tmpl',{
			publication => $publication,
			deletable => $deletableEditions,
			message => $message
		});
	} elsif ($params->{'action'} eq 'edit_edition'){
		&showAdminEdition;
		return;
	}		
	else {
		my $publicationList = CyberArmy::Library->listPublications();
		my $deletablePublications = CyberArmy::Library->listDeletablePublications();
		foreach my $publication (@$publicationList){	
			next unless (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$publication->{administrator}"));
			$publication->{access} = 'granted';
		}

		$r->content_type('text/html');
		$r->printTemplate(
			'library/admin_publications.tmpl', {
				publications => $publicationList,
				deletable => $deletablePublications,
				admin => $admin
			}
		);
	}
	
	return 0;
}


####
# sub showAdminEdition
#
# Enables admins to edit the edition's title, description and status. Lists all articles
# added to the edition (regardless of there status) and gives the option to remove them.
# 
# Access restrictions: Only administrators and publication editors can see this section. Publication admin
# can access if they have access to this edition's publication. 
##

sub showAdminEdition{
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	return 403 unless ($user);
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );

	

	if ($params->{'id'}){
		my $edition = CyberArmy::Library->getEdition($params->{'id'});
		if($edition!='0E0'){
			if(!(CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$edition->{administrator}"))){					
				return 403;
			}
			my $message;
			if($params->{'action'}){
				if ($params->{'action'} eq 'remove_article'){
					if($params->{'article_id'}){				
						CyberArmy::Library->removeFromEdition($params->{'id'}, $params->{'article_id'}, $user);
						#get new copy of the edition			
						$edition = CyberArmy::Library->getEdition($params->{'id'});
						$message = "Article successfully removed from edition.";

					}
					else {
						$message = "Please select an article to remove";
					}
				} elsif ($params->{'action'} eq 'edit_edition'){
					$message="Failed to edit the selected edition.";
					if($params->{'id'} && $params->{'edition_name'}){
						my $desc = $params->{'new_desc'} ? $params->{'new_desc'} : "";
						if(CyberArmy::Library->editEdition($params->{'id'},$params->{'edition_name'},$desc, $user)!='0E0'){
							$message = "Edition successfully edited.";
						}
					}
					$edition = CyberArmy::Library->getEdition($params->{'id'});

				} elsif ($params->{'action'} eq 'edit_edition_status'){
					$message="Failed to update the edition status.";
					if($params->{'id'} && $params->{'new_status'}){
						if(CyberArmy::Library->editEditionStatus($params->{'id'},$params->{'new_status'}, $user)!='0E0'){
							$message = "Edition status successfully updated.";
						}
					}
					$edition = CyberArmy::Library->getEdition($params->{'id'});

				}
			}
			$r->printTemplate('library/admin_edition.tmpl',{
				edition => $edition,
				message => $message ? $message : undef
				});
		}	
		else {
			return 404;
		}
	} 
	else {
		return 404
	}
	
	return 0;
}


####
# sub showAdmin
#
# Displays admin page that enables a user to add/rename/delete categories.
# Access restrictions: Requires admin access.
#
# TODO: enable archiving of categories. This enables us to remove the display
# of articles to users, without losing or having to shuffle content.
##

sub showAdmin {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 1}} );
	$params->{'action'} ||= 'view';
	
	return 403 unless $user && CyberArmy::Library->isAdmin($user);
    my $languages = CyberArmy::Library->listLanguages();    
	my $categoryList = CyberArmy::Library->listCategories();
	my $structuredCategoryList = CyberArmy::Library->getSubCategories(0, 0);
	my $deletableList = CyberArmy::Library->listDeletableCategories();

    if ($params->{'action'} eq 'add_language') {
         ($params->{'language'}) or return 412;
        CyberArmy::Library->addLanguage($params->{'language'});
    	$languages = CyberArmy::Library->listLanguages();
		$r->printTemplate('library/admin.tmpl',{
			languages => $languages,
			categories => $categoryList,
			deletable => $deletableList,
			structuredCats => $structuredCategoryList,
			message => 'Language Added: '.$params->{'language'}
		});
		return 1;
    }  elsif ($params->{'action'} eq 'del_language') {
        $params->{'language'} or return 412;
        CyberArmy::Library->delLanguage($params->{'language'}); 
    	$languages = CyberArmy::Library->listLanguages();
		$r->printTemplate('library/admin.tmpl',{
			languages => $languages,
			categories => $categoryList,
			deletable => $deletableList,
			structuredCats => $structuredCategoryList,
			message => 'Language Deleted'
		});
		return 1;     
	} elsif ($params->{'action'} eq 'add_category'){
		my $message="Couldn't create category.";
		if($params->{'category_name'}){
			my $parent = $params->{'parent'} ? $params->{'parent'} : '0';
			if(CyberArmy::Library->createCategory($params->{'category_name'},$parent, $params->{'description'} ? $params->{'description'} : "", $user)!='0E0'){
					$message="Category successfully created.";
			}
		}
		
		my $categoryList = CyberArmy::Library->listCategories();
		my $structuredCategoryList = CyberArmy::Library->getSubCategories(0, 0);
		my $deletableList = CyberArmy::Library->listDeletableCategories();
		
		$r->printTemplate('library/admin.tmpl',{
			languages => $languages,
			categories => $categoryList,
			deletable => $deletableList,
			structuredCats => $structuredCategoryList,
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'del_category'){
		my $message = "Couldn't delete the selected category.";
		if($params->{'category_id'}){
			if(CyberArmy::Library->deleteCategory($params->{'category_id'}, $user)!='0E0'){
				$message = "Category successfully deleted.";
			}
		}
		$categoryList = CyberArmy::Library->listCategories();
		$structuredCategoryList = CyberArmy::Library->getSubCategories(0, 0);
		$deletableList = CyberArmy::Library->listDeletableCategories();

		$r->printTemplate('library/admin.tmpl',{
			languages => $languages,
			categories => $categoryList,
			deletable => $deletableList,
			structuredCats => $structuredCategoryList,
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'doedit_category'){
		my $message="Failed to edit the selected category.";
		if($params->{'id'} && $params->{'category_name'}){
			my $desc = $params->{'new_desc'} ? $params->{'new_desc'} : "";
			if(CyberArmy::Library->editCategory($params->{'id'},$params->{'category_name'},$desc, $user)!='0E0'){
				$message = "Category successfully edited.";
			}
		}
		$categoryList = CyberArmy::Library->listCategories();
		$structuredCategoryList = CyberArmy::Library->getSubCategories(0, 0);
		$deletableList = CyberArmy::Library->listDeletableCategories();
		
		$r->printTemplate('library/admin.tmpl',{
			languages => $languages,
			categories => $categoryList,
			deletable => $deletableList,
			structuredCats => $structuredCategoryList,
			message => $message#success/failure message
		});
	}elsif ($params->{'action'} eq 'edit_category'){
		if($params->{'old_name'}){
		
			my $category = CyberArmy::Library->getCategory($params->{'old_name'});
				$r->printTemplate('library/edit_category.tmpl',{
					title => $category->{title},
					description => $category->{description},
					id => $category->{id}
				});
		
		}
		else {		
				$r->printTemplate('library/admin.tmpl',{
					languages => $languages,
					categories => $categoryList,
					deletable => $deletableList,
					structuredCats => $structuredCategoryList,
					message => "Please select a category to edit"
		});
		
		
		}
	}	
	else{	
		$r->content_type('text/html');
		$r->printTemplate(
			'library/admin.tmpl', {
				languages => $languages,
				categories => $categoryList,
				deletable => $deletableList,
				structuredCats => $structuredCategoryList
			}
		);
	}
	
	return 0;

}




####
# sub showArticle
#
# Displays an article to the user, and any admin options available.
#
# Access restrictions: Depend on article status.
##

sub showArticle {
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 2}} );

	my $authed = $user ? 'true' : 'false';


	my (undef,$node,$subnode) = split /\//, $r->path_info;
	if(!$params->{'id'} && $subnode =~ /^[0-9]+$/)
	{
		$params->{'id'} = $subnode
	}
	if ($params->{'id'}){
		my $article;
		if($user) {
			$article = CyberArmy::Library->getArticle($params->{'id'}, $user->caID);
		}
		else {
			$article = CyberArmy::Library->getArticle($params->{'id'});		
		}
		if($article!='0E0'){
			my $message;
			if($params->{'action'}){
				if($params->{'action'} eq 'add_to_edition') {
				#TODO: check auth
					$message = "Failed to add article to edition";	
					if($params->{'edition_id'}) {
						my $edition = CyberArmy::Library->getEdition($params->{'edition_id'});
						if($edition != '0E0') {
							if(CyberArmy::Library->isInEdition($article->{'id'},$edition->{'id'}) == 0) {
								if(CyberArmy::Library->addArticleToEdition($params->{'edition_id'}, $params->{'id'}, $params->{'weight'}, $user)!='0E0'){
									$message = 'Article successfully added to edition: '.$edition->{'title'};
								}
							}
							else {
								$message="Failed to add article to edition, article has already been added.";						
							}
						}
						else {
					  		$message="Failed to add article to edition, edition does not exist.";						
						}
					}
					else {
					  $message="Failed to add article to edition, no edition specified.";
					}
				}
			
			}
			my $editor = 'false';
			my $author = 'false';
			my @publications;
			my $count = 0;
			my $editionList = CyberArmy::Library->listEditions();
			if($user) {
				if(CyberArmy::Library->isAdmin($user)||
					CyberArmy::Library->isEditor($user)
					) {
					$editor='true';
				}
				if($user->caID eq $article->{'author_caID'}){
					$author='true';
				}
				if($article->{'status'} ne "draft" ){
					foreach my $edition (@$editionList){	
						next unless ((CyberArmy::Library->isInEdition($article->{'id'},$edition->{'id'}) == 0) && (CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$edition->{administrator}")));
						$edition->{access} = 'granted';
						$count++;
					}
				}
					
			}
			my $authorised = 'true';
			$authorised = ($article->{'status'} eq "draft" && $author eq "false")? 'false' : $authorised;
			$authorised = ($article->{'status'} eq "deleted" && $editor eq "false")?'false' : $authorised;
			$authorised = ($article->{'status'} eq "review" && $editor eq "false" && $author eq "false")?'false' : $authorised;
			
			if($authorised eq 'true'){
				if($params->{'action'} && $params->{'action'} eq 'edit'){				
					return 403 unless (($article->{'status'} eq 'draft' && $author eq "true")
					 || ($article->{'status'} ne 'draft' && $editor eq 'true'));
					editArticle($editor, $author, $article, $params);
				}
				elsif($params->{'action'} && $params->{'action'} eq 'status'){
					return 403 unless ($article->{'status'} ne 'draft' && $editor eq 'true');
					editArticleStatus($article, $params);
				}
				else {
					if($params->{'action'} && $params->{'action'} eq 'vote' && $params->{'vote'} ) {
						CyberArmy::Library->castVote($user, $params->{'vote'}, $article->{'id'})
							or return 404;
						$article = CyberArmy::Library->getArticle($params->{'id'}, $user->caID);
					}
					$r->printTemplate('library/article.tmpl',{
						article => $article,
						editor => $editor,
						author => $author,
						editions => $count>0 ? $editionList : undef,
						message => $message ? $message : undef,
						authed => $authed
					});
				}
			}
			else {
				return 404;		
			}
		}
		else {
			return 404;
		}
	} 
	else{
		return 404;
	}
	
	return 0;
}




####
# sub editArticle
#
# Used to edit the content of an article (not the status!).
# Called from sub showArticle();
#
# Access restrictions: Requires admin or editor access, or author if the article is a draft
##

sub editArticle {
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	my ($editor, $author, $article, $params) = @_;
	my $categoryList = CyberArmy::Library->getSubCategories(0, 0);
	my $languageList = CyberArmy::Library->listLanguages();
	
	my $author_user = new CyberArmy::User('caID' => $article->{'author_caID'});
	
	my $projects;
	if($author_user)
	{
		$projects= $author_user->getProjects();
	}
	if ($params->{'preview'}) {	
		    $r->printTemplate('library/edit-article.tmpl',{ 
			articleid => $article->{'id'},
			categories => $categoryList,
			languages => $languageList,
			title => $params->{'title'},
			preview_summary => substr($params->{'summary'}, 0, 255),
			summary => $params->{'summary'},
			body => $params->{'body'},
			language => $params->{'language'},
			category => $params->{'category'},
			draft => $article->{'status'} eq 'draft' ? 1 : undef,
			article_proj => $params->{'project'},
			projects => $projects,
			admin => CyberArmy::Library->isAdmin($user),
			preview => 1
			});
	}
	elsif ($params->{'update'}) {
			#remove line feeds
			$params->{'body'} =~ s/\r//gs;
			#Reduce line breaks
			$params->{'body'} =~ s/\n{3,}/\n\n/gs;

			if($params->{'title'} && 
				$params->{'summary'} &&
				$params->{'body'} && 
				$params->{'category'}) {
					
				my $category = CyberArmy::Library->getCategory($params->{'category'});
				if($category=='0E0'){
					#category does not exist
					$r->printTemplate('library/edit-article.tmpl',{ 
						message => "Category does not exist, it is possible the category was deleted and is no longer available. Please try again.",
						articleid => $article->{'id'},
						categories => $categoryList,
						languages => $languageList,
						title => $params->{'title'},
						preview_summary => substr($params->{'summary'}, 0, 255),
						summary => $params->{'summary'},
						body => $params->{'body'},
						language => $params->{'language'},
						category => $params->{'category'},
						draft => $article->{'status'} eq 'draft' ? 1 : undef,
						article_proj => $params->{'project'},
						projects => $projects,
						admin => CyberArmy::Library->isAdmin($user),
						preview => 1
					});
					return 1;
				}
				if($author eq "true" && $article->{'status'} eq 'draft'){
					CyberArmy::Library->update_article(
								$article->{'id'},
								$params->{'title'},
								$params->{'summary'},
								$params->{'body'},
								$params->{'category'},
								$params->{'project'},
								$params->{'language'}
					) or return 500;
				}
				else {
					#they must be an admin
					#make sure the action is logged
					CyberArmy::Library->edit_article(
								$user,
								$article->{'id'},
								$params->{'title'},
								$params->{'summary'},
								$params->{'body'},
								$params->{'category'},
								$params->{'project'},
								$params->{'language'}
					) or return 404;
				}
				#get new version of the article
				$article = CyberArmy::Library->getArticle($article->{'id'}, $user->caID);
				$r->printTemplate('library/article.tmpl',{
					article => $article,
					editor => $editor,
					author => $author
				});
			} else {

			       $r->printTemplate('library/edit-article.tmpl',{ 
					articleid => $article->{'id'},
					message => "Mandatory Field(s) Missing - Please try again.",
					categories => $categoryList,
					languages => $languageList,
					title => $params->{'title'},
					preview_summary => substr($params->{'summary'}, 0, 255),
					summary => $params->{'summary'},
					body => $params->{'body'},
					category => $params->{'category'},
					language => $params->{'language'},
					draft => $article->{'status'} eq 'draft' ? 1 : undef,
					article_proj => $params->{'project'},
					projects => $projects,
					admin => CyberArmy::Library->isAdmin($user),
					preview => 1
					});

			}

	}elsif ($params->{'submit'}) {
			#remove line feeds
			$params->{'body'} =~ s/\r//gs;
			#Reduce line breaks
			$params->{'body'} =~ s/\n{3,}/\n\n/gs;

			if($params->{'title'} && 
				$params->{'summary'} &&
				$params->{'body'} && 
				$params->{'category'}) {
					
				my $category = CyberArmy::Library->getCategory($params->{'category'});
				if($category=='0E0'){
					#category does not exist
					$r->printTemplate('library/edit-article.tmpl',{ 
						message => "Category does not exist, it is possible the category was deleted and is no longer available. Please try again.",
						articleid => $article->{'id'},
						categories => $categoryList,
						languages => $languageList,
						title => $params->{'title'},
						preview_summary => substr($params->{'summary'}, 0, 255),
						summary => $params->{'summary'},
						body => $params->{'body'},
						language => $params->{'language'},
						draft => $article->{'status'} eq 'draft' ? 1 : undef,
						article_proj => $params->{'project'},
						projects => $projects,
						admin => CyberArmy::Library->isAdmin($user),
						preview => 1
					});
					return 1;
				}
				if($author eq "true" && $article->{'status'} eq 'draft'){
					CyberArmy::Library->update_article(
								$article->{'id'},
								$params->{'title'},
								$params->{'summary'},
								$params->{'body'},
								$params->{'category'},
								$params->{'project'},
								$params->{'language'}
					) or return 500;
				}
				CyberArmy::Library->submit_article(
					$user,
					$article->{'id'}
				) or return 404;
				#get new version of the article
				$article = CyberArmy::Library->getArticle($article->{'id'}, $user->caID);
				$r->printTemplate('library/article.tmpl',{
					article => $article,
					editor => $editor,
					author => $author
				});
			} else {

			       $r->printTemplate('library/edit-article.tmpl',{ 
					articleid => $article->{'id'},
					message => "Mandatory Field(s) Missing - Please try again.",
					categories => $categoryList,
					languages => $languageList,
					title => $params->{'title'},
					preview_summary => substr($params->{'summary'}, 0, 255),
					summary => $params->{'summary'},
					body => $params->{'body'},
					category => $params->{'category'},
					language => $params->{'language'},
					draft => $article->{'status'} eq 'draft' ? 1 : undef,
					article_proj => $params->{'project'},
					projects => $projects,					
					admin => CyberArmy::Library->isAdmin($user),
					preview => 1
					});

			}

	}
	else {
		$r->printTemplate('library/edit-article.tmpl',{
			articleid => $article->{'id'},
			editor => $editor,
			author => $author,
			categories => $categoryList,
			languages => $languageList,
			title => $article->{'title'},
			summary => $article->{'summary'},
			body => $article->{'body'},
			category => $article->{'category_id'},
			language => $article->{'language_id'},
			draft => $article->{'status'} eq 'draft' ? 1 : undef,
			article_proj => $article->{'project_id'},
			admin => CyberArmy::Library->isAdmin($user),
			projects => $projects
		});
	}
	
	return 0;
}





####
# sub editArticleStatus
#
# Used to edit the status of an article (not the content!).
# Called from sub showArticle();
#
# Access restrictions: Requires admin or editor access
##

sub editArticleStatus {
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	my ($article, $params) = @_;
	my $categoryList = CyberArmy::Library->listCategories();
	
	return 403 unless ($user) &&
		(CyberArmy::Library->isAdmin($user) || 
			CyberArmy::Library->isEditor($user));

	if ($params->{'update'}) {

		if($params->{'newstatus'}) {
			if($article->{'status'} ne $params->{'newstatus'}) {
				if($params->{'newstatus'} eq 'live') {
					CyberArmy::Library->accept_article(
						$user,
						$article->{'id'}
					) or return 404;

				}
				if($params->{'newstatus'} eq 'publication-only') {
					CyberArmy::Library->setPublicationOnly(
						$user,
						$article->{'id'}
					) or return 404;

				}
				elsif($params->{'newstatus'} eq 'review') {
					CyberArmy::Library->review_article(
						$user,
						$article->{'id'}
					) or return 404;

				}
				elsif($params->{'newstatus'} eq 'delete') {
					CyberArmy::Library->delete_article(
						$user,
						$article->{'id'}
					) or return 404;


				}
				else {
					#TODO: error?
				}
			}
			#get new version of the article
			$params->{'action'} = undef;
			showArticle();
		} else {

		       #TODO: error

		}

	}
	else {
		$r->printTemplate('library/edit-article-status.tmpl',{
			article => $article
		});
	}
	
	return 0;
}


####
# sub showQueued
#
# Lists all unreviewed articles.
#
# Access restrictions: Requires admin or editor access.
##
sub showQueued {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 1}} );
	$params->{'action'} ||= 'view';
	
	return 403 unless ($user) && 
		(CyberArmy::Library->isAdmin($user) ||
			CyberArmy::Library->isEditor($user));
		
	#get categories that contain articles
	my $categories = CyberArmy::Library->listCatWithUnreviewed();
	foreach my $category (@$categories){		
		my $articles = CyberArmy::Library->getCatUnreviewedArticles($category->{id});
		$category->{articles} = $articles;
	}
	
	#for each category, get the articles under review
	
	#show template
	$r->printTemplate('library/list_queued.tmpl',{
		categories => $categories
	});

	return 0;

}

####
# sub showCategory
#
# Lists all 'live' articles in a category.
#
# Access restrictions: No access restrictions
##
sub showCategory {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $column = 'article.date_submitted'; #default order
	my $orderby = $column.' DESC';
			
			
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );
	
	
	if($params->{'column'} && $params->{'order'}){ #set user defined order
		if((grep $_ eq $params->{'column'}, ('date', 'author', 'title')) && (grep $_ eq $params->{'order'}, ('asc', 'desc'))){
		
			if($params->{'column'} eq "date")
			{
				$column ="article.date_submitted";
			}
			elsif($params->{'column'} eq "author")
			{
				$column ="users.nickname";			
			}
			elsif($params->{'column'} eq "title")
			{
				$column ="article.title";			
			}
			$orderby = $column.' '.$params->{'order'};
		}
	}
	my (undef,$node,$subnode) = split /\//, $r->path_info;
	if(!$params->{'id'} && $subnode =~ /^[0-9]+$/)
	{
		$params->{'id'} = $subnode
	}
	if ($params->{'id'}){
		my $category = CyberArmy::Library->getCategory($params->{'id'});
		if($category!='0E0'){
			if($params->{'action'} && $params->{'action'} eq 'admin')
			{				
				return 403 unless (CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isEditor($user));
				my $message;
			    if ($params->{'add_resource'}) {
			        $message = 'Failed to add resource';
			         ($params->{'url'} && $params->{'title'}) or return 412;
			        $message = 'Resource Added';
			        CyberArmy::Library->addResource($params->{'url'},$params->{'id'},
			        					$params->{'title'},$params->{'description'});
			    }  elsif ($params->{'del_resource'}) {
			        $message = 'Failed to delete resource';
			        $params->{'resource'} or return 412;
			        CyberArmy::Library->delResource($params->{'resource'},$params->{'id'});
			        $message = 'Resource Deleted';
				}
				my $category = CyberArmy::Library->getCategory($params->{'id'});	
				$r->printTemplate('library/category_admin.tmpl',{
					category => $category,
					message => $message
				});
				return 0;
			}
			#get page number
			my $page = $params->{'page'}; my $all = 0;
			if (not $page) { $page = 1; }
			elsif (not ($page =~ /^\d+$/)) { $page = 1; }
			my $articleCount = CyberArmy::Library->countCategoryArticles($category->{id});
		
			my $pages = $articleCount / 30;
			$pages = int $pages + 1 unless ($pages == int $pages);
			my $articles;
			if($page==1){
				$articles = CyberArmy::Library->getCategoryArticles($params->{'id'}, $orderby, 30, 0);
			}else{
				$articles = CyberArmy::Library->getCategoryArticles($params->{'id'}, $orderby, 30,($page-1) * 30);
			}	
			
			my $subcategories = CyberArmy::Library->getSubCategories($category->{'id'}, 0);
			my $parentcategories = CyberArmy::Library->getParentCategories($category->{'parent_id'}, 0);
			$r->printTemplate('library/category.tmpl',{
				category => $category,
				articles => $articles,
				order => $params->{'order'} ? $params->{'order'} : 'desc',
				column => $params->{'column'} ? $params->{'column'} : 'date',
				pages => $pages,
				page => $page,
				subcategories => $subcategories,
				parentcategories => $parentcategories,
				editor => CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isEditor($user) ? 1 : undef
			});
		}	
		else {
			return 404;
		}
	} 
	else{
		return 404;
	}
	
	return 0;
}


####
# sub showLingua
#
# Lists all 'live' articles in specified Language.
#
# Access restrictions: No access restrictions
##

sub showLingua {
		my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $column = 'article.date_submitted'; #default order
	my $orderby = $column.' DESC';
			
			
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );
	
	
	if($params->{'column'} && $params->{'order'}){ #set user defined order
		if((grep $_ eq $params->{'column'}, ('date', 'author', 'title')) && (grep $_ eq $params->{'order'}, ('asc', 'desc'))){
		
			if($params->{'column'} eq "date")
			{
				$column ="article.date_submitted";
			}
			elsif($params->{'column'} eq "author")
			{
				$column ="users.nickname";			
			}
			elsif($params->{'column'} eq "title")
			{
				$column ="article.title";			
			}
			$orderby = $column.' '.$params->{'order'};
		}
	}
	my (undef,$node,$subnode) = split /\//, $r->path_info;
	if(!$params->{'id'} && $subnode =~ /^[0-9]+$/)
	{
		$params->{'id'} = $subnode
	}
	if ($params->{'id'}){
		my $lingua = CyberArmy::Library->getLingua($params->{'id'});	
		if($lingua!='0E0'){
			my $page = $params->{'page'}; my $all = 0;
			if (not $page) { $page = 1; }
			elsif (not ($page =~ /^\d+$/)) { $page = 1; }
			my $articleCount = CyberArmy::Library->countCategoryArticles($lingua->{id});
		
			my $pages = $articleCount / 30;
			$pages = int $pages + 1 unless ($pages == int $pages);
			my $articles;
			if($page==1){
				$articles = CyberArmy::Library->getLinguaArticles($params->{'id'}, $orderby, 30, 0);
			}else{
				$articles = CyberArmy::Library->getLinguaArticles($params->{'id'}, $orderby, 30,($page-1) * 30);
			}	
			
		#	my $subcategories = CyberArmy::Library->getSubCategories($category->{'id'}, 0);
		#	my $parentcategories = CyberArmy::Library->getParentCategories($category->{'parent_id'}, 0);
			$r->printTemplate('library/lingua.tmpl',{
				lingua => $lingua,
				articles => $articles,
				order => $params->{'order'} ? $params->{'order'} : 'desc',
				column => $params->{'column'} ? $params->{'column'} : 'date',
				pages => $pages,
				page => $page,
				editor => CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isEditor($user) ? 1 : undef
			});
		}	
		else {
			return 404;
		}
	} 
	else{
		return 404;
	}
	
	return 0;								
}

####
# sub showPublicationArticles
#
# Lists all articles that are marked for publication release only. These articles will not
# be released as an article on their own, but can be added to publications/editions. This page
# enables publication admin to locate and add articles to editions.
#
# Access restrictions: library admin, editor, or publication admin. Only need to be admin of one
# or more publications to gain access.
##
sub showPublicationArticles {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $column = 'article.date_submitted'; #default order
	my $orderby = $column.' DESC';
	return 403 unless (CyberArmy::Library->isAdmin($user) || CyberArmy::Library->isEditor($user) || CyberArmy::Library->isPublicationAdmin($user));
	
			
			
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );

	if($params->{'column'} && $params->{'order'}){ #set user defined order
		if((grep $_ eq $params->{'column'}, ('date', 'author', 'title')) && (grep $_ eq $params->{'order'}, ('asc', 'desc'))){
		
			if($params->{'column'} eq "date")
			{
				$column ="article.date_submitted";
			}
			elsif($params->{'column'} eq "author")
			{
				$column ="users.nickname";			
			}
			elsif($params->{'column'} eq "title")
			{
				$column ="article.title";			
			}
			$orderby = $column.' '.$params->{'order'};
		}
	}

	#get page number
	my $page = $params->{'page'}; my $all = 0;
	if (not $page) { $page = 1; }
	elsif (not ($page =~ /^\d+$/)) { $page = 1; }
	my $articleCount = CyberArmy::Library->countPublicationArticles();

	my $pages = $articleCount / 30;
	$pages = int $pages + 1 unless ($pages == int $pages);
	my $articles;
	if($page==1){
		$articles = CyberArmy::Library->getPublicationArticles($orderby, 30, 0);
	}else{
		$articles = CyberArmy::Library->getPublicationArticles($orderby, 30,($page-1) * 30);
	}	

	$r->printTemplate('library/publication-only.tmpl',{
		articles => $articles,
		order => $params->{'order'} ? $params->{'order'} : 'desc',
		column => $params->{'column'} ? $params->{'column'} : 'date',
		pages => $pages,
		page => $page
	});
	
	return 0;
}



####
# sub showLogs
#
# Displays library logs page. 
#
# Access restrictions: Requires admin access.
#
# TODO: enable searching on each field. Currently limits to last 200 log entries, make this 
# also a user option
##

sub showLogs {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;

	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );

	return 403 unless $user && CyberArmy::Library->isAdmin($user);
		
	my $logList = CyberArmy::Library->getLogs();


	$r->content_type('text/html');
	$r->printTemplate(
		'library/logs.tmpl', {
			logs => $logList
		}
	);
	
	return 0;

}


####
# sub showUserArticles
#
# Lists all articles submitted by a user
#
# Access restrictions: none
##

sub showUserArticles {
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
	
	my $column = 'article.date_submitted'; #default order
	my $orderby = $column.' DESC';
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 2}} );
	
	
	if($params->{'column'} && $params->{'order'}){ #set user defined order
		if((grep $_ eq $params->{'column'}, ('date', 'category', 'title')) && (grep $_ eq $params->{'order'}, ('asc', 'desc'))){

			if($params->{'column'} eq "date")
			{
				$column ="article.date_submitted";
			}
			elsif($params->{'column'} eq "category")
			{
				$column ="article_category.title";			
			}
			elsif($params->{'column'} eq "title")
			{
				$column ="article.title";			
			}
			$orderby = $column.' '.$params->{'order'};
		}
	}
	
	if($params->{'name'} && $params->{'name'} ne '') {
		my $author = new CyberArmy::User('nickname' => $params->{'name'}) or return 404;;
		
		#get page number
		my $page = $params->{'page'}; my $all = 0;
		if (not $page) { $page = 1; }
		elsif (not ($page =~ /^\d+$/)) { $page = 1; }
		my $articleCount = CyberArmy::Library->countUserArticles($author->caID);

		my $pages = $articleCount / 30;
		$pages = int $pages + 1 unless ($pages == int $pages);
		my $articles;
		if($page==1){
			$articles = CyberArmy::Library->getUserArticles($author->caID, $orderby, 30, 0);
		}else{
			$articles = CyberArmy::Library->getUserArticles($author->caID, $orderby, 30,($page-1) * 30);
		}

		$r->content_type('text/html');
		$r->printTemplate(
			'library/user_articles.tmpl', {
				title => $author->showname."'s :: Articles",
				user => $author,
				articles => $articles,
				order => $params->{'order'} ? $params->{'order'} : 'desc',
				column => $params->{'column'} ? $params->{'column'} : 'date',
				pages => $pages,
				page => $page
			}
		);
	}
	else 
	{
		return 404;
	}
	
	return 0;
}



####
# sub showCategoryRSS
#
# RSS feed for a category. lists latest articles added to the category.
#
# Access restrictions: none
##
sub showCategoryRSS {
	my $r = CyberArmy::WWW::Request->instance();

	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 2}} );
	
	if($params->{'id'}) {
		my $category = CyberArmy::Library->getCategory($params->{'id'});
		if($category!='0E0'){
		
			my $link = $r->getServerLink().'/library/category/'.$params->{'id'};
			my $rss = new XML::RSS (version => '1.0');
			$category->{'title'} = HTML::Entities::decode_entities($category->{'title'});
			$category->{'description'} = HTML::Entities::decode_entities($category->{'description'});
			$rss->channel(
				title		=> $category->{title},
				link		=> $link,
				description	=> $category->{description},
				dc			=> {
					publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
					language	=> 'en-US'
				},
				syn			=> {
					updatePeriod     => 'daily',
					updateFrequency  => 1,
				}
			);

			$rss->image(
				title		=> 'CyberArmy',
				url		=> $r->getServerLink().'/includes/header/cyberarmy.gif',
				link		=> $r->getServerLink().'/',
			);

			foreach ( @{ CyberArmy::Library->GetLatest(10, $params->{'id'}); } ) {
			$_->{'category_title'} = HTML::Entities::decode_entities($_->{'category_title'});
			$_->{'title'} = HTML::Entities::decode_entities($_->{'title'});
				$rss->add_item(
					title	=> $_->{'title'},
					link	=> $r->getServerLink().'/library/article/'.$_->{'id'},
					dc		=> {
						creator	=> $_->{'showname'},
						subject	=> $_->{'title'},
						date	=> $_->{'date_released'}.' GMT'
					}
				);
			}
			$r->content_type('application/xml');
			$r->print( $rss->as_string );
		}
		else{
			return 404;
		}		

	}
	else{
		return 404;
	}
	
	return 0;
}


####
# sub showLibraryRSS
#
# RSS feed for the library. lists latest articles that are set to 'live'.
#
# Access restrictions: none
##
sub showLibraryRSS {
	my $r = CyberArmy::WWW::Request->instance();

	
	my $link = $r->getServerLink().'/library/';
	my $rss = new XML::RSS (version => '1.0');
	$rss->channel(
		title		=> 'CyberArmy Library',
		link		=> $link,
		description	=> 'The CyberArmy library houses articles, tutorials and guides on various subjects organised into categories.',
		dc			=> {
			publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
			language	=> 'en-US'
		},
		syn			=> {
			updatePeriod     => 'daily',
			updateFrequency  => 1,
		}
	);

	$rss->image(
		title		=> 'CyberArmy',
		url			=> $r->getServerLink().'/includes/header/cyberarmy.gif',
		link		=> $r->getServerLink().'/',
	);

	foreach ( @{ CyberArmy::Library->GetLatest(20); } ) {
		$_->{'category_title'} = HTML::Entities::decode_entities($_->{'category_title'});
		$_->{'title'} = HTML::Entities::decode_entities($_->{'title'});
		$rss->add_item(
			title	=> '['.$_->{'category_title'}.'] '.$_->{'title'},
			link	=> $r->getServerLink().'/library/article/'.$_->{'id'},
			dc		=> {
				creator	=> $_->{'showname'},
				subject	=> $_->{'title'},
				date	=> $_->{'date_released'}.' GMT'
			}
		);
	}
	$r->content_type('application/xml');
	$r->print( $rss->as_string );

	return 0;
}



####
# sub showEdition
#
# Obtains the edition to be displayed. All articles in an edition are shown regardless
# of their status - as some are purposely set to non-live as they are publication release only.
# 
# Access restrictions: No restrictions for 'published' editions. Any other status requires 
# library admin or publication admin.
##
sub showEdition {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance;
	
			
			
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1} );
	

	my (undef,$node,$subnode) = split /\//, $r->path_info;
	if(!$params->{'id'} && $subnode =~ /^[0-9]+$/)
	{
		$params->{'id'} = $subnode
	}
	if ($params->{'id'}){
		my $edition = CyberArmy::Library->getEdition($params->{'id'}, $user);
		if($edition!='0E0'){
			my $admin = 'false';
			if($user){
				if(CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$edition->{administrator}")){
					$admin='true';
				}
			}
			if($edition->{'status'} ne 'published'){
				return 404 unless ($admin eq 'true');
				$r->printTemplate('library/edition.tmpl',{
					admin => $admin,
					edition => $edition
				});
			}
			else {
				$r->printTemplate('library/edition.tmpl',{
					edition => $edition,
					admin => $admin
				});
			}
		}	
		else {
			return 404;
		}
	} 
	else{
		return 404;
	}
	
	return 0;
}

####
# sub showEditionRSS
#
# RSS feed for an edition. Lists all articles within an edition (with the status 'live' or 'publication-only').
#
# Access restrictions: edition must be published.
##
sub showEditionRSS {
	my $r = CyberArmy::WWW::Request->instance();

	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
	$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 2}} );
	
	if($params->{'id'}) {
		my $edition = CyberArmy::Library->getEdition($params->{'id'});
		if($edition!='0E0'){
		
			my $link = $r->getServerLink().'/library/edition?id='.$params->{'id'};
			my $rss = new XML::RSS (version => '1.0');
			$rss->channel(
				title		=> $edition->{title},
				link		=> $link,
				description	=> $edition->{description},
				dc			=> {
					publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
					language	=> 'en-US'
				},
				syn			=> {
					updatePeriod     => 'daily',
					updateFrequency  => 1,
				}
			);

			$rss->image(
				title		=> 'CyberArmy',
				url		=> $r->getServerLink().'/includes/header/cyberarmy.gif',
				link		=> $r->getServerLink().'/',
			);
			my $articles = $edition->{'articles'};
			foreach my $article (@$articles){	
				$rss->add_item(
					title	=> $article->{'title'},
					link	=> $r->getServerLink().'/library/article/'.$article->{'id'},
					dc		=> {
						creator	=> $article->{'showname'},
						subject	=> $article->{'title'},
						date	=> $article->{'date_submitted'}.' GMT'
					}
				);
			}
			$r->content_type('application/xml');
			$r->print( $rss->as_string );
		}
		else{
			return 404;
		}		

	}
	else{
		return 404;
	}
	
	return 0;
}

####
# sub showGuide
#
# Displays library guide - a how-to of use.
#
# Access restrictions: none.
##

sub showGuide {
	my $r = CyberArmy::WWW::Request->instance();
	$r->content_type('text/html');
	$r->printTemplate(
		'library/guide.tmpl', {}
	);
	
	return 0;
}


1;
