#!/usr/bin/perl -W
package CyberArmy::WWW::Forum;

use strict;
use XML::RSS ();
use Time::HiRes;
use Digest::MD5 ();
use CyberArmy::User;
use CyberArmy::Forum;
use CyberArmy::Message;
use CyberArmy::BBCode;

$CyberArmy::WWW::Forum::VERSION = '0.6_0';

our $www;
my ($includes,$forum_master);

sub handler {

	$www = CyberArmy::WWW::Request->instance();
	my $r = $www;
	my $path = $r->path_info; 
	CyberArmy::WWW::Utils::escapeHtml($path);
	my (undef,$first,@args) = split /\// , $path;
	my $form = $r->getParams({from => 'args', escapehtml=>1 });

	if ($form->{'action'}) {
		if ($form->{'action'} eq 'View' && $form->{'forum'}) {
			switchUserForumView(
				$form->{'forum'},$form->{'view'},$form->{'page'});
		}
	}

	$forum_master ||= $r->dir_config('forum_master') or exit (500);
	$includes ||= $r->dir_config('dinah_includes');

	if (! defined $first ) { 
		if ($form->{'view'}) { 
			if ($form->{'view'} eq 'rss') { &ForumRssView() }
			else { return 404 }
		} else { &ForumList }
	} elsif ($first eq 'admin.cs') {
		if (exists $form->{action}) {
			if (($form->{action} eq 'View' || $form->{action} eq 'Admin') && $form->{forum}) {
				$www->redirectTo('/forum/admin.cs/'.$form->{forum})
			} elsif ($form->{action} eq 'Remove' && $form->{forum}) {
				$www->redirectTo('/forum/admin.cs/'.
					&ForumAdminRemove($form->{forum},$form->{confirmed}))
			} else { $www->redirectTo('/forum/admin.cs/') }
		} elsif ($r->method eq 'POST') {
				$www->redirectTo('/forum/admin.cs/'.&ForumAdminModify);
		} else { &ForumAdmin($args[0]) }

	} elsif ($first eq 'search.cs') {
			&ForumSearch($args[0])
	} elsif ($first eq 'bans.cs') {

		if ($form->{view} && $form->{forum}) {
			$www->redirectTo('/forum/bans.cs/'.$form->{forum});
		} else { &ForumBans(@args) }

	} elsif ($first eq 'stats.cs') {
		&ForumStats($args[0])
	} else {
		if (!@args ) {
			if ($form->{'view'}) { 
				if ($form->{'view'} eq 'rss') { &ForumRssView($first) }
				else { return 404 }
			} elsif ($form->{'action'} && $form->{'id'}) {
				my %hash = (
					Prev => sub { $www->redirectTo('/forum/'.ForumGetPrevious($first,$form->{'id'})) },
					Next => sub { $www->redirectTo('/forum/'.ForumGetNext($first,$form->{'id'})) },
					PrevThread => sub { $www->redirectTo('/forum/'.ForumGetPreviousThread($first,$form->{'id'})) },
					NextThread => sub { $www->redirectTo('/forum/'.ForumGetNextThread($first,$form->{'id'})) },
					Delete => sub { ForumDeletePost($form->{'id'},$first,$form->{'confirmed'}) },
					Move => sub { ForumMoveThread($form->{'id'}) },
					headline => sub { toggleAsHeadline($form->{'id'}) },
					Sticky => sub { ForumLabelSticky($form->{'id'}, 1) },
					UnSticky => sub { ForumLabelSticky($form->{'id'}, 0) },
					Vote => sub { ForumCastVote($first,$form->{'id'}) }
				);
				$hash{$form->{'action'}} && &{$hash{$form->{'action'}}} or return 404;
			}
			 else { &ForumView($first) }
		}

		elsif ($args[0] eq 'messages') { &ForumViewPost($args[1],$first) }
		elsif ($args[0] eq 'threads') { &ForumViewThread($args[1],$first) }
		elsif ($args[0] eq 'page' && $args[1] && $args[1]=~ /^(\d+)\.html$/) {
				&ForumView($first,$1)
		}

		elsif ($args[0] eq 'post' && $r->method eq 'POST') {
			ForumPost($first,$args[1]);
		} elsif ($args[0] eq 'search.cs') {
			$www->redirectTo('/forum/search.cs/'.$first);

		} elsif ($args[0] eq 'admin.cs') {
			$www->redirectTo('/forum/admin.cs/'.$first);

		} elsif ($args[0] eq 'bans.cs') {
			$www->redirectTo('/forum/bans.cs/'.$first);

		} elsif ($args[0] eq 'logs') { 
			&ForumLogs($first) 
		} elsif ($args[0] eq 'info') { 
			&ForumInfo($first) 
		}
		else { return 404 }
	}
	return 0;
}

sub _info {
	my %info = ();
	my $user = shift;
	my $attributes = $user->getAttributes('title_abrv');
	my $cmsnumber = $user->getMessages()->CheckForUnreadMsg();
	$info{'user'} = 'Logged In As '.join(' ',@{$attributes->{'title_abrv'}}).' '.$user->showname;
	$cmsnumber = $cmsnumber ? " <a href=\"/my/messages/\">".$cmsnumber." messages</a>"
		: " <a href=\"/my/messages/\">No messages</a>";
	my $todo = $user->todoList->count();
	$todo = ($todo > 0) ? "$todo ToDo(s)":'No ToDo';
	$info{'cmsnumber'} = $cmsnumber;
	$info{'todo'} = $todo;
	return \%info;
}

#########################################################
# ForumList() - List the forums the user have access to #
#########################################################
sub ForumList {
	my $start = Time::HiRes::time; ## Start

	my (%forum_flag,%forum_cat);
	my ($user,$now) = (CyberArmy::WWW::Request::User->instance,time);
	my $r = CyberArmy::WWW::Request->instance();

	my $cookies = $r->getParams({from => 'cookies', escapehtml => 1 });
	my $options = $r->getParams({ escapehtml => 1 });

	## View individual messages (0) or nested thread (1)
	my $nested = defined $cookies->{nested} ? $cookies->{nested} : 0;

	my $excluded = exists $options->{'orphaned'}
		&& $options->{'orphaned'} eq 'yes' ? '' : "WHERE type != 'orphaned' ";

	## Prepare a query to retreive all the forums

	my $forums = $CyberArmy::Utils::Database::dbh->prepare(qq~
		SELECT forum_list.forum,name,smalldesc,access_group,administrator,
		       moderator,guest_post,replyonly,allow_html,searchable,type,icon,
		       threads_num,posts_num,last_post_id,locked,hex_color,
		       thread,subject,rdate,UNIX_TIMESTAMP(rdate),author,author_rank,author_color,category_name, category
		FROM forum_list LEFT JOIN forum_replies ON last_post_id = mid
		LEFT JOIN forum_categories ON type = category $excluded ORDER by category_name,name
	~) or exit(500);
	$forums->execute; ## And Exec our prepared statement

	## Then we generate the appropriate forums table depending on the user
	my (@forums,@catlist,$previous);
	while ( my $forum = $forums->fetchrow_hashref() ) {
		if ($forum->{access_group} && $user) { ## OoOoO, A Restricted Forum
			next unless $user->CheckGroupList(
				"$forum->{access_group};".
				"$forum->{administrator};".
				"$forum->{moderator}");

			$forum->{access_group} =~ s/;/ /g;
		} elsif ($forum->{access_group} && !$user) { next }

		## Fillin the blanks
		$forum->{icon} ||= '/includes/header/forum.gif'; #icon
		$forum->{threads_num} ||= 0; $forum->{posts_num} ||= 0;
		$forum->{last_post_id} ||= 0; $forum->{subject} ||= '';

		my %posted = ();
		if ($forum->{last_post_id} && $forum->{subject}) {

			my $diff = '';
			my $ago = ($now - $forum->{'UNIX_TIMESTAMP(rdate)'});
			foreach (CyberArmy::Utils::getTimeInterval($ago)){
				if ($_->[0]){ $_->[1] =~ /^(\w)/; $diff .= "$_->[0]$1 "}}
			$diff .= $diff ? 'ago' : '0s ago'; ## time interval

			my ($r,$g,$b) = (0xff, 0xba, 0x25);
			my $fade = (log($ago > 0 ? $ago : 1) / log(604800)) - 0.7; # range: 1w, no change before 12h
			$fade = 0 if ($fade < 0); $fade = 1 if ($fade > 1);
			## $fade = int($fade * 10)/10;
			$r -= $fade * 0xd9;
			$g -= $fade * 0x7e;
			$b += $fade * 0x48;
			my $col = sprintf("#%2x%2x%2x", $r, $g, $b);

			$posted{'nested'} = $nested;
			$posted{'thread'} = $forum->{thread};
			$posted{'last_post_id'} = $forum->{last_post_id};
			$posted{'subject'} = CyberArmy::Utils::trimHtmlEncoded($forum->{subject}, 30, 1);
			$posted{'rdate'} = $forum->{'rdate'};
			$posted{'col'} = $col;
			$posted{'diff'} = $diff;
			$posted{'author'} = $forum->{author};
			$posted{'author_rank'} = $forum->{author_rank};
			$posted{'author_color'} = $forum->{author_color};

		}
		if ($previous && $previous ne $forum->{type}) {
			push @catlist,[@forums];
			undef @forums;
		}

		push @forums, ({
				category_name => $forum->{category_name},
				category => $forum->{category},
				type => $forum->{type},
				name => $forum->{forum},
				icon => $forum->{icon},
				hex_color => $forum->{hex_color},
				fullname => $forum->{name},
				smalldesc => $forum->{smalldesc},
				threads_num => $forum->{threads_num},
				posts_num => $forum->{posts_num},
				access_group => $forum->{access_group},
				guest_post => $forum->{guest_post},
				replyonly => $forum->{replyonly},
				allow_html => $forum->{allow_html},
				searchable => $forum->{searchable},
				locked => $forum->{locked},
				p => ({ %posted })
		});
		$previous = $forum->{type};
	}
	push @catlist,[@forums] if (@forums);
	$forums->finish; ## Fetching done, ty...
	$r->printTemplate('forum/forum.tmpl',{
		categories => \@catlist,
		info => ($user) ? _info($user):undef,
		title => 'CyberArmy Forums',
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});

}

############################################
# ForumAdmin() - Top level Admin Function  #
############################################
sub ForumAdmin {
	my $start = Time::HiRes::time; ## Start
	my $adminforum = shift;
	my %template;
	my $r = CyberArmy::WWW::Request->instance();

	## We stop here if no valid user
	my $user = CyberArmy::WWW::Request::User->instance or exit(403);
	## Prepare a query to retreive all the forums

	my $forums = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,name,administrator,secured	FROM forum_list ORDER BY name
	~) or exit(500);
	$forums->execute; ## Exec :)

	## The goal down here is to show all the forums a given user have access
	## to in a drop down list, and have the chosen one selected
	$template{'adminforum'} = $adminforum;
	## Identify global forum masters early
	my $forum_master = $user->IsInGroup($forum_master);

	## Fetch the forums
	my @forums;
	my $ok; while ( my $forum = $forums->fetchrow_hashref ) {
		if ($user->CheckGroupList($forum->{administrator}) ||
				($forum_master && $forum->{'secured'} eq 'N')) {
			push @forums,({
					forum => $forum->{forum},
					name => $forum->{name}
					 });
			$ok++;
		}
	} ## the $ok flag is to see if we have at least a match
	$template{'forums'} = \@forums;

	## We stop here if we don't have any matches..
	if (!$ok && !$forum_master) { exit(403) }

	## 1- We initialize some values

	my ($f_forum,$f_name,$f_smalldesc,$f_administrator,$f_type,$f_logo,$f_icon,
	$f_motd,$f_access_group,$f_moderator,$f_guest_post,$f_member_replyonly,$f_allow_html,
	$f_searchable,$f_locked,$forum,$f_hex_color) = ('','','','','','','','','','','','','','','','');

	## 2- Set some default values
	my $categos = CyberArmy::Forum->GetCategories();

	$f_type = ''; foreach (@$categos) {
		my $selected = $_->{category} eq 'normal' ? 'selected' : '';
		$f_type .= '<option '.$selected.'>'.$_->{category}.'</option>'
	}

	## 3- Now if we are retreiving infos about an existing forum, we set the corresponding
	## 	Values...
	if ($adminforum) {
		## Db query

		$forum = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT forum,name,smalldesc,administrator,type,logo,icon,locked,
				motd,access_group,moderator,guest_post,replyonly,allow_html,searchable,
				hex_color,secured
			FROM forum_list
			WHERE forum = ?
		~,undef,$adminforum);
		$template{'forum'} = $forum;
		## Security Check
		unless (($forum_master && $forum->{'secured'} eq 'N')
					or $user->CheckGroupList($forum->{administrator})) { exit(403) }
		$f_forum = $forum->{forum};
		$f_name = $forum->{name};
		$f_smalldesc = $forum->{smalldesc};
		$f_administrator = $forum->{administrator}? $forum->{administrator} : '';

		my $cats = CyberArmy::Forum->GetCategories();
		$f_type = ''; foreach (@$cats) {

			my $selected = $_->{category} eq $forum->{type} ? 'selected' : '';
			$f_type .= '<option '.$selected.'>'.$_->{category}.'</option>'
		}

		$f_logo = $forum->{logo} ? qq~value="$forum->{logo}"~ : '';
		$f_icon = $forum->{icon} ? qq~value="$forum->{icon}"~ : '';
		$f_motd = $forum->{motd} ? $forum->{motd} : '';
		$f_access_group = $forum->{access_group} ? qq~value="$forum->{access_group}"~ : '';
		$f_moderator = $forum->{moderator} ? qq~value="$forum->{moderator}"~ : '' ;

		$f_guest_post = ''; foreach (['R','ReadOnly'],['X','ReplyOnly'],['S','StartThreads']) {
			my $selected = $forum->{guest_post} && ($_->[0]  eq $forum->{guest_post}) ? 'selected' : '';
			$f_guest_post .= '<option '.$selected.' value ="'.$_->[0].'">'.$_->[1].'</option>'."\n"
		}
		$f_member_replyonly='N';
		if($forum->{member_replyonly} && $forum->{member_replyonly} eq 'selected')
		{
			$f_member_replyonly = 'Y';
		}
		

		$f_hex_color = $forum->{hex_color} ? qq~value="$forum->{hex_color}"~ : '';

	}

	$r->printTemplate('forum/admin.tmpl',{
		%template,
		info => _info($user),
		f_forum => $f_forum,
		f_name => $f_name,
		f_smalldesc => $f_smalldesc,
		f_icon => $f_icon,
		f_logo => $f_logo,
		f_hex_color => $f_hex_color,
		f_type => $f_type,
		f_guest_post => $f_guest_post,
		f_allow_html => $f_allow_html,
		f_searchable => $f_searchable,
		f_locked => $f_locked,
		f_administrator => $f_administrator,
		f_moderator => $f_moderator,
		f_access_group => $f_access_group,
		f_motd => $f_motd,
		f_member_replyonly => $f_member_replyonly,
		title => 'CyberArmy Forums Administration',
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});
}

###########################################
# ForumAdminModify() - Modify/Add A Forum #
###########################################

sub ForumAdminModify {
	my $posted = $www->getParams({ from => 'posted', escapehtml => 1 });
	my $user = CyberArmy::WWW::Request::User->instance or exit(403);
	exit(401) unless $posted->{forum}; ## Sanity Check

	my $forum = new CyberArmy::Forum(
		id => $posted->{forum}, select => 'forum,administrator,secured'
	);

	$posted->{administrator} ||= '~'.$user->nickname;
	$posted->{moderator} ||= '[NOMOD]';

	foreach (';',',') {
		local $/ = $_; chomp (
			$posted->{administrator},
			$posted->{moderator},
			$posted->{access_group}
		); ## remove possible trailing ; or ,
	}

	$posted->{allow_html} = $posted->{allow_html} ? 'Y' : 'N';
	$posted->{searchable} = $posted->{searchable} ? 'Y' : 'N';
	$posted->{locked} = $posted->{locked} ? 'Y' : 'N';
	$posted->{member_replyonly} = $posted->{member_replyonly} ? 'Y' : 'N';

	$posted->{hex_color} = ($posted->{hex_color} && $posted->{hex_color} =~ /^\#[0-9a-fA-F]{6}$/) ? $posted->{hex_color} : '';

	## Update/Add The forum
	if ($forum->{forum} && ($forum->{forum} eq $posted->{forum})) { ## We are updating an existing forum
		$www->checkReferer($www->location . '/admin.cs/'.$forum->{forum}) or exit(403); ## refere chk

		exit(403) unless (
			$user->CheckGroupList($forum->{administrator}) ||
			($user->IsInGroup($forum_master)
				&& $forum->{'secured'} eq 'N' )
		);

		$CyberArmy::Utils::Database::dbh->do(q~
			UPDATE forum_list
			SET name=?,smalldesc=?,administrator=?,type=?,logo=?,
			    icon=?,motd=?,access_group=?,moderator=?,guest_post=?,
			    allow_html=?,searchable=?,locked=?,hex_color=?,replyonly=?,last_modified=NOW()
			WHERE forum = ?
		~,undef,$posted->{name},$posted->{smalldesc},$posted->{administrator},
		$posted->{type},$posted->{logo},$posted->{icon},$posted->{motd},$posted->{access_group},
		$posted->{moderator},$posted->{guest_post},$posted->{allow_html},
		$posted->{searchable},$posted->{locked},$posted->{hex_color},
		$posted->{member_replyonly}, $posted->{forum}
		);
	} else { ## We are adding a new forum
		$www->checkReferer($www->location.'/admin.cs') or exit(403);
		exit(403) unless $user->IsInGroup($forum_master);

		$CyberArmy::Utils::Database::dbh->do(q~
			INSERT INTO forum_list (forum,name,smalldesc,administrator,
				type,logo,icon,motd,access_group,moderator,guest_post,
				allow_html,searchable,locked,hex_color,replyonly,last_modified)
			VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,NOW())
		~,undef,$posted->{forum},$posted->{name},$posted->{smalldesc},$posted->{administrator},
		$posted->{type},$posted->{logo},$posted->{icon},$posted->{motd},$posted->{access_group},
		$posted->{moderator},$posted->{guest_post},$posted->{allow_html},$posted->{searchable},
		$posted->{locked},$posted->{hex_color},$posted->{member_replyonly}
		);
	}

	#All Successfull, redirect to the newly created forum admin panel

	return $posted->{forum};
}

#######################################
# ForumAdminRemove() - Remove A Forum #
#######################################
sub ForumAdminRemove {
	my ($rmforum,$confirmed,$user) = (shift,shift,CyberArmy::WWW::Request::User->instance);

	my $forum = new CyberArmy::Forum (
		id => $rmforum,
		select => 'administrator,forum,secured'
	) or exit(404);

	$www->checkReferer($www->location . '/admin.cs/'.$forum->{forum})
		or ForumError(403,'Security error: Invalid HTTP referer');

	## Then Perform The Security Check
	exit(403) unless ($user && $user->IsInGroup($forum_master));

	unless ($confirmed eq 'on') {
		ForumError(403,qq~Please tick the checkbox, just so we know you really intend to remove the '$forum->{forum}' forum.~)}

	## Start Rm'ing.. 1st we delete the entry from the forum_list

	$CyberArmy::Utils::Database::dbh->do(
	'DELETE FROM forum_list WHERE forum = ?',undef,$rmforum);
	## Then from forum_replies
	$CyberArmy::Utils::Database::dbh->do(
	'DELETE FROM forum_replies WHERE forum = ?',undef,$rmforum);

	## All done, return
	return undef
}

#############################
# ForumBans() - Manage Bans #
#############################

sub ForumBans {
	my $start = Time::HiRes::time; ## Start
	my $user = CyberArmy::WWW::Request::User->instance or exit(403);
	my $bforum = shift;

	my $sforum = new CyberArmy::Forum (
		id => $bforum,
		select => 'forum,name,administrator,moderator'
	);

	if ($sforum) {
		$user->CheckGroupList("$sforum->{moderator};$sforum->{administrator}")
			or exit(403);
	} else {
		exit(403) unless $user->IsInGroup($forum_master);
		$sforum->{forum} = '[GLOBAL]';
	}

	if ($www->method eq 'POST') {
		my $posted = $www->getParams({from => "posted",escapehtml => 1});

		if ($posted->{ban}) {
			exit(412) unless $posted->{banmask};

			## verify we have no dodgy chars that could screw up the regex
			ForumError(403,'Banmask contains invalid chars')
				unless ($posted->{banmask} =~ /^(\w|\.|\-|\?|\*)+$/);

			$CyberArmy::Utils::Database::dbh->do(
				'INSERT INTO forum_bans (forum,type,ban,ban_by, '.
				'ban_date,ban_comment) VALUES (?,?,?,?,NOW(),?)'
				,undef,$sforum->{forum},$posted->{type},$posted->{banmask}
				,$user->showname,$posted->{comment}
			);
		} elsif ($posted->{unban} && $posted->{action}) {
			if ($posted->{action} eq 'unban' && $posted->{bans}) {

				$CyberArmy::Utils::Database::dbh->do(
					'DELETE FROM forum_bans WHERE ban_id=?'
					,undef,$posted->{bans}
				);
			} elsif ($posted->{action} eq 'flush') {
				$CyberArmy::Utils::Database::dbh->do(
					'DELETE FROM forum_bans WHERE forum=?'
					,undef,$sforum->{forum}
				);
			}
		}

		$www->redirectTo('/forum/bans.cs/'.$sforum->{forum}.'/');
	}

	## retreive all the forums

	my $forums = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,name,administrator,moderator FROM forum_list
	~);
	$forums->execute;

	my $forum;
	my @forlist;
	while ($forum = $forums->fetchrow_hashref ) {
		my $selected = $sforum &&
			$forum->{forum} eq $sforum->{forum} ? 'selected' : '';

		if ($user->CheckGroupList("$forum->{administrator};$forum->{moderator}")) {
			push @forlist, {
				selected => $selected,
				forum => $forum
			};
		}
	}

	my $bans = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,type,ban,ban_by,ban_date,ban_comment,ban_id
		FROM forum_bans WHERE forum = ? ORDER BY ban_date DESC
	~);
	$bans->execute($sforum->{forum});

	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate('forum/bans.tmpl',{
		forums => \@forlist,
		bans => $bans->fetchall_arrayref({}),
		info => _info($user),
		forum_master => ($user->IsInGroup($forum_master)) ? 1:0,
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});
}

###################################
# ForumView() - View The Forum... #
###################################

sub ForumView {
	my $start = Time::HiRes::time; ## Start
	my $viewforum = shift;
	my $page = shift || 1;
	my (%template, @parray, @sarray, $cached_threads);
	my $r = CyberArmy::WWW::Request->instance();
	my $cookies = $r->getParams({from => 'cookies', escapehtml => 1 });

	## Put this in user profile someday, together with signature, image link etc. ?
	## View expanded (WWWBoard = 1) or collapsed (UBB = 0) threads
	my $threaded = defined $cookies->{threaded} ? $cookies->{threaded} : 1;

	## Sort by thread (0) or by latest post (1)
	my $sorted = defined $cookies->{sorted} ? $cookies->{sorted} : 0;

	## View individual messages (0) or nested thread (1)
	my $nested = defined $cookies->{nested} ? $cookies->{nested} : 0;

	## Fetch forum infos...

	my $forum = new CyberArmy::Forum ( id => $viewforum,
		select => 'forum,name,access_group,motd,searchable,threads_num'.
		',administrator,logo,guest_post'.
		',replyonly,moderator,UNIX_TIMESTAMP(last_modified)'
	) or exit(404);

	my $user = CyberArmy::WWW::Request::User->instance; ## get the user


	## Then we perform a security check
	if ($forum->{access_group} &&
	(!$user || !$user->CheckGroupList(
			"$forum->{access_group};".
			"$forum->{moderator};".
			"$forum->{administrator}"
			)
	)) { exit(403) }

	my $pages = $forum->getNumberOfPages;
	exit(404) if ($page != 1 && $page > $pages);

	$template{searchable} = $forum->{searchable};

	$template{administrator} = ($user && $user->CheckGroupList($forum->{administrator})) ? '1':undef;

	$forum->{logo} ||= $includes.'/images/elecman.gif';

	my $nav_pages;

	$template{'page'} = $page;
	$template{'pages'} = $pages ||=1;

	my($cache,$cache_name);

	if ($page == 1) {
		$cache_name = $forum->{forum};
		$cache_name .= $threaded ? '_1' : '_0';
		$cache_name .= $sorted ? '_1' : '_0';
		$cache_name .= $nested ? '_1' : '_0';

		$cache = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT name,value,UNIX_TIMESTAMP(cdate)
			FROM cache
			WHERE name = ?
		~,undef,$cache_name);
	}

	## Caching Hack
	if ($page > 1 || !defined($cache) || ($cache->{'UNIX_TIMESTAMP(cdate)'} < $forum->{'UNIX_TIMESTAMP(last_modified)'})) {

	if ($threaded) {
		my $threads = $forum->GetThreadsList(50,$page,$sorted);
		my $stickies = $forum->GetStickyList(10) if ($page == 1); ## chawmp: limited to top 10 stickies per forum
		my $mid;
		my $list; foreach( @{$threads}, @{$stickies} ) { $list .= $_->[1].','  }
		local $/ = ','; chomp $list if $list;

		my $posts = $forum->GetPostsFromThreads($list) if $list;

		my (%replies,%replies_index); my $counter = 0;
		foreach (@{$posts}) { ## Build replies list
			## Bit weird? it just has to be an array to
			## be sure order will be respected.
			push @{$replies{$_->{pid}}},$_->{mid} if $_->{pid};
			$replies_index{$_->{mid}} = $counter; $counter++;
		}
		if ($page == 1) {
			foreach my $thread (@{$stickies}) {
				## First print the main thread
				$mid = $thread->[0];
				push @sarray, FormatStickyLine({
					%{$posts->[$replies_index{$mid}]},
					forum => $forum->{forum},
					thread => $thread->[1],
					nested => $nested
				});
			}
		}
		foreach my $thread (@{$threads}) { ## Now foreach thread
			my $mid = $thread->[0];

			## First print the main thread
			my $depth = 1;
			push @parray,{
				depth => $depth,
				text => FormatReplyLine({
						%{$posts->[$replies_index{$mid}]},
						forum => $forum->{forum},
						thread => $thread->[1],
						nested => $nested
					})
			};

			if ($replies{$mid}) { ## g0t replies?
				my $target = $mid; # initialize some values and flags
				$depth++;
				while ($target) { ## As long as we have a running target
				## if the target has childs
				if (@{$replies{$target}}) {
					foreach (@{$replies{$target}}) { ## process them
							if (!$posts->[$replies_index{$_}]->{printed}) {
								push @parray, {
									depth => $depth,
									text => FormatReplyLine({
										%{$posts->[$replies_index{$_}]},
										forum => $forum->{forum},
										thread => $thread->[1],
										nested => $nested
									})
								};
								$posts->[$replies_index{$_}]->{printed}++;
								## and mark we did so...
							}

							## g0t replies?
							if ( exists $replies{$_} && ${$replies{$_}}[0]) {
								$posts->[$replies_index{$_}]->{parent} = $target;
								$depth++;
								$target = $_;
								last;
							} else {
								shift @{$replies{$target}};
								last;
							}
					}
				} else {
					if (exists $posts->[$replies_index{$target}]->{parent}) {
						$target = $posts->[$replies_index{$target}]->{parent};
						$depth--;
					} else {
						undef $target;
						last;
					}
				}
			}
			} ## End replies
		}
	} else {
		my $threads = $forum->GetThreadsList2(50,$page,$sorted);

		my $list; foreach( @{$threads} ) { $list .= $_->{min_mid}.','  }
		local $/ = ','; chomp $list if $list;

		my $posts = $forum->GetPostsFromMids($list) if $list;

		my (%posts_index); my $counter = 0;
		foreach (@{$posts}) { ## Build post list
			$posts_index{$_->{mid}} = $counter; $counter++;
		}

		my $count = 0;
		foreach my $thread (@{$threads}) { ## Now foreach thread
			my $mid = $thread->{min_mid};
			next unless exists $posts_index{$mid};

			$posts->[$posts_index{$mid}]->{subject} =
				CyberArmy::Utils::trimHtmlEncoded($posts->[$posts_index{$mid}]->{subject}, 40, 1);

			## Print the main thread
			push @parray, FormatThreadLine ({
				%{$posts->[$posts_index{$mid}]},
				forum => $forum->{forum},
				date => $thread->{max_rdate},
				replies => $thread->{count_mid},
				lastid => $thread->{max_mid},
				thread => $thread->{thread},
				nested => $nested,
				alt => $count % 2
			});
			$count++;
		}
	} ## End not threaded

	CyberArmy::Template->instance->process(
		'forum/forumthreads.tmpl', {
			threaded => $threaded,
			parray => \@parray,
			sarray => \@sarray
		}, \$cached_threads
	);

	if ($page == 1) {
		if (!defined($cache)) {
			$CyberArmy::Utils::Database::dbh->do(qq~
				INSERT INTO cache (name,value,cdate)
				VALUES (?, ?, NOW())
			~,undef,$cache_name,$cached_threads);
		} else {
			$CyberArmy::Utils::Database::dbh->do(qq~
				UPDATE cache
				SET cdate=NOW(),value=?
				WHERE name = ?
			~,undef,$cached_threads,$cache_name);
		}
	}
	} ## END

	my $disabled = ''; ## Some chks on guests
	if (!$user && defined $forum->{guest_post}) {
		$cookies->{forumguest} ||= '';
		$disabled = 'disabled' unless $forum->{guest_post} eq 'S';
		$template{'guest'} = 'true';
		$template{'forumguest'} = $cookies->{forumguest};
		$template{'gpdisabled'} = $disabled;
	}
	
	my $IsPrivileged = $user ?
		$user->CheckGroupList("$forum->{moderator};$forum->{administrator}"
	) : 0; ## Set if the poster is privileged
	if(!$IsPrivileged && $disabled eq '')
	{	
		$disabled = 'disabled' unless $forum->{replyonly} eq 'N';
	}

	my $bold_chk = $user &&
		$user->CheckGroupList("$forum->{moderator};$forum->{administrator}") ?
	'<input type="checkbox" name="bold_chk" value="ON" /> Bold?' : '';

	my $sticky_chk = $user &&
		$user->CheckGroupList("$forum->{moderator};$forum->{administrator}") ?
	'<input type="checkbox" name="sticky_chk" value="ON" /> Sticky?' : '';


	$template{'info'} = _info($user) if ($user);

	$r->printTemplate('forum/view.tmpl',{
		%template,
		motd => $forum->{motd} ? CyberArmy::BBCode::Parse($forum->{motd}):'[None]',
		forum => $forum,
		threaded => $threaded,
		sorted => $sorted,
		nested => $nested,
		viewforum => $viewforum,
		logo => $forum->{logo},
		threaded => $threaded,
		bold_chk => $bold_chk,
		cache => ($cached_threads)? $cached_threads : $cache->{value},
		sticky_chk => $sticky_chk,
		image_link => $cookies->{image_link} ||= '',
		disabled => $disabled,
		signature => ($user && $user->signature) ? "\n\n" . $user->signature:'',
		title => $forum->{name},
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});

}

###########################################
# ForumViewPost() - View individual posts #
###########################################

sub ForumViewPost {
	my $start = Time::HiRes::time; ## Start
	my ($viewpost,$viewforum,$user) =
		(shift =~ m/(\d+)\.html/ ? $1 : '',shift,CyberArmy::WWW::Request::User->instance);

	my %template;

	my $r = CyberArmy::WWW::Request->instance();
	## Get the message object
	my $message = new CyberArmy::Forum::Message (
		select =>'mid,pid,forum,thread,author,author_rank,'.
		'author_ip,author_badge,author_host,subject,img_url,'.
		'body,signature,rdate,bold,frontpage,sticky',
	id => $viewpost	) or exit(404);

	## Sanity Check, check whether the url is correct
	if (!$message or ($message->{forum} ne $viewforum)) { exit(404) }

	## Get the forum object
	my $forum = $message->GetForum('forum,name,access_group,'.
		'guest_post,replyonly,allow_html,moderator,administrator,last_post_id');

	## Here we perform a security check
	if ($forum->{access_group} && ( !$user || !$user->CheckGroupList("$forum->{access_group};$forum->{moderator};$forum->{administrator}")))
		{ exit(403) }

	my ($selected_post_mid,$starter_post_mid);
	my (%replies,%posts_index);
	my $counter = 0;

	## We get all the messages in the selected message's thread
	my $posts = $message->GetThread('mid,pid,thread,subject,author,'.
		'author_color,author_ip,author_host,author_rank,rdate,bold,author_badge');

	## And we go thru them...
	foreach (@{$posts}) {
		## If it has a parent ID, we add it to the childs
		## list of that parent message
		push @{$replies{$_->{pid}}}, $_->{mid} if ($_->{pid});

		## If the message id, matche the message id of the
		## originally requested message, we save it
		$selected_post_mid = $_->{mid} if ($_->{mid} == $message->{pid});

		## If the parent id is 0, means that this is the
		## starter message of the thread, we save it
		$starter_post_mid = $_->{mid} if ($_->{pid} == 0);

		## And finnaly we save the position of which the
		## message was listed (because hashes don't preserve
		## order)
		$posts_index{$_->{mid}} = $counter; $counter++;
	}

	$template{forum} = $forum;
	$template{repl} = $replies{$starter_post_mid};
	$template{viewpost} = $viewpost;
	$template{last_post_id} = $forum->{last_post_id};

	my $access;
	if ($user && $user->CheckGroupList("$forum->{administrator};$forum->{moderator}")) {
		$access++;
		$template{access} = $access;
		$template{sticky} = $message->{sticky};
		$template{votes} = CyberArmy::Forum->CountDisapprovalVotes($viewpost);
		$template{voteLimit} = $CyberArmy::Forum::VoteLimit;

		if ($message->{pid} == 0) {
			if ($access && $user->IsInGroup('staff')) {
				$template{staff} = 1;
				$template{frontpage} = $message->{frontpage};
			}
		}
	}
	if($user) {
		$template{hasVoted} = $user ? CyberArmy::Forum->HasVoted($viewpost, $user) : 0;
	}

	## boolean guest test
	my $is_a_guest = $message->{author_rank} eq 'Guest' ? 1 : 0;
	$template{is_a_guest} = $is_a_guest;

	#only display any IP info if staff/mod/admin
	if (!$access) {
		$message->{author_ip} = undef;
		$message->{author_host}= undef;
	 }

	$template{form_nick} = FormatShowName($message->{author_rank},$message->{author});
	if (! $is_a_guest) {

		my $nick = lc($message->{author}); ## lcase it
		$nick =~ s/ //g; ## strip spaces
		$template{nick} = $nick;
	}

	$template{author_ip} = $message->{author_ip};
	$template{author_host} = $message->{author_host};
	$template{rdate} = $message->{rdate};
	$template{selected_post_mid} = $selected_post_mid;

	if ($selected_post_mid) {
		my $is_a_guest2 = $posts->[$posts_index{$selected_post_mid}]
		->{author_rank} eq 'Guest' ? 1 : 0; ## boolean guest test

		#only display any IP info if staff/mod/admin
		if (!$access) {
			$posts->[$posts_index{$selected_post_mid}]->{author_ip} = undef;
			$posts->[$posts_index{$selected_post_mid}]->{author_host} = undef;
		 }
		$template{spm} = $posts->[$posts_index{$selected_post_mid}];
		$template{spm}->{fsm} = FormatShowName($posts->[$posts_index{$selected_post_mid}]
					->{author_rank},$posts->[$posts_index{$selected_post_mid}]->{author});
	}
	$template{includes} = $includes;

	## So some substitutions in the message body..
	my $message_body = $message->{body}; ## copy
	$message_body .= "\n".$message->{signature} if ($message->{signature});

	## render quoted text ('>') in italic
	$message_body =~ s/^(&gt;.*)$/<i>$1<\/i>/mgi;

	## Enable BBCode in posts
	if ($forum->{allow_html} eq 'Y') {
		CyberArmy::BBCode::Parse($message_body);
	} else {
		$message_body =~ s/\n/<br\/>\n/gs;
	}
	my @replies;
	if ($replies{$starter_post_mid}) { ## g0t replies In thread?
		my $depth = 1;
		push @replies, {
			depth => 0,
			text => FormatReplyLine ({
				%{$posts->[$posts_index{$starter_post_mid}]},
				forum => $forum->{forum},
				star => ($starter_post_mid == $message->{mid}) ? 1 : 0
			})};

		my $target = $starter_post_mid;
		while ($target) {
			if (@{$replies{$target}}) {
				foreach (@{$replies{$target}}) {
					if (!$posts->[$posts_index{$_}]->{printed}) {
						push @replies, {
							depth => $depth,
							text =>	FormatReplyLine ({
								%{$posts->[$posts_index{$_}]},
								forum => $forum->{forum},
								star => $_ == $message->{mid} ? 1 : 0,
						})}; $posts->[$posts_index{$_}]->{printed}++;
					}

					if ( exists $replies{$_} && @{$replies{$_}}[0]) { ## g0t replies?
						$posts->[$posts_index{$_}]->{parent} = $target;
						$depth++;
						$target = $_; last;
					} else { shift @{$replies{$target}}; last; }
				}
			} else {
				if ($posts->[$posts_index{$target}]->{parent}) {
					$target = $posts->[$posts_index{$target}]->{parent};
					$depth--;
				} else {undef $target; last; }
			}
		}
	}
	$message->{body} =~ s/\n/\n&gt;/g; # format

	my $cookies = $www->getParams({from => 'cookies' , escapehtml => 1 });
	my $disabled = ''; unless ($user) {
		$cookies->{forumguest} ||= '';
		$disabled = $forum->{guest_post} ne 'R' ? '' : 'disabled';
	}

	$cookies->{image_link} ||= '';

	my $reply = $message->{subject};
	if ($reply !~ /^RE: /) {
		$reply = 'RE: ' . $reply;
	}

	my $sticky_chk = $user &&
		$user->CheckGroupList("$forum->{moderator};$forum->{administrator}") ?
		'<input type="checkbox" name="sticky_chk" value="ON" /> Sticky?' : '';

	my %postform = (
		reply => $reply,
		sticky_chk => $sticky_chk,
		signature => ($user && $user->signature) ? "\n\n".$user->signature:'',
		image_link => $cookies->{image_link},
		disabled => $disabled
	);
	if ($user ) {$template{'user'} = 1;} else {$postform{forumguest} = $cookies->{forumguest};}
	$template{'info'} = _info($user) if ($user);
	$r->printTemplate('forum/viewpost.tmpl',{
		%template,
		message_body => $message_body,
		replies => \@replies,
		postform => \%postform,
		message => $message,
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});
}

###############################################
# ForumViewThread() - View individual threads #
###############################################
sub ForumViewThread {
	my $start = Time::HiRes::time; ## Start
	my %template;
	my $r = CyberArmy::WWW::Request->instance();
	my ($viewthread,$viewforum,$user) =
		(shift =~ m/(\d+)\.html/ ? $1 : undef,shift,CyberArmy::WWW::Request::User->instance);

	exit (404) unless defined ($viewthread);
	## Get the forum object
	my $forum = new CyberArmy::Forum ( id => $viewforum,
		select => 'forum,name,access_group,moderator,administrator,'.
		'guest_post,replyonly,allow_html'
	) or exit(404);

	## Here we perform a security check
	if ($forum->{access_group} && ## If forum restricted
		(	!$user || ## And there is no valid user
			!$user->CheckGroupList(      ## Or the user
			  "$forum->{access_group};". ## doesn't have
			  "$forum->{moderator};".    ## the required
			  "$forum->{administrator}"  ## clearance
			)
		)
	) { exit(403) }

	my $access;

	if ($user && $user->CheckGroupList(  ## See if the user has...
		"$forum->{administrator};$forum->{moderator}")
	) {
		$access++;
		$template{access} = $access;
	}

	## Note : we don't retrieve badges or image URL here...

	## Prepare a query to retreive all the posts of a thread
	my $thread = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT mid,pid,forum,thread,subject,author,
		       author_color,author_ip,author_host,author_rank,
		       body,rdate,bold
		FROM forum_replies
		WHERE thread = ? AND forum = ?
		ORDER BY trail ASC
	~) or exit(500);
	$thread->execute($viewthread,$viewforum); ## And Exec our prepared statement

	my $counter = 0;
	my %level;
	$level{0} = 0;
	my $oldlevel = 0;
	my $subject;
	my $post;

	my $voteLimit = $CyberArmy::Forum::VoteLimit;
	my @thread;
	while ( my $message = $thread->fetchrow_hashref() ) {
		$counter++;
		#only display any IP info if staff/mod/admin
		if (!$access) {
			$message->{author_ip} = undef;
			$message->{author_host} = undef;
		}

    	$message->{votes}= CyberArmy::Forum->CountDisapprovalVotes($message->{mid});

		$level{$message->{mid}} = $level{$message->{pid}} + 1;
		$subject ||= $message->{subject};
		$post ||= $message->{mid};
		my $nick;
		if ($message->{author_rank} ne 'Guest') {

			$nick = lc($message->{author}); ## lcase it
			$nick =~ s/ //g; ## strip spaces
		}

		# Note : we don't show badges or image url here...

		my $message_body = '';
		unless (!($message->{subject} =~ /^RE:/i)
			&& $message->{subject}  =~ /(?:\W)+(n(?:\W)??t)(?:\W+$|$)/i) {
			## Assuming there is text in the body (guess based on the subject)
			## So some substitutions in the message body..
			$message_body = $message->{body}; ## copy

			## render quoted text ('>') in italic
			$message_body =~ s/(&gt;.*(?:\n|$))/<i>$1<\/i>/gi;

			## Enable HTML in posts
			if ($forum->{allow_html} eq 'Y') {
				CyberArmy::BBCode::Parse($message_body);
			} else {
				$message_body =~ s/\n/<br\/>\n/gs;
			}
		}
		push @thread, {
			depth => $level{$message->{mid}},
			message => $message,
			message_body => $message_body,
			nick => $nick,
			name => FormatShowName($message->{author_rank},$message->{author})
		};
	}

	## ugly way to do it, but i can't think a better way
	## to see if the thread exists or no, for now.
	## new forum version on the way, anyhow.
	exit (404) if ($counter == 0);

	$template{'info'} = _info($user) if ($user);
	$r->printTemplate('forum/threads.tmpl',{
		%template,
		thread => \@thread,
		viewthread => $viewthread,
		subject => $subject,
		post => $post,
		forum => $forum,
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) },
		voteLimit => $voteLimit
	});
}

###########################################################
# ForumPost() - Add/Reply/Preview a post to a given forum #
###########################################################
sub ForumPost {
	my $start = Time::HiRes::time; ## Start
	my ($forumpost,$replypost,$user) = (shift,shift,CyberArmy::WWW::Request::User->instance);
	exit(412) unless $forumpost; ## Sanity Check

	$www->checkReferer($www->location. '/'.$forumpost) ## referer check
	or ForumError(403,'Your browser is not HTTP/1.0 compliant!');

	my $posted = $www->getParams({from => 'posted', escapehtml => 1 }); ## POSTed

	## Prevent empty/spaced-out subject lines
	$posted->{subject} =~ s/(^\s+|\s+$)//g if $posted->{subject};

	$posted->{message} =~ tr/\r//d if $posted->{message};

	## Clear message body of [nt] posts
	# extra checking added to prevent wiping out of messages with subjects like "Windows NT"
	if ($replypost &&	# must be a reply
		not ($posted->{subject} =~ /^RE:/i) &&	# subject does not start with "RE:"
		($posted->{subject} =~ /(?:\W)+(n(?:\W)??t)(?:\W+$|$)/i) &&	# subject ends with "nt" permutation
		($posted->{message} &&						# body empty, or...
		($posted->{message} =~ /^On [^\n]+ wrote(\n&gt;[^\n]*)+\n*$/s))	# body looks like it is *unmodified*
	) {
		$posted->{message} = '[nt]'; $posted->{signature} = undef;
	}

	## Input Sanity Check
	exit(412) unless ($posted->{subject} && $posted->{message} &&
		(($posted->{action} eq 'Post') || ($posted->{action} eq 'Preview')));

	## Get the forum object
	my $forum = new CyberArmy::Forum (
		id => $forumpost, select => 'forum,access_group,guest_post,'.
		'locked,replyonly,administrator,moderator,allow_html,name' ) or exit(404);

	my $IsPrivileged = $user ?
		$user->CheckGroupList("$forum->{moderator};$forum->{administrator}"
	) : 0; ## Set if the poster is privileged

	## If the forum is locked, only moderators and administrators can post
	if ( $forum->{locked} eq 'Y' && !$IsPrivileged) { exit(403) }
	
	## If the forum is replyonly, only moderators and administrators can post new threads
	if ( $forum->{replyonly} eq 'Y' && !$replypost && !$IsPrivileged) { exit(403) }

	## Security Checks
	if (!$user && !$forum->{guest_post}) { exit(403) } ## No Guests Post
	elsif ($forum->{access_group} && ## If restricted
		$user && !$user->CheckGroupList(  ## Restrict access :)
			"$forum->{access_group};$forum->{moderator};$forum->{administrator}"
	)) { exit(403) }

	## bans hack
	#if (!$IsPrivileged) { ## chks bans, unless privileged

	my $bans = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,type,ban FROM forum_bans
		WHERE forum = ? OR forum = '[GLOBAL]'
	~); $bans->execute($forum->{forum});

	my $ip = $www->get_remote_host;
	my $host = gethostbyaddr(pack('C4',split(/\./,$ip)),2);
	if ($host) { undef $host unless (join('.',unpack('C4',gethostbyname($host))) eq $ip) }

	my ($ban,$regex);
	while ($ban = $bans->fetchrow_hashref ) {
		if ($ban->{type} eq 'host') {
			$regex = '^'; $regex .= $ban->{ban};
			$regex =~ s/\./\\./g;
			$regex =~ s/\?/\\w/g;
			$regex =~ s/\*/\\w+/g;
			$regex .= '$';
			if ($host && ($ban->{ban} =~ /^[a-z]$/i)) { ## host
				exit(403) if ($host =~ $regex);
			} else { ## hostname
				exit(403) if ($ip =~ $regex);
			}
		} elsif ($ban->{type} eq 'user' && $user) {
			exit(403) if $user->nickname eq $ban->{ban};
		}

	}#} ## end bans hack

	my ($name,$caid,$rank,$color,$badge,$bold,$sticky);
	$sticky = 0;
	if ($user) { ## If we have a user
		my $attributes = $user->getAttributes('title_abrv','badge');
		$name = $user->showname; $caid = $user->caID;
		$rank = join(' ',@{$attributes->{'title_abrv'}});
		$badge = join(',',@{$attributes->{'badge'}});

		$color = $user->brigade_hex_color if $user->brigade_pos;

		if ($posted->{bold_chk} && !$replypost) {
			$bold = 'X' if $IsPrivileged;
		}
		if ($posted->{sticky_chk}) {
			$sticky = 1 if $IsPrivileged;
		}

	} else { ## Else we are dealing with a guest user

		## Security Checks
		if ($replypost) {
			exit(403) unless ($forum->{guest_post} ne 'R');
		} elsif ($forum->{guest_post} ne 'S') { exit(403) }

		## Sanity Check
		$posted->{name} =~ s/;//g;
		exit(412) unless ($posted->{name});

		my $captcha = CyberArmy::WWW::Captcha
			->new({string => $posted->{'security_code'}});
		if ($captcha) {	$captcha->drop() }
		else { ## no valid captcha
			exit(412) if exists $posted->{'security_code'}; ## failed before, die!
			$captcha = CyberArmy::WWW::Captcha->new(); ## else issue one
			$www->status(403);
			my $r = CyberArmy::WWW::Request->instance();
			delete $posted->{'security_code'};
			$r->printTemplate('forum/captcha.tmpl',{
				posted => $posted,
				id => $captcha->{id},
				forum => $forum,
				elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
			});
			return;
		}

		$www->err_headers_out->add( ## Save his name...
			'Set-Cookie' => 'forumguest='.$posted->{name}.'; '.
			'path=/forum/; expires=Tue, 1-Jan-2030 13:31:33 GMT'
		); ## ... in a cookie
		$name = $posted->{name}; $rank = 'Guest';
	}

	my $cookies = $www->getParams ({ from => 'cookies' , escapehtml => 1 });

	if ($posted->{image_link}) {
		$posted->{image_link} =~ s/^http:\/\/|\(|\)|://g; ## cleaning it up
		unless ($cookies->{image_link} &&
			$posted->{image_link} eq $cookies->{image_link}) {
			$www->err_headers_out->add(
				'Set-Cookie' => 'image_link='.$posted->{image_link}.'; '.
				'path=/forum/; expires=Tue, 1-Jan-2030 13:31:33 GMT'
			);
		}
	} elsif ($cookies->{image_link}) {
		$www->err_headers_out->add(
			'Set-Cookie' => 'image_link=expired; '.
			'path=/forum/; expires=Tue, 1-Jan-1998 13:31:33 GMT'
		);
	}

	if ($posted->{action} eq 'Post') {
		$forum->PostMessage(
			pid => $replypost,
			author => $name,
			author_caID => $caid,
			author_rank => $rank,
			author_color => $color,
			author_ip => $ip,
			author_host => $host,
			author_badge => $badge,
			subject => $posted->{subject},
			img_url => $posted->{image_link},
			body => $posted->{message},
			signature => $posted->{signature},
			bold => $bold,
			sticky => $sticky
		) or exit(404);

		$www->redirectTo('/forum/'.$forum->{forum}.'/');
	} else {	## Preview
		my $r = CyberArmy::WWW::Request->instance();

		my $message_body = $posted->{message}; ## copy
		$message_body .= "\n".$posted->{signature} if ($posted->{signature});

		## render quoted text ('>') in italic
		$message_body =~ s/^(&gt;.*)$/<i>$1<\/i>/mgi;

		## Enable HTML in posts
		if ($forum->{allow_html} eq 'Y') {
			CyberArmy::BBCode::Parse($message_body);
		} else {
			$message_body =~ s/\n/<br\/>\n/gs;
		}

		my $disabled = ''; unless ($user) {
			$cookies->{forumguest} ||= '';
			$disabled = $forum->{guest_post} ne 'R' ? '' : 'disabled';
		}

		my $bold_chk = '';
		if (!$replypost && $user && $user->CheckGroupList("$forum->{moderator};$forum->{administrator}")) {
			$bold_chk = '<input type="checkbox" name="bold_chk" value="ON"'.
			($posted->{bold_chk} ? 'CHECKED ' : '').
			' /> Bold?';
		}
		my $sticky_chk = '';
		if ($user && $user->CheckGroupList("$forum->{moderator};$forum->{administrator}")) {
			$sticky_chk = '<input type="checkbox" name="sticky_chk" value="ON"'.
			($posted->{sticky_chk} ? 'CHECKED ' : '').
			' /> Sticky?';
		}
		my %postform = (
			reply => $posted->{subject},
			bold_chk => $bold_chk,
			sticky_chk => $sticky_chk,
			signature => ($user && $user->signature) ? "\n\n".$user->signature:'',
			image_link => $cookies->{image_link},
			disabled => $disabled
		);
		my %message = (
			mid => $replypost,
			author => $name,
			author_caID => $caid,
			author_rank => $rank,
			author_color => $color,
			author_ip => $ip,
			author_host => $host,
			author_badge => $badge,
			subject => $posted->{subject},
			img_url => $posted->{image_link},
			body => $posted->{message},
			signature => "\n".$posted->{signature},
			bold => $bold,
			sticky => $sticky,
			forum => $forum->{forum}
		);
		if (! $user) { $postform{forumguest} = $cookies->{forumguest} }
		$r->printTemplate('forum/viewpost.tmpl',{
			disabled => $disabled,
			form_nick => FormatShowName($rank,$name),
			postform => \%postform,
			message => \%message,
			message_body => $message_body,
			preview => 1,
			forum => $forum,
			includes => $includes,
			guest => (!$user) ? 'true':undef,
			user => ($user) ? 1:undef,
			info => ($user) ? _info($user):undef,
			elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
		});
	}
}
##############################################
#
# ForumLabelSticky() - Make a thread sticky/non-sticky
#

sub ForumLabelSticky {
	my ($post, $glue) = @_;

	my $user = CyberArmy::WWW::Request::User->instance or exit(403);

	my $message = new CyberArmy::Forum::Message ( id => $post,
		select => 'pid,mid,forum,thread,subject,author_rank,author,sticky,trail'
	) or exit(404);

	my $forum = new CyberArmy::Forum(
		id => $message->{forum}, select => 'forum,administrator,moderator'
	);
	exit(403) unless $user->CheckGroupList(
		"$forum->{administrator};$forum->{moderator}");

	my $update = $CyberArmy::Utils::Database::dbh->prepare('UPDATE forum_replies SET sticky = ? WHERE mid = ? ');
	exit(500) unless $update->execute($glue, $message->{'mid'});

	my $cache = $CyberArmy::Utils::Database::dbh->prepare('UPDATE forum_list SET last_modified=NOW() WHERE forum = ? ');
	exit(500) unless $cache->execute($message->{forum});

	$www->redirectTo('/forum/'.$forum->{forum}.'/');
}
#############################################################
# ForumMoveThread() - Move a thread from a forum to another #
#############################################################

sub ForumMoveThread {
	my $start = Time::HiRes::time; ## Start
	my @moveto;
	my $r = CyberArmy::WWW::Request->instance();
	my ($post) = (shift);

	my $user = CyberArmy::WWW::Request::User->instance or exit(403);

	my $message = new CyberArmy::Forum::Message ( id => $post,
		select => 'pid,mid,forum,thread,subject,author_rank,author,rdate,trail'
	) or exit(404);

	my $forum = new CyberArmy::Forum(
		id => $message->{forum}, select => 'forum,administrator,moderator'
	);

	exit(403) unless $user->CheckGroupList(
		"$forum->{administrator};$forum->{moderator}");


	if ($www->method eq 'POST') {
		my $posted = $www->getParams({from => 'posted', escapehtml => 1});
		exit(412) unless ($posted->{moveto} && $posted->{reason});

		my $toforum = new CyberArmy::Forum (
			id => $posted->{moveto}, select => 'forum,access_group,moderator,administrator'
		) or exit(404);

		exit(403) unless $user->CheckGroupList(
			"$toforum->{access_group};$toforum->{administrator};$toforum->{moderator}"
		);

		$message->moveTo($toforum->{forum}) or exit(500);
		my $votes= CyberArmy::Forum->CountDisapprovalVotes($message->{mid});
		$CyberArmy::Utils::Database::dbh->do(q~
			INSERT INTO log_forum
			(forum, action, dest, thread_id, subject, author, rdate, logby, logby_id, ldate, reason, votes)
			VALUES (?,'move',?,?,?,?,?,?,?,NOW(), ?, ?)~,undef,
		$forum->{forum}, $toforum->{forum}, $message->{mid}, $message->{subject},
			$message->{author_rank}.' '.$message->{author},
			$message->{rdate}, $user->showname, $user->caID, $posted->{reason}, $votes);

		$www->header_out('Location' => '/forum/'.$toforum->{forum}.
			'/messages/'.$message->{mid}.'.html');

		$www->status(302); return;

	} else {
		## Prepare a query to retreive all the forums
		my $forums = $CyberArmy::Utils::Database::dbh->prepare(q~
			SELECT forum,name,administrator,moderator,access_group
			FROM forum_list
			ORDER BY name
		~) or exit(500);
		$forums->execute; ## Exec :)

		while ( my $lforum = $forums->fetchrow_hashref ) {
			next if ($lforum->{forum} eq $forum->{forum});
			if ($user->CheckGroupList("$lforum->{administrator};$lforum->{moderator};$lforum->{access_group}")) {
				push @moveto, {
					forum => $lforum->{forum},
					name => $lforum->{name},
				};
			}
		}
	}
	$r->printTemplate('forum/movethread.tmpl',{
		moveto => \@moveto,
		message => $message,
		forum => $forum,
		title => 'Move Threads',
		includes => $includes,
		info => _info($user),
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
	});
}


#####################################
# ForumDeletePost() - Delete A Post #
#####################################
sub ForumDeletePost {
	my $start = Time::HiRes::time; ## Start
	my ($targetpost,$targetforum,$targethash) = (shift,shift,shift);

	## Sec Check
	my $user = CyberArmy::WWW::Request::User->instance or exit(403);

	my $post = new CyberArmy::Forum::Message (
		id => $targetpost,
		select => 'mid,forum,pid,thread,subject,author_rank,author,rdate'
	) or exit(404);

	## Sanity Check.. half dummy :)
	exit(404) if ($post->{forum} ne $targetforum);

	my $forum = $post->GetForum('forum,administrator,moderator,name');

	exit(403) unless $user->CheckGroupList(
		"$forum->{administrator};$forum->{moderator}"
	);

	if (!$targethash) {
		my $r = CyberArmy::WWW::Request->instance();
		$r->printTemplate('forum/confdel.tmpl',{
			targetforum => $targetforum,
			targetpost => $targetpost,
			subject => $post->{subject},
			forum => $forum,
			title => 'Delete Post',
			includes => $includes,
			info => _info($user),
			elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
		});
		return 1;
	} else {

		$www->checkReferer($www->location . "/$targetforum/") or exit(403);

		my $posted = $www->getParams({ from => 'args' , escapehtml => 1 }); ## POSTed args
		exit(412) unless ($posted->{reason});

		my $page = $post->getPageNumber;
		my ($rows,$tpthread) = (0,0); ## Set Scope...
		if ($post->{pid} == 0) { ## Is a top level thread,
			## then we will simply delete all the thread
			my $kill = $CyberArmy::Utils::Database::dbh->prepare(q~
				DELETE FROM forum_replies
				WHERE thread = ?
			~); $kill->execute($post->{thread});
			$kill->finish;
			$rows = $kill->rows; $tpthread++;

			my $votes= CyberArmy::Forum->CountDisapprovalVotes($post->{mid});
			$CyberArmy::Utils::Database::dbh->do(q~
				INSERT INTO log_forum
				(forum, action, thread_id, subject, author, rdate, logby, logby_id, ldate, reason, votes)
				VALUES (?,'delete',?,?,?,?,?,?,NOW(),?, ?)~,undef,
			$forum->{forum}, $post->{mid}, $post->{subject},
				$post->{author_rank}.' '.$post->{author},
				$post->{rdate}, $user->showname, $user->caID, $posted->{reason}, $votes);
		} else {## else we'll gotta select the target's followups only

			my $getthread = $CyberArmy::Utils::Database::dbh->prepare(qq~
				SELECT mid,pid FROM forum_replies WHERE thread = ? ORDER BY rdate DESC
			~);

			$getthread->execute($post->{thread});
			my $posts = $getthread->fetchall_arrayref({}); ## fetchx0r
			$getthread->finish; ## kill the handle...

			## Initialize some flags...
			my $have_kids; my $i = -1; my $replies = {}; my $ind = {};
			foreach (@{$posts}) { ## Build replies list
				if ( $_->{pid} ) {
					$have_kids++ if $_->{pid} eq $post->{mid};
					push @{ $replies->{ $_->{pid} } } , $_->{mid} }
				$i++; $ind->{$_->{mid}} = $i;
			}

			my $sweep; my $target = $post->{mid};

			if ($have_kids) {
				while ($target) {
					if (@{$replies->{$target}}) {
						foreach (@{$replies->{$target}}) {
							if (!$posts->[$ind->{$_}]->{printed}) {
								$posts->[$ind->{$_}]->{printed}++;
								$sweep .= $_.',';
							}

							if ( exists $replies->{$_} && @{$replies->{$_}}) { ## g0t replies?
								$posts->[$ind->{$_}]->{parent} = $target; $target = $_; last;
							} else { shift @{$replies->{$target}}; last;}

						}
					} else {
						if (exists $posts->[$ind->{$target}]->{parent}) {
							$target = $posts->[$ind->{$target}]->{parent};
						} else { undef $target; last  }
					}
				}

				$sweep .= $post->{mid}.',';
			} else { $sweep = $target }

			{ local $/ = ','; chomp $sweep; }
			my $kill = $CyberArmy::Utils::Database::dbh->prepare(qq~
				DELETE FROM forum_replies WHERE mid IN ($sweep)
			~); $kill->execute;
			$rows = $kill->rows;
			$kill->finish;


			my $votes= CyberArmy::Forum->CountDisapprovalVotes($post->{mid});
			$CyberArmy::Utils::Database::dbh->do(q~
				INSERT INTO log_forum
				(forum, action, thread_id, subject, author, rdate, logby, logby_id, ldate, reason, votes)
				VALUES (?,'delete',?,?,?,?,?,?,NOW(),?, ?)~,undef,
			$forum->{forum}, $post->{mid}, $post->{subject},
				$post->{author_rank}.' '.$post->{author},
				$post->{rdate}, $user->showname, $user->caID, $posted->{reason}, $votes);

		}

		my $last_post = $CyberArmy::Utils::Database::dbh
		->selectrow_hashref(q~
			SELECT MAX(mid) as max_mid FROM forum_replies WHERE forum = ?
		~,undef,$forum->{forum});

		 $CyberArmy::Utils::Database::dbh->do(qq~
			UPDATE forum_list
			SET last_modified = NOW(),
			threads_num = threads_num - $tpthread,
			posts_num = posts_num - $rows,
			 last_post_id = ?
			WHERE forum = ?
		~,undef,$last_post->{max_mid},$targetforum);

		if ($page > 1) {
			$www->redirectTo('/forum/'.$post->{forum}.'/page/'.$page.'.html');
		} else { $www->redirectTo('/forum/'.$post->{forum}.'/') }
	}
}

##################################
# ForumLogs() - Logging facility #
##################################

sub ForumLogs {
	my $start = Time::HiRes::time; ## Start
	my ($logsforum,$user) = (shift,CyberArmy::WWW::Request::User->instance);

	my $forum = new CyberArmy::Forum (
		id => $logsforum,
		select =>'access_group,moderator,administrator,name,forum'
	) or exit(404);

	if ($forum->{access_group}) {
		exit(403) unless $user && $user->CheckGroupList(
			"$forum->{access_group};".
			"$forum->{moderator};".
			"$forum->{administrator}"
		)
	}

	my $logs = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,action,dest,thread_id,subject,author,rdate,logby,ldate,reason,votes
		FROM log_forum WHERE forum = ? OR dest = ? ORDER by ldate DESC LIMIT 0,50~
	);

	$logs->execute($forum->{forum}, $forum->{forum});

	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate('forum/logs.tmpl',{
		logs => $logs->fetchall_arrayref({}),
		includes => $includes,
		info => ($user) ? _info($user):undef,
		title => "$forum->{name}'s Logs",
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
		});
	$logs->finish;
}



#########################################
# ForumInfo() - Displays info on forum #
# to membership (e.g. moderator list)   #
#########################################

sub ForumInfo {
	my $start = Time::HiRes::time; ## Start
	my ($logsforum,$user) = (shift,CyberArmy::WWW::Request::User->instance);

	my $forum = new CyberArmy::Forum (
		id => $logsforum,
		select =>'access_group,moderator,administrator,name,forum,smalldesc,allow_html,searchable,locked,replyonly,administrator,moderator,access_group'
	) or exit(404);


	if ($forum->{access_group}) {
		exit(403) unless $user && $user->CheckGroupList(
			"$forum->{access_group};".
			"$forum->{moderator};".
			"$forum->{administrator}"
		)
	}
	my @access_list = getAccessListUsers($forum->{access_group});
	my @moderator_list = getAccessListUsers($forum->{moderator});
	my @administrator_list = getAccessListUsers($forum->{administrator});
	
	
	
	

	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate('forum/info.tmpl',{
		forum => $forum,
		access_list => @access_list ? \@access_list : undef,
		moderator_list => @moderator_list ? \@moderator_list : undef,
		administrator_list => @administrator_list ? \@administrator_list : undef,
		includes => $includes,
		info => ($user) ? _info($user):undef,
		title => "$forum->{name}'s Details",
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
		});
}



#####################################
# ForumSearch() - Search The Forums #
#####################################
sub ForumSearch {
	my $start = Time::HiRes::time; ## Start
	my ($searchforum,$user,$html_out) = (shift,CyberArmy::WWW::Request::User->instance);
	my $posted = $www->getParams({ from => 'posted' , escapehtml => 1 }); ## POSTed args

	my @results; ## Results output..
	if ($posted->{forum}) { ## SomeOne Is Searching
		$searchforum = $posted->{forum}; ## Set the searched forum...

		my $forum = new CyberArmy::Forum (
			id => $searchforum,
			select =>'access_group,searchable,moderator,administrator'
		);

		## Security Checks
		if ($forum->{access_group}) {
			exit(403) unless $user && $user->CheckGroupList(
				"$forum->{access_group};".
				"$forum->{moderator};".
				"$forum->{administrator}"
			)
		} elsif ( $forum->{searchable} ne 'Y' ) {
			exit(403) unless $user && $user->CheckGroupList(
					"$forum->{moderator};".
					"$forum->{administrator}"
				)
		}

		## Sanity Check
		exit(412) if (!$posted->{terms});

		## Initialize...
		my @fields; my $sqlquery .= q~
			SELECT mid,forum,bold,subject,author_rank,author,author_badge,
				author_color,rdate
			FROM forum_replies
			WHERE ~;

		my $inchk; ## Specify what fields are to be searched
		if ($posted->{in_msg}) { push @fields, 'body'; $inchk++}
		if ($posted->{in_subjs}) { push @fields, 'subject'; $inchk++}
		if ($posted->{in_auth}) { push @fields, 'author'; $inchk++}

		exit(412) unless $inchk;

		$sqlquery .= '(';

		## Terms Search
		if ($posted->{match} eq 'All') {
			my @terms = split / /,$posted->{terms};
			my $j; foreach my $field (@fields) {
				$j++; $sqlquery .= ' OR ' if ($j > 1);
				my $i; foreach my $term (@terms) {
					$i++; my $term = $CyberArmy::Utils::Database::dbh->quote($term); $term =~ s/'(.*)'/'%$1%'/;
					$sqlquery .= "($field like $term)";  $sqlquery .= ' AND ' if ($i <= $#terms);
				}
			}
		} elsif ($posted->{match} eq 'Any') {
			my @terms = split / /,$posted->{terms};
			my $j; foreach my $field (@fields) {
				$j++; $sqlquery .= ' OR ' if ($j > 1);
				my $i; foreach my $term (@terms) {
					$i++; my $term = $CyberArmy::Utils::Database::dbh->quote($term); $term =~ s/'(.*)'/'%$1%'/;
					$sqlquery .= "($field like $term)";  $sqlquery .= ' OR ' if ($i <= $#terms);
				}
			}
		} elsif ($posted->{match} eq 'Exact') {
			my $i; foreach my $field (@fields) {
				$i++; my $term = $CyberArmy::Utils::Database::dbh->quote($posted->{terms}); $term =~ s/'(.*)'/'%$1%'/;
				$sqlquery .= "($field like $term)";  $sqlquery .= ' OR ' if ($i <= $#fields);
			}

		} else { exit(412)  }

		$sqlquery .= ')';

		##Date Parameters
		if ($posted->{date}) {
		if ($posted->{date} eq 'before') {
			my $date = $posted->{year_1}.'-'.$posted->{month_1}.'-'.$posted->{day_1};
			$sqlquery .= ' AND (rdate < '.$CyberArmy::Utils::Database::dbh->quote($date).')';
		} elsif ($posted->{date} eq 'after') {
			my $date = $posted->{year_1}.'-'.$posted->{month_1}.'-'.$posted->{day_1};
			$sqlquery .= ' AND (rdate > '.$CyberArmy::Utils::Database::dbh->quote($date).')';
		} elsif ($posted->{date} eq 'between') {
			my $date = $posted->{year_1}.'-'.$posted->{month_1}.'-'.$posted->{day_1};
			my $date2 = $posted->{year_2}.'-'.$posted->{month_2}.'-'.$posted->{day_2};
			$sqlquery .= ' AND (rdate > '.$CyberArmy::Utils::Database::dbh->quote($date).') AND (rdate < '.$CyberArmy::Utils::Database::dbh->quote($date2).') ';
		}}

		## No Guest?
		$sqlquery .= q~ AND (author_rank != 'Guest' )~ if $posted->{no_guest};

		## Final Touch
		$sqlquery .= ' AND (forum = '.$CyberArmy::Utils::Database::dbh->quote($posted->{forum}).') ORDER BY rdate DESC';

		## Limits?
		$sqlquery .= ' LIMIT 0,'.$posted->{num} if ($posted->{num} =~ /^\d+$/ && $posted->{num} <= 100);
		## Finally Execute The Query
		my $search = $CyberArmy::Utils::Database::dbh->prepare($sqlquery); $search->execute;

		while ( my $found = $search->fetchrow_hashref  ) {
			push @results, { text => FormatReplyLine({ %{$found} }) };
		}
	}

	## Select the list of forums the user has search access to
	my $forums = $CyberArmy::Utils::Database::dbh->prepare(q~
		SELECT forum,name,administrator,moderator,access_group
		FROM forum_list	WHERE searchable = 'Y'
	~); $forums->execute;

	## Generate the html list of forums the user has search access to
	my @list; while ( my $one = $forums->fetchrow_hashref ) {
		next if ($one->{access_group} && ! ($user && $user->CheckGroupList("$one->{access_group};$one->{moderator};$one->{administrator}")) );
		my $selected = $searchforum && $one->{forum} eq $searchforum ? 'selected' : '';
		push @list, {
			selected => $selected,
			forum => $one->{forum},
			name => $one->{name}
		};
	}

	## Print Out The Html
	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate('forum/search.tmpl',{
		list => \@list,
		results => \@results,
		includes => $includes,
		info => ($user ) ? _info($user):undef,
		title => 'Search The CyberArmy Forums',
		elapsed => sub { return sprintf('%.5f',Time::HiRes::time - $start) }
		});
}

sub ForumRssView {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance(); 

	my $messages = [];
	my $rss = new XML::RSS (version => '1.0');
	my $link = $r->getServerLink().'/forum/';

	if (my $id = shift) { ## specific forum
		my $forum = CyberArmy::Forum->new( id => $id, select =>
			'forum,name,smalldesc,access_group,moderator,administrator'
		) or $r->exit (404);

		## security check
		$r->exit(403) if ($forum->{access_group} && ## If restricted
			(	!$user || ## And there is no valid user
				!$user->CheckGroupList(      ## Or the user
				  "$forum->{access_group};". ## doesn't have
				  "$forum->{moderator};".    ## the required
				  "$forum->{administrator}"  ## clearance
				)
			)
		);

		$messages = CyberArmy::Forum->getLatestMessages({
			from => [$forum->{'forum'}], count => 50
		});

		$rss->channel(
			title		=> $forum->{'name'},
			link		=> $link.$forum->{'forum'}.'/',
			description	=> $forum->{'smalldesc'},
			dc			=> {
				publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
				language	=> 'en-US'
			},
			syn			=> {
				updatePeriod     => 'hourly',
				updateFrequency  => 1,
			}
		);

	} else {
		my $getForumList = $CyberArmy::Utils::Database::dbh->prepare('
			SELECT forum,access_group,administrator,moderator
				FROM forum_list ORDER by forum
		');
		$getForumList->execute;

		my (@authorized_forums);
		while ( my $f = $getForumList->fetchrow_hashref() ) {
			if ($f->{'access_group'}) {
				next unless defined($user) && 
					$user->CheckGroupList(
						"$f->{access_group};".
						"$f->{administrator};".
						"$f->{moderator}"
					);
			}
			push @authorized_forums, $f->{'forum'};
		}
		
		$messages = CyberArmy::Forum->getLatestMessages({
			from => \@authorized_forums, count => 100
		});

		$rss->channel(
			title		=> 'CyberArmy Forums',
			link		=> $link,
			description	=> 'Your CyberArmy Forums messages',
			dc			=> {
				publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
				language	=> 'en-US'
			},
			syn			=> {
				updatePeriod     => 'hourly',
				updateFrequency  => 1,
			}
		);

	}

	foreach (@$messages) {
		$rss->add_item(
			title	=> $_->{'subject'},
			link	=> $link.$_->{'forum'}.'/messages/'.$_->{'mid'}.'.html',
			dc		=> {
				creator	=> $_->{'author_rank'}.' '.$_->{'author'},
				subject	=> $_->{'forum'},
				date	=> $_->{'w3cdate'},
			}
		);
	}

	$r->content_type('application/xml');
	$r->print( $rss->as_string );

}


###############################################
#
# ForumGetPrevious() - Get the previous (=newer) message (weird, but consistent with pages)
#
sub ForumGetPrevious {
	my ($getforum,$getmid) = (shift,shift);

	my $prev = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
		SELECT forum,mid
		FROM forum_replies
		WHERE forum = ? AND mid > ?
		ORDER BY mid ASC
		LIMIT 1
	~,undef,$getforum,$getmid);
	if (defined $prev->{forum} && defined $prev->{mid}
	    && $getforum eq $prev->{forum}) {
		return $getforum . '/messages/' . $prev->{mid} . '.html';
	} else {
	#	ForumError(404,'Message does not exist');
		## Get the forum object
		my $forum = new CyberArmy::Forum ( id => $getforum,
			select => 'forum,name,access_group,moderator,administrator'
		) or exit(404);
		## we'll check access rights when viewing the forum
		return $getforum . '/';
	}
}

###############################################
#
# ForumGetNext() - Get the next (=older) message (weird, but consistent with pages)
#
sub ForumGetNext {
	my ($getforum,$getmid) = (shift,shift);

	my $next = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
		SELECT forum,mid
		FROM forum_replies
		WHERE forum = ? AND mid < ?
		ORDER BY mid DESC
		LIMIT 1
	~,undef,$getforum,$getmid);
	if (defined $next->{forum} && defined $next->{mid}
	    && $getforum eq $next->{forum}) {
		return $getforum . '/messages/' . $next->{mid} . '.html';
	} else {
	#	ForumError(404,'Message does not exist');
		## Get the forum object
		my $forum = new CyberArmy::Forum ( id => $getforum,
			select => 'forum,name,access_group,moderator,administrator'
		) or exit(404);
		## we'll check access rights when viewing the forum
		return $getforum . '/';
	}
}

###############################################
#
# ForumGetPreviousThread() - Get the previous (=newer) thread (weird, but consistent with pages)
#
sub ForumGetPreviousThread {
	my ($getforum,$getthread) = (shift,shift);

	my $cookies = $www->getParams({ from => 'cookies' , escapehtml => 1 });
	## Sort by thread (0) or by latest post (1)
	my $sorted = defined $cookies->{sorted} ? $cookies->{sorted} : 0;

	my $prev;
	if ($sorted) {
		my $thread = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT forum,thread,MAX(rdate) as max_rdate
			FROM forum_replies
			WHERE forum = ? AND thread = ?
			GROUP BY thread
		~,undef,$getforum,$getthread);
		if (defined $thread->{forum} && defined $thread->{thread}
		    && $getforum eq $thread->{forum} && $getthread == $thread->{thread}) {
			$prev = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
				SELECT forum,thread,MAX(rdate) as max_rdate
				FROM forum_replies
				WHERE forum = ?
				GROUP BY thread
				HAVING max_rdate > ?
				ORDER BY max_rdate ASC
				LIMIT 1
			~,undef,$getforum,$thread->{max_rdate});
		}
	} else {
		$prev = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT forum,thread
			FROM forum_replies
			WHERE forum = ? AND thread > ?
			ORDER BY thread ASC
			LIMIT 1
		~,undef,$getforum,$getthread);
	}
	if (defined $prev->{forum} && defined $prev->{thread}
	    && $getforum eq $prev->{forum}) {
		return $getforum . '/threads/' . $prev->{thread} . '.html';
	} else {
	#	ForumError(404,'Thread does not exist');
		## Get the forum object
		my $forum = new CyberArmy::Forum ( id => $getforum,
			select => 'forum,name,access_group,moderator,administrator'
		) or exit(404);
		## we'll check access rights when viewing the forum
		return $getforum . '/';
	}
}

###############################################
#
# ForumGetNextThread() - Get the next (=older) thread (weird, but consistent with pages)
#
sub ForumGetNextThread {
	my ($getforum,$getthread) = (shift,shift);

	my $cookies = $www->getParams({ from => 'cookies' , escapehtml => 1 });
	## Sort by thread (0) or by latest post (1)
	my $sorted = defined $cookies->{sorted} ? $cookies->{sorted} : 0;

	my $next;
	if ($sorted) {
		my $thread = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT forum,thread,MAX(rdate) as max_rdate
			FROM forum_replies
			WHERE forum = ? AND thread = ?
			GROUP BY thread
		~,undef,$getforum,$getthread);
		if (defined $thread->{forum} && defined $thread->{thread}
		    && $getforum eq $thread->{forum} && $getthread == $thread->{thread}) {
			$next = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
				SELECT forum,thread,MAX(rdate) as max_rdate
				FROM forum_replies
				WHERE forum = ?
				GROUP BY thread
				HAVING max_rdate < ?
				ORDER BY max_rdate DESC
				LIMIT 1
			~,undef,$getforum,$thread->{max_rdate});
		}
	} else {
		$next = $CyberArmy::Utils::Database::dbh->selectrow_hashref(q~
			SELECT forum,thread
			FROM forum_replies
			WHERE forum = ? AND thread < ?
			ORDER BY thread DESC
			LIMIT 1
		~,undef,$getforum,$getthread);
	}
	if (defined $next->{forum} && defined $next->{thread}
	    && $getforum eq $next->{forum}) {
		return $getforum . '/threads/' . $next->{thread} . '.html';
	} else {
	#	ForumError(404,'Thread does not exist');
		## Get the forum object
		warn $getforum;
		my $forum = new CyberArmy::Forum ( id => $getforum,
			select => 'forum,name,access_group,moderator,administrator'
		) or exit(404);
		## we'll check access rights when viewing the forum
		return $getforum . '/';
	}
}

###############################################
#
# ForumCastVote() - Attempts to add a disapproval vote against a post.
#
sub ForumCastVote {
	my ($first,$post_id,$user) = (shift,shift,CyberArmy::WWW::Request::User->instance);
	my $message;
	if($user) {
			my $hasVoted = CyberArmy::Forum->HasVoted($post_id, $user);
			if($hasVoted == 0) {
				CyberArmy::Forum->AddDisapprovalVote($post_id, $user);
				$message = 'Vote successfully recorded! <br/>Moderation will be notified when the number of votes against this post reach a set threshold.';
			}
			else {
				$message = 'You have already voted against this post - you can only vote once. <br/>Moderation will be notified when the number of votes against this post reach a set threshold.';
			}
	} else {
		$message = 'You must have a CyberArmy account and be logged in to cast a vote of disapproval on posts!';
	}

	my $link = '/forum/'.$first.'/messages/'.$post_id.'.html';
	my $r = CyberArmy::WWW::Request->instance();

	$r->printTemplate('forum/vote.tmpl',{
		link => $link,
		message => $message,
		info => ($user) ? _info($user):undef,
		});

}

sub ForumStats {
	my $recentTopPosters = CyberArmy::Forum->getRecentTopPosters();
	my $topPosters = CyberArmy::Forum->getTopPosters();
	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate('forum/stats.tmpl',{
		recentTopPosters => $recentTopPosters,
		topPosters => $topPosters
		});

}

##############################################################################################
# Small And Shortcut Subroutines...							  #
##############################################################################################


##############################################
#
# FormatStickyLine() - Common or not so common shortcut for formatting
#
sub FormatStickyLine {
	my $tags = shift; my $rep = '<LI>';

	#if ($tags->{star}) { $rep .= qq~<font color="red"><b>&gt;</b></font>~ }
	if ($tags->{mid}) {
		if ($tags->{nested} && $tags->{thread}) {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/threads/$tags->{thread}.html#$tags->{mid}"><font color="#FFAA33"><b>[Sticky]</b>~;
		} else {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/messages/$tags->{mid}.html"><font color="#FFAA33"><b>[Sticky]</b>~;
		}
	}
	if ($tags->{bold}) { $rep .= '<b>' }
	$rep .= ' '.$tags->{subject};
	if ($tags->{bold}) { $rep .= '</b>' }
	if ($tags->{mid}) { $rep .= '</font></a>' }
	$rep .= ' - ';
	$rep .= FormatShowName($tags->{author_rank},$tags->{author},$tags->{author_color}) . ' ';
	if ($tags->{'author_badge'}) {
		foreach (split/,/,$tags->{'author_badge'}) {
			$rep .= qq~<img src="$includes/$_" border="0" height=15 width=15 alt="$_" /> ~
		}
	}
	$rep .= '<i>'.$tags->{rdate}.'</i>'."\n";
	#if ($tags->{star}) { $rep .= qq~<font color="red"><b>&lt;</b></font>~ }
	return $rep;
}
###############################################
#
# FormatReplyLine() - Common ShortCut for formating
#
sub FormatReplyLine {
	my $tags = shift;
	my $rep = $tags->{subject};
	$tags->{mid} =  ($tags->{nested} && $tags->{thread}) ? qq~<a class="mark" href="/forum/$tags->{forum}/threads/$tags->{thread}.html#$tags->{mid}">~:
			qq~<a class="mark" href="/forum/$tags->{forum}/messages/$tags->{mid}.html">~;
	$rep = "$tags->{mid}$rep</a> - ";
	if ($tags->{bold}) { $rep = "<b>$rep</b>" }
	$rep .= FormatShowName($tags->{author_rank},$tags->{author},$tags->{author_color}) . ' ';
	if ($tags->{'author_badge'}) {
		foreach (split/,/,$tags->{'author_badge'}) {
			$rep .= qq~<img src="$includes/$_" border="0" height=15 width=15 alt="$_" /> ~
		}
	}
	$rep .= '<i>'.$tags->{rdate}.'</i>'."\n";
	if ($tags->{star}) { $rep = qq~<font color="red"><b>&gt;</b></font>$rep<font color="red"><b>&lt;</b></font>~ }
	return '<LI>'.$rep;
}

###############################################
#
# FormatThreadLine() - Common ShortCut for formating
#
sub FormatThreadLine {
	my $tags = shift; my $rep = '';
	if ($tags->{alt}) {
		$rep .= '<tr class="alt"><td>';
	} else {
		$rep .= '<tr><td>';
	}

	if ($tags->{star}) { $rep .= qq~<font color="red"><b>&gt;</b></font>~ }
	elsif ($tags->{mid}) {
		if ($tags->{nested} && $tags->{thread}) {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/threads/$tags->{thread}.html#$tags->{mid}">~;
		} else {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/messages/$tags->{mid}.html">~;
		}
	}
	if ($tags->{bold}) { $rep .= '<b>' }
	$rep .= ' '.$tags->{subject};
	if ($tags->{mid}) { $rep .= '</a>' }
	if ($tags->{threaded}) { $rep .= ' - '; }
	if ($tags->{bold}) { $rep .= '</b>' }
	$rep .= '</td><td width=220 nowrap>';
	$rep .= FormatShowName($tags->{author_rank},$tags->{author},$tags->{author_color});
	if ($tags->{author_badge}) {
		foreach (split/,/,$tags->{author_badge}) {
			$rep .= qq~ <img src="$includes/$_" border="0" height=15 width=15 alt="$_"/> ~
		}
	}
	$tags->{replies} ||= 1;
	$tags->{replies} -= 1;
	$rep .= qq~</td><td align="center" width=60>$tags->{replies}</td><td width=150 nowrap>~;
	if ($tags->{lastid} != $tags->{mid}) {
		if ($tags->{nested} && $tags->{thread}) {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/threads/$tags->{thread}.html#$tags->{lastid}">~
		} else {
			$rep .= qq~<a class="mark" href="/forum/$tags->{forum}/messages/$tags->{lastid}.html">~;
		}
	}
	$rep .= '<font size="-1">'.$tags->{date}.'</font>';
	if ($tags->{lastid} != $tags->{mid}) {
		$rep .= '</a>';
	}
	if ($tags->{star}) { $rep .= qq~<font color="red"><b>&lt;</b></font>~ }
	$rep .= "</td></tr>\n";
	return $rep;
}

###############################################
#
# FormatShowName() - Common ShortCut for formating
#
sub FormatShowName {
	my ($rank,$name,$color) = @_;
	if ($rank eq 'Guest') { return '<b>Guest</b>('.$name.')' }
	elsif ($color) { return '<b>'.$rank.' <font color="'.$color.'">'.$name.'</font></b>'}
	else { return '<b>'.$rank.' '.$name.'</b>' }
}

###############################################
#
# ForumError() - ShortCut to print out errors
#
sub ForumError {
	my $code = $_[1] ? $_[0] : 0;
	my $err = defined $code ? $_[1] : $_[0];

	Apache::exit($code);
}


sub switchUserForumView {

	my $cookie = '';

	if ($_[1] eq 'collapse') {
		$cookie = 'threaded=0';
	} elsif ($_[1] eq 'expand') {
		$cookie = 'threaded=1';
	} elsif ($_[1] eq 'bythread') {
		$cookie = 'sorted=0';
	} elsif ($_[1] eq 'bypost') {
		$cookie = 'sorted=1';
	} elsif ($_[1] eq 'viewthread') {
		$cookie = 'nested=1';
	} elsif ($_[1] eq 'viewpost') {
		$cookie = 'nested=0';
	}


	$www->err_headers_out->add('Set-Cookie' => $cookie.'; path=/forum/; expires=Tue, 1-Jan-2030 13:31:33 GMT') if $cookie;

	($_[2] && $_[2] > 1) ?
		$www->redirectTo('/forum/'.$_[0].'/page/'.$_[2].'.html') :
		$www->redirectTo('/forum/'.$_[0].'/');
}

sub toggleAsHeadline {
	my $user = CyberArmy::WWW::Request::User->instance or exit(403);
	my $message = CyberArmy::Forum::Message->new( id => shift,
	select => 'mid,pid,forum,frontpage' ) or exit(404); ## check the id is valid

	## security checks
	my $forum = $message->GetForum('administrator,moderator');
	exit(403) unless $user->CheckGroupList(
		"staff,$forum->{administrator};$forum->{moderator}"
	);

	$message->{'frontpage'} ? ## toggle it
		$message->unsetAsHeadline() : $message->setAsHeadline();

	$www->redirectTo('/my/');
}

sub getAccessListUsers {
	
	my $access_group = $_[0] ? $_[0] : 0;
	my @access_list;
	foreach my $subgroup (split /;+/, $access_group) {
		foreach my $subsubgroup(split /,+/, $subgroup) {
		    my ($type) = substr($subsubgroup,0,1);
			my ($num) = substr($subsubgroup,1,length($subsubgroup));
		    my $added=0;
			if($type eq '~'){
				push @access_list, $num;
			}
			elsif (($num && $num =~ /^\d+$/)
					&& ($type eq 't'||$type eq 'b'||$type eq 'x')) {
				my $users;				
				my $brigade = CyberArmy::Brigades->new($num,
					'brigade_name,brigade_color,brigade_url,brigade_recruiting,'
						.'brigade_joinmode,brigade_parent,brigade_invite'
				);
				if($brigade){
					if ($type eq 't') { 
						 $users = $brigade->GetUsers( subusers => 1,
								select => 'showname,nickname,retired,'.
									'brigade,brigade_pos,brigade_parent,brigade_name,away'
							); } ## somewhere in the chain
					elsif ($type eq 'b') { 
						 $users = $brigade->GetUsers(
								select => 'showname,nickname,retired,'.
									'brigade,brigade_pos,brigade_parent,brigade_name,away'
							); }  ## normal
					elsif ($type eq 'x')  { 
						 $users = $brigade->GetUsers(
								select => 'showname,nickname,retired,'.
									'brigade,brigade_pos,brigade_parent,brigade_name,away'
							); 
					}  	## C|X/O
					
					foreach my $member (@$users) {
						if($type eq 't' ||
						   $type eq 'b' ||
						   ($type eq 'x' && 
						   	(($member->brigade eq $brigade->id && $member->brigade_pos >= 3) || 
						   	 ($member->brigade ne $brigade->id && $member->brigade_pos >= 4)) ) )
						{
							push @access_list, $member->nickname;
						}
					}
				}
			}
			else {
				my $members = CyberArmy::Groupware->getUsers($subsubgroup);
				foreach my $member (@$members) {
					push @access_list, $member->{nickname};
				}
			}

	
		}
	}
	#do not duplicate usernames
	my @access_list2;
	foreach my $member(@access_list){
		my $added=0;
			foreach my $added_member(@access_list2){
					if ($member eq $added_member){
						$added=1;
					}
				}
				if($added==0){
				  	push @access_list2, $member;
				}
	}
	
	return @access_list2;

}

1;
