#!perl -wW
package CyberArmy::WWW::Brigades;

use strict;

use CyberArmy::Brigades;
use CyberArmy::User;
use CyberArmy::Forum;

$CyberArmy::WWW::Brigades::VERSION = '1.0b';

sub handler  {

	my $r = CyberArmy::WWW::Request->instance();

	my $path = $r->path_info;
	CyberArmy::WWW::Utils::escapeHtml($path);
	my (undef,$node,@subnode) = split /\//, $path;

	if (!$node) { &showbrig(1) }
	elsif ($node =~ /^[0-9]+$/ && !@subnode) { &showbrig($node) }
	elsif ($subnode[0] eq "lastlogins") { &lastlogins($node) }
	elsif ($subnode[0] eq "homepage") { &BrigadeHomepage($node) }
	elsif ($subnode[0] eq 'join' ) { &JoinBrigade($node,$subnode[1]) }
	elsif ($subnode[0] eq "alogs") { &logs($node,"all",$subnode[1]) }
	elsif ($subnode[0] eq "logs") { &logs($node,"single",$subnode[1]) }
	elsif ($subnode[0] eq "tree") { &brigadetree($node) }
	elsif ($subnode[0] eq "report") {
		if ($r->method eq "POST") { &submitreport($node) }
		else { &reportform($node) }
	}
	elsif ($subnode[0] eq "command") {
		if ($subnode[1] && $subnode[1] eq "promote"){
				if ($r->method eq "POST") {
					&PromotePost($node)
				} else { &PromoteForm($node) }
		} else {
			if ($r->method eq "POST") {
				&UpdateBrigade($node,$subnode[1])
			} else { &showcommand($node,$subnode[1]) }
		}
	}
 	elsif ($subnode[0] eq "adminproject" && $subnode[1]) {
 		shift @subnode;
		if ($r->method eq "POST") {
			&UpdateProject($node,@subnode)
		} else { &AdminProject($node,@subnode) }

	} else { return -1; } ## DECLINED

	## we return whatever value our called subroutine returns

}


sub showbrig {
	my $target=shift;
	my $user = CyberArmy::WWW::Request::User->instance();

	my $r = CyberArmy::WWW::Request->instance();

	my $brigade = CyberArmy::Brigades->new($target,
		'brigade_name,brigade_description,brigade_parent,brigade_color,'.
		'brigade_hex_color,brigade_recruiting,brigade_joinmode,brigade_invite'
	) or $r->errorTemplate(404,'brigades/error_nobrig.tmpl');

	my $toplevel;
	if ($user && ( $user->IsInGroup('x'.$brigade->id) ||
					$user->IsInGroup('brigade_coordinator') )) {
		$toplevel = 1;
	}


	my $parent = $brigade->parent ?
		CyberArmy::Brigades->new($brigade->parent,
			'brigade_name,brigade_parent,brigade_color'	) : undef;

	my $users = $brigade->GetUsers(
		mclass => 1, select => 'showname,nickname,pubemail,'.
		'retired,brigade_id,brigade,brigade_pos,brigade_name,'.
		'brigade_parent,brigade_hex_color'
	);

	my ($cmdr,@vicecmdrs,@members,@subcmdrs);

	foreach (@$users) {
		if ($_->brigade == $brigade->id) {
			if ($_->brigade_pos == 1) { push @members, $_  }
			elsif ($_->brigade_pos == 2) { push @vicecmdrs, $_  }
			elsif ($_->brigade_pos >= 3) { $cmdr = $_  }
		} else {
			if ($_->brigade_pos == 3) { push @subcmdrs, $_  }
			elsif ($_->brigade_pos == 5) {
				push @vicecmdrs, $_; push @subcmdrs, $_
			}
			else { push @subcmdrs, $_;  }
		}
	}
	
	my $invited = $brigade->invite 
		? ($user &&$user->CheckGroupList($brigade->invite)) : 0;

	my $projects = $brigade->GetProjects();

	$r->errorTemplate(403,'brigades/error_custom.tmpl',{
		msg => "This group hasn't got a commander yet!"
	}) unless $cmdr;

	$r->printTemplate('brigades/brigade.tmpl', {
		user => $user,
		invited => $invited,
		toplevel => $toplevel,
		brig  => $brigade,
		parent => $parent,
		cmdr => $cmdr,
		members => @members ? \@members : undef,
		vicecmdrs => @vicecmdrs ? \@vicecmdrs : undef,
		subcmdrs => @subcmdrs ? \@subcmdrs : undef,
		projects => $projects ? $projects : undef,
		projecttype => \%CyberArmy::Projects::AccessHash
	});

	return 0;
}



sub showcommand {
	my ($cbrig,@options) = @_;
	my $r = CyberArmy::WWW::Request->instance();

	my $user = CyberArmy::WWW::Request::User->instance()
		or return 403;

	if ($options[0]) {

		if ($options[0] ne "newsubbrigade") {

			my $brigade = CyberArmy::Brigades->new($options[0],
			'brigade_name,brigade_parent,brigade_color,brigade_hex_color'
			.',brigade_description,brigade_acr,brigade_projac'
			) or return 404;

			my $possible_cos = $brigade->getPosCOs($cbrig);

			$r->printTemplate('brigades/brigedit.tmpl',{
				brig => $brigade,
				possible_cos => $possible_cos,
			});

		} else { #add new subbrigade

			my $brigade = CyberArmy::Brigades->new($cbrig,
			'brigade_name,brigade_parent,brigade_color,brigade_hex_color'
			.',brigade_description,brigade_acr,brigade_projac'
			) or return 404;

			my $possible_cos = $brigade->getPosCOs;

			$r->printTemplate('brigades/brignew.tmpl', {
				parent => $cbrig,
				possible_cos => $possible_cos
			});
		}


	} else { #show command panel

		my $brigade = CyberArmy::Brigades->new($cbrig,
			'brigade_name,brigade_color,brigade_url,brigade_recruiting,'
				.'brigade_joinmode,brigade_parent,brigade_invite'
		) or return 404;

		return 403 unless (
			$user->IsInGroupList(
				'brigade_coordinator','x'.$brigade->id)
			|| $user->IsInGroup('sushi'));

		my $toplevel;
		if ( $user && ( $user->IsInGroup('topco') ||
						$user->IsInGroup('brigade_coordinator') ) ) {
			$toplevel = 1;
		}



		my $users = $brigade->GetUsers( mclass => 1,
			select => 'showname,nickname,retired,'.
				'brigade,brigade_pos,brigade_parent,brigade_name,away'
		);

		my (@vicecmdrs,@members,@subcmdrs,@queue,@relatedbrigades);
		foreach (@$users) {
			if ($_->brigade == $brigade->id) {
				if (!$_->brigade_pos) { push @queue, $_  }
				elsif ($_->brigade_pos == 1) { push @members, $_  }
				elsif ($_->brigade_pos == 2) { push @vicecmdrs, $_  }
			} else {
				if ($_->brigade_pos == 5) { push @vicecmdrs, $_ }
				if ($_->brigade_pos >= 3) {
					push(@subcmdrs, $_);
					push(@relatedbrigades,$_->brigade);
				}
			}
		}


		foreach (split(/,/,$brigade->chain)) {
			push(@relatedbrigades,$_) if (($_ != $brigade->id) &&
				$user->IsInGroupList('brigade_coordinator','x'.$_));
		}

		my $team = ''; # The team cms fix, by Ranok
		foreach(@$users) {
			$team .= $_->nickname.';';
		} # end team fix


		my $topbrigade = $user->GetTopLevelBrigade();
		my $url = $brigade->url ?  $brigade->url : '';
		my $invite = $brigade->invite ? $brigade->invite : '';
		my $projac = $brigade->projac;
		my $projects = $brigade->GetProjects();

		$r->printTemplate('brigades/command.tmpl',{
			user => $user,
			toplevel => $toplevel,
			projects => $projects,
			vicecmdrs => \@vicecmdrs,
			subcmdrs => \@subcmdrs,
			team => $team,
			members => \@members,
			joinqueue => \@queue,
			brig => $brigade,
			relatedbrigades => \@relatedbrigades
		});

	}

	return 0;
}


sub PromoteForm {
	my $brig = shift;
	my $r = CyberArmy::WWW::Request->instance();

	my $get = $r->getParams(  {escapehtml => 1} );
	my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	$r->errorTemplate(403,'brigades/error_custom.tmpl',{
		msg => "You have not enough access to promote a brigade member!"
	}) unless (($user->IsInGroup('topco')) && ($user->IsInBrigade($brig)));

	unless (($get->{user}) && ($get->{user} ne "")) {
		$r->errorTemplate(412,'brigades/error_nouser.tmpl');
	}
	my $victim = CyberArmy::User->new( nickname => $get->{user} )
		or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

	$r->errorTemplate(404,'brigades/error_custom.tmpl',{
		msg => "Promotee is not in your brigade!"
	}) unless ($victim->IsInBrigade($brig));

	$r->printTemplate('brigades/promote_form.tmpl', { user => $victim });

	return 0;
}


sub PromotePost {
	my $brig = shift;
	my $r = CyberArmy::WWW::Request->instance();
	my $posted = $r->getParams( {from => 'posted', escapehtml => 1} );
    my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	return 403 unless (
		(($user->IsInGroup('topco')) && ($user->IsInBrigade($brig)))
		|| $user->IsInGroup('sushi')
	);

	$r->errorTemplate(412,'brigades/error_nouser.tmpl')
		unless (($posted->{user}) && ($posted->{user} ne ""));

	my $victim = CyberArmy::User->new( nickname => $posted->{user} )
		or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

	$r->errorTemplate(412,'brigades/error_custom.tmpl', {
		msg => "Promotee not in your brigade!"
	}) unless ($victim->IsInBrigade($brig));

	return 412 unless (($posted->{user}) && ($posted->{reason}));

	my $current_rank = $victim->getAttributes('title');

	# Better way to do this?
	my $new_rank = "";
	if ($victim->IsInGroup('lieutenant')) { $new_rank = 'captain'; }
	elsif ($victim->IsInGroup('secondlieutenant')) { $new_rank = 'lieutenant'; }
	elsif ($victim->IsInGroup('trooper')) { $new_rank = 'secondlieutenant'; }
	else { $new_rank = 'trooper'; }

	my $forum = CyberArmy::Forum->new(
		id => $r->dir_config->get("Default_Forum"));
	my $showname = $victim->showname;
	my $cmdr_rank = $user->getAttributes('title_abrv');

	$forum->PostMessage(
		author=> $user->nickname,
		author_caID => $user->caID,
		author_rank => join(' ',@{$cmdr_rank->{'title_abrv'}}),
		subject => "[Promotion] ",
		body => "Promotion of ".$showname." from ".$current_rank->{title}[1].
				" to ".$new_rank."Reason: ".$posted->{reason}
	);

	$victim->Update( addtogroup => $new_rank );

	$victim->Log( type => 'promoted',
		      logby => $user->nickname,
		      lobycaID => "$user->caID",
		      msg => "to $new_rank"
	);

	$r->redirectTo('/brigades/'.$brig.'/command');

}



sub logs {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	my $brigade = CyberArmy::Brigades->new($_[0],
		'brigade_parent,brigade_name') or return 404;

	return 403 unless (
		$user->IsInGroupList('brigade_coordinator','x'.$brigade->id)
		|| $user->IsInGroup('sushi')
	);

	my $logs = $brigade->getLogs(mode => $_[1], type => $_[2], limit => $_[3]);

	if ($_[1] eq "single") {
		$r->printTemplate('brigades/logs.tmpl', {
			brig => $brigade, logs => $logs
		});
	} else {
		$r->printTemplate('brigades/alogs.tmpl', {
			brig => $brigade, logs => $logs
		});
	}

	return 0;
}



sub reportform {
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance() or $r->errorTemplate(403,'brigades/error_loginfirst.tmpl');

	my $brigade = CyberArmy::Brigades->new(shift,'brigade_id,brigade_name')
			or $r->errorTemplate(404,'brigades/error_nobrig.tmpl');

	return 403 unless (
		$user->IsInGroup('x'.$brigade->id) ||
		$user->IsInGroup('brigade_coordinator') ||
		$user->IsInGroup('sushi')
	);

	$r->printTemplate('brigades/report_form.tmpl', { brig => $brigade });

	return 0;
}


sub submitreport {
	my $r = CyberArmy::WWW::Request->instance();

	my $brigade = CyberArmy::Brigades->new(shift,
		'brigade_parent,brigade_name'
	) or return 404;

	my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	return 403 unless (
		$brigade->parent == 1 && ($user->IsInGroup('x'.$brigade->id)
			|| $user->IsInGroup('sushi'))
	);

	my $posted = $r->getParams( {from => 'posted' , escapehtml => 1} );
	my $subject = "[Report] ".$brigade->name;
	## TODO this needs to be templated
	my $message ="\n[b]SWOT Analysis[/b]\n".((defined($posted->{swot})
							&& $posted->{swot} !~ /^\s*$/)?$posted->{swot}:"N/A").
				"\n[b]Long-term Goals[/b]\n".((defined($posted->{ltgoals})
							&& $posted->{ltgoals} !~ /^\s*$/)?$posted->{ltgoals}:"N/A").
				"\n[b]Short-term Goals[/b]\n".((defined($posted->{stgoals})
							&& $posted->{stgoals} !~ /^\s*$/)?$posted->{stgoals}:"N/A").
				"\n[b]Completed Goals[/b]\n".((defined($posted->{completedgoals})
							&& $posted->{completedgoals} !~ /^\s*$/)?$posted->{completedgoals}:"N/A").
				"\n[b]Problems[/b]\n".((defined($posted->{problems})
							&& $posted->{problems} !~ /^\s*$/)?$posted->{problems}:"N/A").
				"\n[b]Progress[/b]\n".((defined($posted->{progress})
							&& $posted->{progress} !~ /^\s*$/)?$posted->{progress}:"N/A").
				"\n[b]Sub-brigade Activity[/b]\n".((defined($posted->{sbactivity})
							&& $posted->{sbactivity} !~ /^\s*$/)?$posted->{sbactivity}:"N/A");


	my $forum = new CyberArmy::Forum (
		id => $r->dir_config->get("Default_Forum")
	) or  $r->errorTemplate(500,'brigades/error_failed.tmpl');

	my $attr = $user->getAttributes('title_abrv','badge');

	$forum->PostMessage(
		author => $user->showname,
		author_rank => join(' ',@{$attr->{'title_abrv'}}),
		author_color => ($user->brigade_pos?$user->brigade_hex_color:undef),
		author_ip => $r->get_remote_host,
		author_host => $r->hostname,
		author_badge => join(' ',@{$attr->{'badge'}}),
		subject => $subject,
		body => $message
	) or  $r->errorTemplate(403,'brigades/error_failed.tmpl');

	CyberArmy::Message->new(
		caID => [$user->caID],
		user => $user,
		subject => $subject,
		body => $message
	);

	$r->redirectTo('/brigades/'.$brigade->id.'/command/');

}



sub lastlogins {
	my $r = CyberArmy::WWW::Request->instance();
	my @bp = ("Queued","Member","X/O","Sub C/O",'',"X/O");
	my $user = CyberArmy::WWW::Request::User->instance() or return 403;
	my $brigade = CyberArmy::Brigades
		->new(shift,'brigade_id,brigade_name') or return 404;

	return 403 unless (
		$user->IsInGroup('x'.$brigade->id) ||
		$user->IsInGroup('brigade_coordinator') ||
		$user->IsInGroup('sushi')
	);

	my $users = $brigade->GetUsers( mclass => 1,
		select => 'nickname,showname,prvemail,brigade,brigade_pos,'.
		'retired,session_time,session_ltime,session_id,session_timeout'
	);

	my @members;
	foreach (@$users) {
		if ($_->brigade == $brigade->id) { next; }
		else { if ($_->brigade_pos == 5) { push @members, $_ } }
	}

	foreach (@$users) {
		if ($_->brigade == $brigade->id) {
			if ($_->brigade_pos == 2) { push @members, $_ }
		} else { next; }
	}

	foreach (@$users) {
		if ($_->brigade == $brigade->id) { next; }
		else {
			if ($_->brigade_pos == 3) { push @members, $_ }
		}
	}

	foreach (@$users) {
		if ($_->brigade == $brigade->id) {
			if ($_->brigade_pos == 1) { push @members, $_ }
		} else { next; }
	}

	foreach (@$users) {
		if ($_->brigade == $brigade->id) {
			if ($_->brigade_pos == 0) { push @members, $_ }
		}
		else { next; }
		undef $_;
	}


	my @memberdiffs;
	foreach (@members) {
		if ($_->HasValidSessionTime) {
			@memberdiffs[$_->caID] = "Logged In Since ".
				gmtime($_->session_time)." UTC";

		} elsif ($_->session_ltime) {
			my $diff = '';
			my $now = time;
			foreach (CyberArmy::Utils::getTimeInterval(
				($now - $_->session_ltime))
			) {
				if ($_->[0]) {
					$_->[1] =~ /^(\w)/;
					$diff .= "$_->[0]$1 "
				}
			}

			$diff .= 'ago' if ($diff);
			$memberdiffs[$_->caID] = "Last Seen ".$diff." UTC";
			undef $diff;
		} else {
			$memberdiffs[$_->caID] = "User Never Logged In";
		}
	}

	$r ->printTemplate('brigades/lastlogins.tmpl',{
		user => $user,
		brig => $brigade,
		members => \@members,
		memberdiffs => \@memberdiffs,
		bp => \@bp,
	});

	return 0;
}

sub JoinBrigade {
	my ($target,$overwrite) = (shift,shift);
	my $r = CyberArmy::WWW::Request->instance();
	my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	## Check If Already In Another Brigade
	$r->errorTemplate(403,'brigades/error_custom.tmpl',{
		msg => "You are already in a brigade (".$user->brigade.")"
	}) if $user->brigade_pos;

	my $brigade = CyberArmy::Brigades->new($target,
		'brigade_name,brigade_recruiting,brigade_joinmode,brigade_invite')
		or $r->errorTemplate(404,'brigades/error_nobrig.tmpl');

	my $invited = $brigade->invite ? $user->CheckGroupList($brigade->invite):0;

	## Verify that this brigade allows new recruits
	$r->errorTemplate(403,'brigades/error_custom.tmpl', {
		msg => "This brigade doesn't accept new recruits at the moment!"
	}) unless ($brigade->recruiting eq 'O' || $invited);

	$r->redirectTo('/brigades/'.$target)
		unless ($r->checkReferer('/brigades/'.$target));

	## If Already Queued In Another Brigade
	if ($user->brigade && ($user->brigade_pos == 0)) {
		if ($user->brigade == $brigade->id) {
			$r->errorTemplate(403,'brigades/error_alreadyqueued.tmpl')
		}

		unless ($overwrite) {
			$r->errorTemplate(403,'brigades/error_alreadyqueued.tmpl',{
				brig => $brigade, user => $user , same => 1
			})
		}
	}

	## Verify We Have A Commander
	my $commander; unless ($commander = $brigade->Commander) {
		$r->errorTemplate(403,'brigades/error_failed.tmpl');
	}

	if ($brigade->joinmode eq 'O' || $invited ) {## Open Joins or invited

		## We Add The User Immediately
		$user->Update(addtobrigade => $brigade->id)
			or $r->errorTemplate(500,'brigades/error_failed.tmpl');

		## Tell Him
		$r->printTemplate('brigades/join.tmpl',
			{ brig => $brigade, user => $user });

		CyberArmy::Message->new( ## And Mail His Commander
			caID => [$commander->caID], user => $user,
			subject => 'New Recruit In '.$brigade->name,
			body => $user->showname.' joined your brigade'.
					($invited ? ' (invited)' : '')
		);

	    &autoPromoteRecruits($user);

		if ($invited) {

			##log it
			$brigade->Log(
				on => 'user',
				user => \$user,
				type => 'user_join',
				msg => 'Invited',
			);
		} else {
			##log it
			$brigade->Log(
				on => 'user',
				user => \$user,
				type => 'user_join',
				msg => 'Brigade open',
			);
		}

	} else { ## If Joins Are Restricted
		## We add him to the queue
		$user->Update(addtobrigadeq => $brigade->id)
			or $r->errorTemplate(500,'brigades/error_failed.tmpl');
		$r->printTemplate('brigades/join.tmpl',
			{ brig => $brigade, user => $user });
	}

	return 0;
}


sub UpdateBrigade {
	my ($brig,@options) = (shift,@_);
	my $r = CyberArmy::WWW::Request->instance();
	my $commander = CyberArmy::WWW::Request::User->instance() or return 403;

	my $brigade = CyberArmy::Brigades->new($brig,
		'brigade_name,brigade_parent,brigade_color,brigade_url,'.
		'brigade_color,brigade_hex_color'
	) or return 404;

	return 403 unless (
		$commander->IsInGroup('x'.$brigade->id) ||
		$commander->IsInGroup('sushi')
	);

	return 412	unless $r->checkReferer('/brigades/'); ## Referer Check

	my $posted = $r->getParams( {from => 'posted' , escapehtml => 1} );

	if ($posted->{settings}) {
		return 412 unless ($posted->{brigade_url} &&
			$posted->{brigade_joinmode} && $posted->{brigade_recruiting});

		my $projac;
		my $topbrigade = $commander->GetTopLevelBrigade();
		if (($brigade->parent == 1) || ($commander->IsInGroup('x'.$topbrigade))) {
			$projac = ($posted->{projac} eq "on") ? "y" : "n";
		}


		$brigade->update(
			url => $posted->{brigade_url},
			recruiting => $posted->{brigade_recruiting},
			joinmode => $posted->{brigade_joinmode},
			application => $posted->{brigade_application},
			invite => $posted->{brigade_invite},
			projac => $projac,
		);


		$brigade->Log(
			on => 'brigade',
			type => 'brig_modified',
			msg => $posted->{log},
			action_by => $commander->nickname,
			action_by_caID => $commander->caID
		);

	} elsif ($posted->{project_action}) {

		$r->errorTemplate(412,'brigades/error_custom.tmpl', {
			msg => "Select project and confirm please!"
		}) unless (($posted->{confirm}) && ($posted->{list2}));

		# Security checks..
		my $project = CyberArmy::Projects->new($posted->{list2},
			"proj_id,proj_brigade_parent,proj_group"
		) or  $r->errorTemplate(412,'brigades/error_noproject.tmpl');
		return 403 unless ($project->brigade_parent == $brigade->id);

		$project->delete;

		$brigade->Log(
			on => 'brigade',
			type => 'brig_subdeleted',
			msg => $posted->{log} || 'deleted project',
			action_by => $commander->nickname,
			action_by_caID => $commander->caID
		);

	} elsif ($posted->{user_action}) { ## user related updates

		my $user = CyberArmy::User->new(
			nickname => $posted->{list},
			select => 'showname,brigade,brigade_pos,brigade_parent'
		) or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

		my $level = $user->IsInBrigade($brigade->id)
			## user isn't in the given brigade, stop here
			or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

		## commander doesn't have enough privileges to modify the user
		return 403 if (($commander->IsInBrigade($brigade->id)<=$level)
				and not $commander->IsInGroup('sushi')
		);

		if ($level >= 1 && $level <= 5) { ## Normal User Operations
			## flag if the user is in a privileged context, namely,
			## if he is an XO, or a sub-brigade commander
			my $priviliged_role = (($user->IsInBrigade($brigade->id)==2)
					|| ($user->IsInBrigade($brigade->id)==4)) ? 1 : 0;

			if (($posted->{action} eq 'move')) {

				$r->errorTemplate(403,'brigades/error_custom.tmpl', {
					msg => "revoke user's command, first!"
				}) if $priviliged_role;

				my $moveto = $posted->{"moveto".($user->nickname)} or
					$r->errorTemplate(412,'brigades/error_custom.tmpl', {
						msg => "Where do you want to move the user to..?"
					}); #Dont know where to move the user to...

				my $targetbrigade =	CyberArmy::Brigades->new($moveto,
					'brigade_name,brigade_parent,brigade_color,'.
					'brigade_url,brigade_color,brigade_hex_color'
				) or return 404;

				return 403 unless ## moving to a brigade that isn't his
					($commander->IsInBrigade($targetbrigade->id)
					 || $commander->IsInGroup('sushi')
				);

				if ($user->Update(addtobrigade => $moveto))	{ ## move him
					$brigade->Log(
						on => 'user',
						user => \$user,
						type => 'user_discharge',
						msg => $posted->{log} || 'moving to #'.$targetbrigade->id,
						action_by => $commander->nickname,
						action_by_caID => $commander->caID
					);

					$targetbrigade->Log(
						on => 'user',
						user => \$user,
						type => 'user_join',
						msg => $posted->{log} || 'moved from #'.$brigade->id,
						action_by => $commander->nickname,
						action_by_caID => $commander->caID
					);
				}

			} elsif ($posted->{action} eq 'promote') {
				my $victim = CyberArmy::User->new( nickname => $posted->{list} )
					or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

				return 403 unless (
					$commander->IsInGroup('topco') ||
					$commander->IsInGroup('sushi')
				);

				$r->errorTemplate(403,'brigades/error_custom.tmpl', {
					msg => "You can't promote higher than Captain!"
				}) unless (not $victim->IsInGroup('captain'));

				$r->redirectTo('/brigades/'.$brig.
					'/command/promote/?user='.$posted->{list});

			} elsif ($posted->{action} eq 'discharge' ||
						$posted->{action} eq 'kick') {

				$r->errorTemplate(403,'brigades/error_custom.tmpl', {
					msg => "revoke user's command, first!"
				}) if $priviliged_role;

				$user->Update(leavebrigade => 1);

				$brigade->Log(
					on => 'user',
					user => \$user,
					type => 'user_'.$posted->{'action'},
					msg => $posted->{log},
					action_by => $commander->nickname,
					action_by_caID => $commander->caID
				);
				
				# Quick hack to fix grammar & avoid dischargeed/kickd
				my $grammarsuffix;
				if($posted->{'action'} eq 'kick') {
					$grammarsuffix = "ed";
				} else {
					$grammarsuffix = "d";
				}
				CyberArmy::Message->new(
					caID => [$user->caID], user => $commander,
					subject => ucfirst($posted->{'action'})
						. $grammarsuffix .' from '.$brigade->name,
					body => $posted->{log} ||  ## Reason
						'You have been '.$posted->{'action'}.$grammarsuffix.
							' from '.$brigade->name
				) ;

			} elsif ($posted->{action} eq 'setxo') {

				$r->errorTemplate(403,'brigades/error_custom.tmpl', {
					msg => "user is already an xo"
				}) unless ($commander->IsInBrigade($brigade->id) > 4);

				$user->Update( setasbrigadexo => $brigade->id)
					or $r->errorTemplate(500,'brigades/error_failed.tmpl');

				$brigade->Log(
					on => 'user',
					user => \$user,
					type => 'user_setxo',
					msg => $posted->{log},
					action_by => $commander->nickname,
					action_by_caID => $commander->caID
				);
			} elsif ($posted->{action} eq 'unsetxo') {
				$user->Update( unsetasbrigadexo => $brigade->id)
					or $r->errorTemplate(500,'brigades/error_failed.tmpl');

				$brigade->Log(
					on => 'user',
					user => \$user,
					type => 'user_unsetxo',
					msg => $posted->{log},
					action_by => $commander->nickname,
					action_by_caID => $commander->caID
				);

			} else { return 404}

		} elsif ($level == -1) { ## Queue Operations :)

			if ($posted->{action} eq 'accept_recruit') {
				$user->Update( addtobrigade => $brigade->id)
					or $r->errorTemplate(500,'brigades/error_failed.tmpl');

				$brigade->Log(
					on => 'user',
					user => \$user,
					type => 'user_join',
					msg => $posted->{log},
					action_by => $commander->nickname,
					action_by_caID => $commander->caID
				);

				my $msg = $posted->{log} || 'Hello, '.$user->showname.
					'. Your request to join '.$brigade->name.
					' has been accepted! Please contact your commanding officer as soon as possible.';
				CyberArmy::Message->new(
					caID => [$user->caID], user => $commander,
					subject => $brigade->name, body => $msg
				);
				
	    		&autoPromoteRecruits($user);

			} elsif ($posted->{action} eq 'reject_recruit') {
				$user->Update( leavebrigade => $brigade->id)
					or $r->errorTemplate(500,'brigades/error_failed.tmpl');

				my $msg = $posted->{log} || 'Sorry, '.$user->showname.
					': your request to join '.
					$brigade->name.' has been declined.';

				CyberArmy::Message->new(
					caID => [$user->caID], user => $commander,
					subject => $brigade->name, body => $msg
				);
			}
		} else { return 404 }

	} elsif ($posted->{brigade_action} && $posted->{brigade_action_type}) {

		## Updating  Or Adding A New subbrigade
		if ($posted->{brigade_action_type} eq 'update') {

			return 412 unless (	$posted->{name} && exists $posted->{bID}
				&& $posted->{brigade_commander}
			);

			my $user = CyberArmy::User->new(
				nickname => $posted->{brigade_commander},
				select=>'brigade,brigade_pos,brigade_parent'
			) or $r->errorTemplate(412,'brigades/error_nouser.tmpl');

			## Update Existing Brigade
			if (exists $posted->{bID} && $posted->{bID} > 0) {
				my $subbrigade = CyberArmy::Brigades->new($posted->{bID},
					'brigade_parent,brigade_color,brigade_hex_color'
				) or return 404;

				## make sure hierarchy is respected..
				return 403 unless ($subbrigade->parent == $brigade->id);

				return 412 if  ( ## make sure he didn't miss colors
					$subbrigade->parent == 1 && ## when required
					(! $posted->{color} || ! $posted->{hex_color})
				);

				return 403 unless (
					$user->IsInGroup('b'.$subbrigade->id)
					|| $user->IsInGroup('b'.$brigade->id)
					|| $user->IsInGroup('sushi')
				);

				if ($subbrigade->update(
						name => $posted->{name},
						acronym => $posted->{acronym},
						description => $posted->{description},
						color => $posted->{color},
						hex_color => $posted->{hex_color}
					)) {


					my $current_co =
						$subbrigade->Commander('nickname,caID');

					## Update The CO
					unless ($user->caID == $current_co->caID) {
						## If same id, don't bother updating

						return 403 unless (
							$user->IsInGroup('b'.$subbrigade->id)
							|| $user->IsInGroup('b'.$brigade->id)
							|| $user->IsInGroup('sushi')
						);

						$user->Update(
							setasbrigadeco => $posted->{bID}
						) or  $r->errorTemplate(500,
							'brigades/error_failed.tmpl');

						## Log that we removed the old CO
						$subbrigade->Log(
							on => 'user',
							user => \$current_co,
								type => 'user_unsetco',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

						$brigade->Log(
							on => 'user',
							user => \$current_co,
							type => 'user_unsetsubco',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

						## Log that we added the new one
						$subbrigade->Log(
							on => 'user',
							user => \$user,
							type => 'user_setco',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

						$brigade->Log(
							on => 'user',
							user => \$user,
							type => 'user_setsubco',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

					}

					$subbrigade->Log(
						on => 'brigade',
						type => 'brig_cmodified',
						msg => $posted->{log},
						action_by => $commander->nickname,
						action_by_caID => $commander->caID
					);
				} else {
					$r->errorTemplate(500,'brigades/error_custom.tmpl',{
						msg => "Updating failed, please be sure to notify an admin!"
					});
				}

			## Add New Brigade
			} elsif (exists $posted->{bID} && $posted->{bID} == -1) {

				$r->errorTemplate(403,'brigades/error_accessdenied')
					unless ($user->IsInGroup('b'.$brigade->id)
						|| $user->IsInGroup('sushi')
					);


				my ($color,$hex_color);
				if ($brigade->id == 1) {
					return 412 unless
						($posted->{color} && $posted->{hex_color});
					$color = $posted->{color};
					$hex_color = $posted->{hex_color};
				} else {
					 $color = $brigade->color;
					 $hex_color = $brigade->hex_color
				}

				if (my $nID = CyberArmy::Brigades->create(
						parent_id => $brigade->id,
						chain => $brigade->chain,
						name => $posted->{name},
						commander => $posted->{brigade_commander},
						acronym => $posted->{acronym},
						description => $posted->{description},
						color => $color,
						hex_color => $hex_color
					)
				) {
					## Log it!
						$posted->{log} .= ' ('.$posted->{name}
						.'('.$nID.'))';
						$brigade->Log(
							on => 'brigade',
							type => 'brig_subcreated',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

						CyberArmy::Brigades->new($nID)->Log(
							on => 'brigade',
							type => 'brig_created',
							msg => $posted->{log},
							action_by => $commander->nickname,
							action_by_caID => $commander->caID
						);

				} else {
					$r->errorTemplate(500,'brigades/error_custom.tmpl',{
						msg => "Wasn't able to create new Subgroup!"
					});
				}
			}

		} elsif ($posted->{brigade_action_type} eq 'delete') {
			return 412 unless ($posted->{bID});

			my $subgroup = CyberArmy::Brigades->new($posted->{bID},'brigade_id')
				or return 404;

			if ((not $subgroup->hasSubs) && (not $subgroup->hasProjs)) {
				if ($subgroup->delete) {

					$posted->{log} .=
						' ('.$posted->{name}.'['.$posted->{bID}.'])';
					$brigade->Log(
						on => 'brigade',
						type => 'brig_subdeleted',
						msg => $posted->{log},
						action_by => $commander->nickname,
						action_by_caID => $commander->caID
					);

				} else {
					$r->errorTemplate(500,'brigades/error_custom.tmpl',{
						msg => "Wasn't able to delete group!"
					})
				}

			} else {
				$r->errorTemplate(403,'brigades/error_custom.tmpl',{
					msg => "Remove Subgroups/Projects from this group first!"
				})
			}

		} else { return 412 }
	}

	$r->redirectTo('/brigades/'.$brig.'/command/');
}


sub brigadetree {
	my $r = CyberArmy::WWW::Request->instance();
	my $brigade =
		CyberArmy::Brigades->new(shift,'brigade_id,brigade_name')
	or return 404;

	my $bid = $brigade->id;
	my $user = CyberArmy::WWW::Request::User->instance() or return 403;

	return 403 unless (
		$user->IsInGroup('x'.$brigade->id) ||
		$user->IsInGroup('brigade_coordinator') ||
		$user->IsInGroup('sushi')
	);

	my ($tree, $treedetails) = $brigade->getBrigadeTree;

	$r->printTemplate('brigades/brigadetree.tmpl',{
		tree => $tree,
		treedetails => $treedetails
	});

	return 0;
}


sub AdminProject {
	my ($brigadenum,@options) = @_;
	my $user = CyberArmy::WWW::Request::User->instance();
	my $r = CyberArmy::WWW::Request->instance();

	# Access + Stupidity checks
	my $parent_brig = CyberArmy::Brigades->new($brigadenum,
		'brigade_name,brigade_description,brigade_parent,brigade_color,'.
		'brigade_hex_color,brigade_recruiting,brigade_joinmode'
	) or return 404;
	#First problem lies in here (C/O getting a 403)!
	
	return 403 unless (	
		$user->IsInGroupList(
				'brigade_coordinator',
				'x'.$parent_brig->id,
				'proj_coord'
			) || $user->IsInGroup('sushi')
	);
	# </>

	#Define the pre-list vars for brigade command and categories
	my $cat = " ";
	my $brigade_commander = " ";

	my $project;
	# Prepend the current C/O to the list if we're editing the project..
	# And sort out the access type list..
	my $leadername;
	my $type;
	my $accesslist;
	if ($options[0] eq "edit") {
		$project = CyberArmy::Projects->new($options[1],
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,'.
			'proj_type,proj_status,proj_start,proj_end,proj_modified,proj_access'
		) or return 404;

		$leadername = $project->GetLeader()->nickname;
		$brigade_commander .= qq~
			<option SELECTED value="$leadername">$leadername</option>
		~;
		$type = $project->type;
		$cat .= qq~ <option SELECTED value="$type">$type</option> ~;

		my @acl = ('P','I','C');
		my $access = $project->access;
		$accesslist = qq~ <option value="$access" SELECTED>$CyberArmy::Projects::AccessHash{$access}</option>\n~;

		foreach (@acl) {
			$accesslist .= qq~ <option value="$_">$CyberArmy::Projects::AccessHash{$_}</option> ~
			unless ($_ eq $access);
		}

	} elsif ($options[0] eq "accept") {

		$project = CyberArmy::Projects->new($options[1],
			'proj_id,proj_name,proj_brigade_parent,proj_desc,proj_type,proj_status,'.
			'proj_start,proj_end,proj_modified,proj_access,proj_q,proj_group'
		) or return 404;

		my $chain = $parent_brig->chain;
		my ($ca,$top) = split /,/, $chain;
		my $t = ($top) ? $top : $ca;

		#my $top_brigade = CyberArmy::Brigades->new($t);
		#my $topco = $top_brigade->Commander("nickname");

		$r->errorTemplate(403,'brigades/error_custom.tmpl', {
			msg => "top level C/O has to accept the project first"
		}) unless ($user->IsInGroup('x'.$t));

		$r->errorTemplate(412,'brigades/error_custom.tmpl', {
			msg => "project isn't queued"
		}) if ($project->q eq "n");

		my $db = CyberArmy::Database->instance();
		$db->do( ## WARNING, PROJECTS API MISSING ##
			"Update projects set proj_q = 'n' where proj_id = ?",
		undef,$project->id);
	}

	#@{$ -> @$ Phishy 1/1/06
	foreach (@{$parent_brig->GetUsers()}) {
			my $n = $_->nickname;
			my $s = $_->showname;
			$brigade_commander .= qq~ <option value="$n">$s</option>~
				unless ($n eq $leadername);
			undef $n; undef $s;
	}

	if ($options[0] eq "edit") {
		foreach( @{$project->GetUsers()} ) {
			my $n = $_->nickname;
			my $s = $_->showname;
			$brigade_commander .= qq~ <option value="$n">$s</option>~;
			undef $n; undef $s;
		}
	}

	# Get the possible Categories

	my $categories = CyberArmy::Database->instance()->prepare(
			"SELECT * FROM project_categories"
	);

	$categories->execute();
	while (my $row = $categories->fetchrow_hashref()) {
		$cat .= qq~ <option value="$row->{category}">$row->{category_name}</option> ~
			unless ($row->{category} eq $type);
	}

	# Print the template accordingly...
	if ($options[0] eq "edit" || $options[0] eq "new") {
		$r->printTemplate('projects/edit.tmpl',	{
				brig_cmd => $brigade_commander,
				category => $cat,
				project => $project,
				access => $accesslist,
				new => $options[0] eq 'new' ? 1 : 0
		});
	} else {
		$r->redirectTo('/brigades/'.$brigadenum.'/command/');
	}

	return 0;

}


sub UpdateProject {
	my ($brig,@options) = (shift,@_);
	my $r = CyberArmy::WWW::Request->instance();

	my $project; if ($options[1]) {
		$project = CyberArmy::Projects->new($options[1],'
			proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,
			proj_type,proj_status,proj_start,proj_end,proj_modified,proj_access
		') or $r->errorTemplate(404,'brigades/error_noproject.tmpl');
	}

	my $commander = CyberArmy::WWW::Request::User->instance()
		or return 403;

	my $brigade = CyberArmy::Brigades->new($brig,
		'brigade_name,brigade_parent,brigade_color,brigade_url'.
		',brigade_color,brigade_hex_color,brigade_projac'
	) or return 404;

	return 409 unless ($commander->IsInGroup('x'.$brigade->id));


	my $posted = $r->getParams( {from => 'posted' , escapehtml => 1} );

	foreach(qw(name group project_leader access
			description category end_date)) {
		return 412 unless ($posted->{$_});
	}


	$r->errorTemplate(403,'brigades/error_custom.tmpl', {
		msg => "invalid category"
	}) unless CyberArmy::Projects->CheckCategory($posted->{category});

	my $leader = CyberArmy::User->new(
		nickname => $posted->{'project_leader'}
	) or  $r->errorTemplate(412,'brigades/error_nouser.tmpl');


	## if the group name exists
	my $group = CyberArmy::Groupware->get($posted->{'group'});
	if ($group) {
		## check if another project is using this group
		if (CyberArmy::Projects->new($posted->{'group'})) {
			return 409 unless $project
				&& ($project->group eq $posted->{'group'})
		}

		#return 403 unless ## and if we have privileges to (ab)use it
		#	$commander->IsInGroupList('x'.$group->{'brigade'},$group->{'master'})
		

	}

	my $topLvlBrigID = $commander->GetTopLevelBrigade;
	my $BrigID = $commander->brigade;

	my $db = CyberArmy::Database->instance();

	if ($posted->{'project_action_type'} eq 'create') { ## new project
		 return 403 unless ($leader->IsInGroup('t'.$BrigID));

#		my $pc = $brigade->projac;
#		my $q = ($pc eq "y") ? "n" : "y";
		my $q = "n";

		$db->do('
			INSERT INTO projects (
				proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,
				proj_status,proj_start,proj_end,proj_modified,proj_access,proj_q
			) VALUES (?,?,?,?,?,?,NOW(),?,NOW(),?,?)',
		undef,$posted->{name},$brigade->id,$posted->{'group'},
		$posted->{description},$posted->{category},"New",
		$posted->{end_date},$posted->{access},$q);

		$project = CyberArmy::Projects->new(
			$db->{'mysql_insertid'}
		) or return 500; ## insert failed?

		$project->Log( on => "project",
			type => "project_created",
			msg => "\"$posted->{name}\" Project Commissioned",
			action_by => $commander->showname,
			action_by_id => $commander->caID
		);

		my $topco = CyberArmy::Brigades->new($topLvlBrigID)->Commander();
		CyberArmy::Message->new(
			nickname => [$topco->nickname],
			user => $commander, subject => "Project created",
			body => $commander->showname." has created the \"".$posted->{name}.
			"\" project under your brigade. You may now ".
			"<a href=\"/projects/".$project->group."\">review the project</a>.\n\n"
		);

	} elsif ($posted->{'project_action_type'} eq "edit") { #edit a project
		 #COMMENTED FOR REVIEW AND REMOVAL (I think this fucks the whole situation)
		 #$r->errorTemplate(403,'brigades/error_failed.tmpl')
		 #unless ( $leader->IsInGroupList('t'.$brigade->id, $project->group)); #?		

		 #$leader? I think that brigade (C|X)/Os can change the leader and the rest of the project fields
		 #so ...
		 my $luser = CyberArmy::WWW::Request::User->instance() or return 403; #not logged (we checked it before, but to be sure)
			$r->errorTemplate(403,'brigades/error_failed.tmpl') unless (
				$luser->IsInGroupList('x'.$brigade->id, $project->group));
		 	
		my $old_leader = $project->GetLeader();

		$db->do('
			UPDATE projects SET
				proj_name=?, proj_group=?, proj_desc=?, proj_type=?,
				proj_status=?, proj_end=?, proj_modified=NOW(), proj_access=?
			WHERE proj_id = ?',
		undef,$posted->{name},$posted->{group},$posted->{description},
		$posted->{category},$posted->{status},$posted->{end_date},
		$posted->{access},$project->id);

		unless ($old_leader->caID == $leader->caID) {
			$old_leader->Update( addtogroup => $project->group."=1" );
			$leader->Update( addtogroup => $project->group."=3" );
		}

		my $log = ($posted->{log} && ($posted->{log} ne 'log')) ?
			'Project Updated '.$posted->{log} : 'Project Updated';

		$project->Log( on => "project",
			type => "project_updated",
			msg => $log,
			action_by => $commander->nickname,
			action_by_id => $commander->caID
		);
	}

	#In case the POST data is not one we expected (either 'create' or 'edit')
	else
	{
		$r->errorTemplate(403,'brigades/error_failed.tmpl');
	}

	#If we create a new project or change the name of an existing one
	my $new = $posted->{'project_action_type'} eq 'create' ? 1 : 0;
	if ($project->group ne $posted->{'group'} or $new) {

		CyberArmy::Groupware->move(
			$project->group,$posted->{'group'}
		) unless $new and $group;

		CyberArmy::Groupware->set({ ## create/reset properties
			id		=> $posted->{'group'},
			name		=> 'Project: '. $posted->{'name'},
			master		=> 'proj_coord',
			priority	=> 0,
			hitpoints	=> 0,
			brigade		=> $BrigID,
			description	=> $posted->{'description'},
			deprecated_by	=> '',
			depends_on	=> '',
			add_title	=> '',
			add_title_abrv	=> '',
			add_badge	=> '',
			add_role	=> ''
		});

		if ($new) {
			#Add leader to the user_group table
			$leader->Update( addtogroup => $posted->{'group'}."=3" );
		}
	}

	$r->redirectTo('/brigades/'.$brig.'/command/');
}

sub BrigadeHomepage {
	my $r = CyberArmy::WWW::Request->instance();
	my $brigade =
		CyberArmy::Brigades->new(shift,'brigade_name,brigade_url')
			or return 404;

	if ($brigade->name) {
		if ($brigade->url) {
			$r->redirectTo('http://'.$brigade->url);
		} else { $r->errorTemplate(404,'brigades/error_failed.tmpl') }
	} else { $r->errorTemplate(404,'brigades/error_nobrig') }

	return 0;
}

sub autoPromoteRecruits()
{

	my ($user,@options) = (shift);

		if (!$user->IsInGroup('trooper')) {
			# We have a new recruit joining a brigade -> promote him.
			$user->Update( addtogroup => 'trooper' );
			# cMS Him

			# Safe to make a CinC object?
			my $cinc = CyberArmy::Brigades->new(1)->Commander();
			CyberArmy::Message->new(
				caID => [$user->caID],
				user => $cinc,
				subject => 'Congratulations! '.$user->showname,
				body => qq~ Congratulations! Joining a brigade and committing some of your time to CyberArmy is a big step, for this you have been promoted to Trooper. Further promotions will rely on you achieving the tasks set for you by your commanding officer - the higher the rank, the more time it will take you to get there, but the larger the rewards and respect.
					\nGood luck and congratulations again!
					\nCommander in Chief ~.$cinc->showname
			);

			$user->Log(
				type => 'promoted',
				msg => 'auto-promo to trooper',
				logby => $cinc->showname,
				logbycaID => $cinc->caID
			);
			undef $cinc;
		}
}


1;
