package CyberArmy::User;

#
# This Module Provide A Managed OO Read/Write Access
# To The Users (And Indirectly Brigades) Table
#

use strict;
use Digest::MD5;
use CyberArmy::Utils;
use CyberArmy::Brigades;
use CyberArmy::Projects;
use CyberArmy::Database;
use CyberArmy::Groupware;

use CyberArmy::User::Todo;
use CyberArmy::User::PGPKey;
use CyberArmy::User::Skills;
use CyberArmy::User::Messages;


$CyberArmy::User::VERSION = '0.5.5';

our $AUTOLOAD;

sub DESTROY {
	## Since we are using AUTOLOAD(), we define a DESTROY sub
	## so AUTOLOAD() won't bother looking for it
}


## Main Public Object Constructor: Create a new user object
sub new {
	my $class = shift; my %tags = @_; my $key = CyberArmy::Utils::GetKey(\%tags) or return undef;

	my $selects = $tags{select} ? 'caID,'.$tags{select} : 'caID';
	if ( my $fields = _GetData($selects,$key->{name},$key->{value}) ) {
		my $self = { key => 'caID', fields => $fields };
		bless($self,$class); return $self;
	} else { return undef }	
}

sub Forge {
	my $class = shift; my %tags = @_;
	#return undef unless $tags{input}->{$tags{key}};

	my $self = { key => $tags{key}, fields => $tags{input} };
	bless($self,$class); return $self;
}

## Add A User To The Db
sub Add {
	shift; my %tags = @_;
	## See if any required fields are missing	
	foreach ('showname','prvemail','passwd') { 
		return undef unless $tags{$_} 
	}

	my $showname = CyberArmy::Utils::niceShowName($tags{'showname'}) || return undef;
	my $nickname = CyberArmy::Utils::nickFromShowName($showname);
	
	my $db = CyberArmy::Database->instance;
	$db->do(
		'INSERT INTO users (caID,nickname,showname,'.
		'passwd,pubemail,prvemail,challenge_number,joined_on)'.
		' VALUES (?,?,?,MD5(?),?,?,?,NOW())',
	undef,$tags{'caID'},$nickname,$showname,$tags{'passwd'},
	$tags{'prvemail'},$tags{'prvemail'},int(rand(1000))
	) && ($tags{'caID'} || $db->{'mysql_insertid'});
}

## For gettin raw data, used internally
sub _GetData {
	warn
		'(user_query): SELECT '.$_[0].' FROM users '.
		'LEFT JOIN brigades ON brigade = brigade_id '.
		'WHERE '.$_[1].' = ?,'.$_[2] 
			if $ENV{'DINAH_VERBOSE'};

	my $db = CyberArmy::Database->instance;
	$db->selectrow_hashref(
		'SELECT '.$_[0].' FROM users '.
		'LEFT JOIN brigades ON brigade = brigade_id '.
		'WHERE '.$_[1].' = ?',undef,$_[2]
	);
}



sub Select { ## (re-)Select field(s) and cache them
	my $self = shift;

	my $select = shift
		## unless we specify columns to select
		## we will just refresh the cached ones
		|| join(',',keys(%{$self->{'fields'}}));

	if ($select) {
		my $selects = _GetData($select,$self->{'key'},
			$self->{'fields'}->{$self->{'key'}} );

		foreach (keys %{$selects}) { 
			$self->{'fields'}->{$_} = $selects->{$_} 
		} 

		return $self->{'fields'} ? 1 : 0;
	} else { return undef }
}

## Get a field value
sub Field { 
	$_[0]->Select($_[1]) unless (exists $_[0]->{fields}->{$_[1]});
	return $_[0]->{fields}->{$_[1]};
}

## Verify the password
sub Passhash { 
	my $hash = $_[0]->Field('passwd');
	return ($_[1] and ($hash eq Digest::MD5::md5_hex($_[1]))) ? $hash : undef;
}

sub IsInGroup {
	my $self = shift;
	my $groups = $self->groups();
	my $code = 1;

	foreach my $group (@_) {
		#my ($name,$level) = split(/=/, lc $group);
		my ($name,$level) = split(/=/, $group);
		$level = 1 unless defined $level;
		if ($level >= 0) {
			## !! START DEPRECATED, MARKED FOR REMOVAL !!
			my ($type) = substr($name,0,1);
			my ($num) = substr($group,1,length($name));
			if (($num && $num =~ /^\d+$/)
					&& ($type eq 't'||$type eq 'b'||$type eq 'x') 
					&& (my $pos = $self->IsInBrigade($num) )) {
				## Then we check if it is a brigade group 
				if ($type eq 't' &&  $pos >= 1) { next } ## somewhere in the chain
				elsif ($type eq 'b' &&  $pos >= 3) { next } ## normal
				elsif ($type eq 'x' && $pos >= 4) { next } 	## C|X/O
			## !! END DEPRECATED!!
			} elsif ($groups->{$name}) {
				$code = $groups->{$name}->{'level'};
				next if $code >= $level;
			} elsif (substr($group,0,1) eq '~') {
				next if lc $group eq lc '~'.$self->nickname 
			}
		} else { next unless $groups->{$name} }

		return $groups->{'commander'} ? '0E0' : 0; 
	}

	return $code;
}

sub IsInGroupList {
	my ($user,@groups) = (shift,@_);
	foreach (@groups) { 
		if (my $val = $user->IsInGroup($_)) {
			return $val == '0E0' ? 'commander' : $_ 
		}
	} return undef; ## No Matches...
}

# CheckGroupList(user, grouplist) - Parses a grouplist and returns nonzero if
# the user is in the necessary groups, undef otherwise.
#
# i.e. a,b;c;d,e,f ==> user must be in (a and b) or (c) or (d and e and f)
#
# -- Chawmp 2001-11-02
#
sub CheckGroupList {
	my ($user, $grouplist) = @_;
	
	foreach my $subgroup (split /;+/, $grouplist) {
		my $passed = 1;
		foreach (split /,+/, $subgroup) {
			last unless $passed = $user->IsInGroup($_);
		}
		return $passed if $passed;
	}
	return 0;
}

sub getProjects {
	my $self = shift;
	my $db = CyberArmy::Database->instance;
	#Only list non-complete projects
	my $query = $db->prepare('
		SELECT * FROM projects 
		WHERE proj_group IN (
			SELECT group_id FROM user_group WHERE caid = ? AND proj_status !=3
		)
	');	$query->execute($self->caID);

	my @projects; while (my $row = $query->fetchrow_hashref()) {
		push(@projects,CyberArmy::Projects->forge(input=>$row));
	}

	return \@projects;
}


## Lexical Only Subs... make things prettier, that's all
sub IsBanned {
	my $bantime = $_[0]->bantime;
	if ($bantime && ($bantime > time)) {
		$_[0]->IsInGroup('banned') ? $bantime : 0
	} else {
		if ($_[0]->IsInGroup('banned') > 0) {
			$_[0]->Update(ban => 0);
			$_[0]->IsBanned();
		} else { return 0 }
	}
}
sub IsSecurityFlagged {
	$_[0]->IsInGroup('securityflag') > 0
	? $_[0]->Field('securityflag') : 0;
}
sub IsRetired {
	$_[0]->IsInGroup('retired') > 0
	? $_[0]->Field('retired') : 0;
}

sub IsAway {
	$_[0]->IsInGroup('away') > 0
	? $_[0]->Field('away') : 0;
}

sub HasValidSessionTime {
	my ($self,$now) = @_; $now ||= time;
	return undef unless $_[0]->session_id;
	
	return ($_[0]->session_ltime < ($now  - 
	($_[0]->session_timeout * 60 * 60))) ?  0 : 1;
}

sub IsInBrigade {
	my $brig = $_[1] || $_[0]->brigade;
	if ( my $bid = $_[0]->brigade ) {
		if ($bid == $brig) { 
			my $bpos = $_[0]->brigade_pos;
			if ($bpos == 0) {return -1} ## queued
			elsif ($bpos == 1) {return 3} ## normal member
			elsif ($bpos == 2) {return 4} ## XO
			elsif ($bpos >= 3) {return 5} ## CO
		} elsif ($brig == $_[0]->brigade_parent && $_[0]->brigade_pos) {
				if ($_[0]->Field('brigade_pos') == 5) { return 4 } ## XO (& SubCo)
				else { return 2 } ## Simple SubCO
		} else {
			return undef unless $_[0]->brigade_pos;
			
			## We search in the user's chain
			foreach (split /,/ , $_[0]->Field('brigade_chain'))
			{ if ($brig == $_) { return 1 } } ## somewhere down in the chain

			## We attempt to look in the upstream chain
			## we look up the specified brigade's bID
			my $db = CyberArmy::Database->instance;
			my $bchain = $db->selectrow_arrayref(q~
				SELECT brigade_chain 
				FROM brigades 
				WHERE brigade_id = ?~,undef,$brig) or return undef;
			
			## And search for a match thru it
			my $bpos = $_[0]->Field('brigade_pos');
			if ($bpos == 5) {$bid = $_[0]->Field('brigade_parent')}
			foreach (split /,/ , $bchain->[0] ) { 
				if ($bid == $_) { 
					if ($bpos == 2 || $bpos == 5) {return 6} ## Higher Level XO
					elsif ($bpos == 3) {return 7} ## Higher Level CO
					last; ## we stop here, nothing to look for anymore
				} 
			}
		}	
		return 0;
	}
	return undef;
}


sub GetTopLevelBrigade {
	if (my $chain = $_[0]->brigade_chain) {
		my ($ca,$top) = split /,/, $chain;
		if ($top) { return $top } else { $ca }
	} else { return undef }
}

sub getAttributes {
	my $self = shift;
	my @attribute = @_ ? (@_) : qw(title title_abrv badge role);

	my %attibutes;
	
	my $do_role = 0;
	
	my $groups = $self->groups();
	my @groups = sort { ## sort them by priority
			$groups->{$b}->{'priority'} 
				<=> $groups->{$a}->{'priority'}
	} keys %$groups;
	
	foreach (@attribute) {
		$do_role++ if $_ eq 'role';
		foreach my $group (@groups) {
			if ($groups->{$group}->{'add_'.$_}) {
				push(@{$attibutes{$_}},$groups->{$group}->{'add_'.$_})
					unless 
						$groups->{$group}->{'deprecated_by'}
						&& $groups->{ $groups->{$group}->{'deprecated_by'} };
				last if $groups->{$group}->{'priority'} == 99;
			}
		}
	}
	
	if ($do_role && (my $bpos = $self->IsInBrigade)) { ## brigades based ones
		if ($bpos == 3) { 
			push (@{$attibutes{'role'}},
				('Member of '.$self->brigade_name." (#".$self->brigade_id.")"));
		} elsif ($bpos == 4) {
			push (@{$attibutes{'role'}},
				('XO of '.$self->brigade_name." (#".$self->brigade_id.")"));
		} elsif ($bpos == 5) { 
			push (@{$attibutes{'role'}},
				('CO of '.$self->brigade_name." (#".$self->brigade_id.")"));
			unshift(@{$attibutes{'role'}}, ## prepend that he is an xo too
				'XO of '.CyberArmy::Brigades->new($self->brigade_parent,
					'brigade_name')->name." (#".$self->brigade_parent.")") 
				if ($self->IsInBrigade($self->brigade_parent) == 4);
		}	
	}

	return @_ ? {map {$_ => $attibutes{$_} || []} @_} : \%attibutes;
}

## IP logging
sub LogIP {
	my ($self, $ip) = @_;

	# not cacheing previous IP in user object, since it would not work reliably across
	# Apache processes when user is not using secure sessions

	my $row = $CyberArmy::Utils::Database::dbh->selectrow_arrayref(
		'SELECT ip FROM iplog WHERE caid = ? ORDER BY dt DESC LIMIT 1',
		undef, $self->caID);
	my $previp = $row ? $row->[0] : undef;
	
	if (!$previp || ($previp ne $ip)) {
		my $host = gethostbyaddr(pack('C4',split(/\./,$ip)),2);
		if ($host) { undef $host unless (join('.',unpack('C4',gethostbyname($host))) eq $ip) }
		
		$CyberArmy::Utils::Database::dbh->do(
			'INSERT INTO iplog (caid,dt,ip,host) VALUES (?,NOW(),?,?)',
			undef, $self->caID, $ip, $host);
	}
}

sub GetLoggedIPs {
	my $self = shift;

	my $sth = $CyberArmy::Utils::Database::dbh->prepare(
		'SELECT dt,ip,host FROM iplog WHERE caid = ? ORDER BY dt DESC LIMIT 5');
	$sth->execute($self->caID);
	my $result = $sth->fetchall_arrayref({});
	$sth->finish;

	return $result;
}

## Update fields
sub Update {
	my $self = shift; my %tags = @_;
	my ($failed,@update,$string);

	$self->{'backupfields'} = $self->{'fields'}; ## BackUp

	## To Improve
	foreach (keys (%tags)) {
		my $update_cmd = $CyberArmy::User::Update::wrapped{$_} 
			|| 'UpdateString($self,$_ ,$tags{$_})';
		
		if (defined($string = eval('CyberArmy::User::Update::'.$update_cmd))) { 
			push(@update, $string) if $string;
		} else { 
			$self->{'fields'} = $self->{'backupfields'};
			delete $self->{'backupfields'}; return undef
		}

	} 
	
	if ($#update >= 0) {
		my $db = CyberArmy::Database->instance;
		unless ($db->do(
					'UPDATE users SET '.join(', ',@update).
					' WHERE '.$self->{'key'}.' = ? ',
				undef,$self->{'fields'}->{$self->{'key'}})
			) {
				$self->{'fields'} = $self->{'backupfields'};
				delete $self->{'backupfields'}; return undef
			}
	} else {
		## empty updates will update $self itself
		$self->Select(); ## by re-querying the db
		## also re-query groups, if needed
		$self->groups() if delete $self->{'groups'};
	}

	return '0E0';
}

sub Log {
	my $self = shift; my %tags = @_; 
	
	my $db = CyberArmy::Database->instance;
	
	if ($tags{type} eq 'joined') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,date) VALUES (?,?,\'joined\',?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg}) or return undef;
	} elsif ($tags{type} eq 'passreset') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,date) VALUES (?,?,\'passreset\',?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg}) or return undef;		
	} elsif ($tags{type} eq 'modified') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,logby,logbycaID,date) VALUES (?,?,\'modified\',?,?,?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg},$tags{logby},$tags{logbycaID}) or return undef;	
	} elsif ($tags{type} eq 'promoted') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,logby,logbycaID,date) VALUES (?,?,\'promoted\',?,?,?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg},$tags{logby},$tags{logbycaID}) or return undef;	
	} elsif ($tags{type} eq 'banned') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,logby,logbycaID,date) VALUES (?,?,\'banned\',?,?,?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg},$tags{logby},$tags{logbycaID}) or return undef;
	} elsif ($tags{type} eq 'unbanned') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,logby,logbycaID,date) VALUES (?,?,\'unbanned\',?,?,?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg},$tags{logby},$tags{logbycaID}) or return undef;
	} elsif ($tags{type} eq 'note') {
		$db->do(
		'INSERT INTO log_users (caID,nickname,type,msg,logby,logbycaID,date) VALUES (?,?,\'note\',?,?,?,NOW())',
		undef,$self->caID,$self->nickname,$tags{msg},$tags{logby},$tags{logbycaID}) or return undef;	
	}
	
}

sub History {
	my $self = shift; my %tags = @_;

	my $sql = 'SELECT type,nickname,msg,logby,logbycaID,date FROM log_users WHERE caID = ?';
	$sql .= ' AND type = ?' if $tags{events};

	$tags{limit} ||= 1;

	my $db = CyberArmy::Database->instance;
	my $logs = $db->prepare($sql.' ORDER BY date DESC LIMIT 0,'.$tags{limit});
	$tags{events} ? $logs->execute($self->caID,$tags{events})
		: $logs->execute($self->caID);

	return $logs->fetchall_arrayref({});
}


## Delete an account and the user object
sub Kaputt {
	my $db = CyberArmy::Database->instance;

	$db->do(
		'DELETE FROM users WHERE '.$_[0]->{key}.'= ?',
		undef,$_[0]->{fields}->{$_[0]->{key}}
	) or return undef;

	$db->do(
		'DELETE FROM cMS WHERE caID = ?',
		undef,$_[0]->caID
	);

	undef $_[0];
	return 1;
}

## load individual fields

sub AUTOLOAD {
	$AUTOLOAD =~ s/.*://;
	#eval "sub $AUTOLOAD { \$_[0]->Field('$AUTOLOAD') }";
	$_[0]->Field($AUTOLOAD);
}

sub groups {
	my $db = CyberArmy::Database->instance;
	$_[0]->{'groups'} ||= 
		$db->selectall_hashref(
			'SELECT * FROM user_group '.
			'JOIN groupware ON group_id = id '.
			'WHERE caid = ?',
		'group_id',undef,$_[0]->caID);
}

sub pgpkey {
	$_[0]->getPGPKeyRing->pgpkey();
}

package CyberArmy::User::Update;

use strict;
use Digest::MD5;


%CyberArmy::User::Update::wrapped = (
	away => '_wrap_status_group($self,$_ ,$tags{$_})',
	retired => '_wrap_status_group($self,$_ ,$tags{$_})',
	publicplan => '_wrap_text($self,$_ ,$tags{$_})',
	passwd => '_wrap_passwd($self,$tags{$_})',
	showname => '_wrap_showname($self,$tags{$_})',
	nickname => '_wrap_nickname($self,$tags{$_})',
	session_timeout => '_wrap_session_timeout($self,$tags{$_})',
	signature => '_wrap_text($self,$_ ,$tags{$_},256)',

	addtogroup => '_wrap_addtogroup($self,$tags{$_})',
	delfromgroup => '_wrap_delfromgroup($self,$tags{$_})',
	ban => '_wrap_ban($self,$tags{$_})',

	addtobrigadeq => '_wrap_addtobrigadeq($tags{$_})',
	addtobrigade => '_wrap_addtobrigade($tags{$_})',
	setasbrigadexo => '_wrap_setasbrigadexo($self,$tags{$_})',
	unsetasbrigadexo => '_wrap_unsetasbrigadexo($self,$tags{$_})',
	setasbrigadeco => '_wrap_setasbrigadeco($self,$tags{$_})',
	unsetasbrigadeco => '_wrap_unsetasbrigadeco($self,$tags{$_})',
	leavebrigade => '_wrap_leavebrigade($tags{$_})',
);


###### Here Comes The Wrappers ################################################
                                                                              #
                                                                              #
                                                                              #
	sub UpdateString {
		my $db = CyberArmy::Database->instance;
		$_[0]->{fields}->{$_[1]} = $_[2];
		return $_[1].' = '.$db->quote($_[2])
	}

	sub _wrap_passwd {
		my ($self,$pass) = @_;
		my $length = $pass ? length $pass : 0;
		
		if ($length >= 6 ) {
			my $db = CyberArmy::Database->instance;
			 $self->{fields}->{passwd} = Digest::MD5::md5_hex($pass); ## emmdee5ied
			 return 'passwd= '.$db->quote($self->{fields}->{passwd});
		} else { return undef }
	}
	sub _wrap_showname {
		my ($self,$show) = @_;
		
		$show = CyberArmy::Utils::niceShowName($show) || return undef;
		
		if ($self->Field('nickname') eq CyberArmy::Utils::nickFromShowName($show)) { ## compare
			$self->{fields}->{showname} = $show;
			my $db = CyberArmy::Database->instance;
			return 'showname= '.$db->quote($show);
		} else { return undef }
	}

	sub _wrap_nickname {
		my ($self,$nick) = @_;
		## Update the nickname AND showname fields
		$self->{fields}->{showname} = $nick;
		$self->{fields}->{nickname} = $nick;
		my $db = CyberArmy::Database->instance;
		$nick = $db->quote($nick);
		return 'nickname='.$nick.', showname='.$nick
	}

	sub _wrap_session_timeout {
		($_[1] && $_[1] =~ /^\d+$/ && $_[1] >= 1 && $_[1] <= 24) or return undef;
		return 'session_timeout='.$_[1]
	}
	
	sub _wrap_addtogroup { 
		my ($self,$group_add) = @_;

		my $db = CyberArmy::Database->instance;
		
		my ($group,$level) = split(/=/, $group_add);
		
		## make sure the group is valid, and isn't set yet
		my $grpchk = CyberArmy::Groupware->get($group)
			or return undef;

		if ($grpchk->{'depends_on'}) { ## there is a dependency
			_wrap_addtogroup($self,$grpchk->{'depends_on'});
		}
		
		$db->do( ## add any new groups
			'REPLACE INTO user_group (caid,group_id,level) 
				VALUES (?,?,?)',
		undef,$self->caID,$group,($level || 1));
		
		delete $_[0]->{'groups'};

		return '';
	}
	
	sub _wrap_delfromgroup {
		my ($self,$group) = @_;

		my $db = CyberArmy::Database->instance;
		
		my $grpchk = CyberArmy::Groupware->get('depends_on',$group);
		foreach (@$grpchk) { _wrap_delfromgroup($self,$_->{'id'}) }

		$db->do( ## del from the specified group
			'DELETE FROM user_group WHERE caid=? AND group_id=?',
		undef,$self->caID,$group);
		
		delete $_[0]->{'groups'};

		return '';
	}

	sub _wrap_status_group {
		$_[2] ?	_wrap_addtogroup(@_[0,1]) 
				: _wrap_delfromgroup(@_[0,1]);
		&UpdateString
	}
	
	sub _wrap_ban {
		my ($self,$bantime) = @_;
		$bantime += time if $bantime;
		$self->{fields}->{bantime} = $bantime;
		$self->{fields}->{session_id} = undef;
		_wrap_status_group($self,'banned',$bantime);
		my $db = CyberArmy::Database->instance;
		return 'bantime='.$db->quote($bantime).',session_id=NULL';
	}

	sub _wrap_text { ## cause 'text' sql types can't be specified a custom length
		my ($self,$field,$text,$limit) = @_;
		my $length = $text ? length $text : 0;
		$limit ||= 10240;
		if ( $length < $limit ) {
			$self->{fields}->{$field} = $text;
			my $db = CyberArmy::Database->instance;
			return $field.'='.$db->quote($text);
		} else { return undef }
	}

	sub _wrap_addtobrigadeq {
		return undef unless $_[0];
		return "brigade='$_[0]',brigade_pos=0";
	}

	sub _wrap_addtobrigade {
		return undef unless $_[0];
		return "brigade='$_[0]',brigade_pos=1";
	}
	
	sub _wrap_setasbrigadexo {
		my ($self,$brignum) = @_;
		return undef unless $brignum;
		if (my $brigade = CyberArmy::Brigades->new($brignum)) { ## Brigade Exists
			if ($self->IsInBrigade($brignum) == 2) ## Magic for ppls with
			{ return "brigade=".$self->Field('brigade').",brigade_pos=5" } ## subCO status
			else { return "brigade='$brignum',brigade_pos=2"; }
		} else { return undef }
	}

	sub _wrap_unsetasbrigadexo {
		my ($self,$brigade) = @_;
		return undef unless $brigade;
		
		if ($self->IsInBrigade($brigade) == 4) {
			if ($self->brigade_pos == 5) {
				return "brigade='".$self->brigade."',brigade_pos=3"
			} elsif ($self->brigade_pos == 2) { return "brigade='".$self->brigade."',brigade_pos=1" }
			else { return undef }
		
		}
	}

	sub _wrap_setasbrigadeco {
		my ($self,$brignum) = @_;
		return undef unless $brignum;
		
		if (my $brigade = CyberArmy::Brigades->new($brignum,'brigade_parent')) { ## Brigade Exists
			if (my $current_comm = $brigade->Commander) { ## Remove The Current Commander
				$current_comm->Update( unsetasbrigadeco => $brignum  ) 
					unless ($self->caID == $current_comm->caID)
			}
			
			if ($self->IsInBrigade($brigade->parent) == 4) { ## If XO In Parent
				return "brigade='$brignum',brigade_pos=5"
			} else { "brigade='$brignum',brigade_pos=3" }
		} else { return undef }
	}

	sub _wrap_unsetasbrigadeco {
		my ($self,$brigade) = @_;
		return undef unless $brigade;
		
		my $pos = $self->IsInBrigade($brigade);
		if ($pos && ($pos == 5)) { 
			my $position = ($self->brigade_pos == 5 ? 2 : 1) or return undef;
	
			if ($self->brigade_parent == 0) { return "brigade=1,brigade_pos=$position" }
			else { return "brigade='".$self->brigade_parent."',brigade_pos=$position" }	
		
		} else { return undef }
	}
	
	sub _wrap_leavebrigade { return 'brigade=NULL,brigade_pos=NULL' }

1337;
