#!perl -wW
package CyberArmy::WWW::Wiki;

use strict;

use Digest::MD5 ();
use Algorithm::Diff ();

use CyberArmy::Wiki;
use CyberArmy::Template;
use CyberArmy::WWW::Request;

$CyberArmy::WWW::Wiki::VERSION = '0.3';

my ($wiki,$location);

sub handler {
	my $r = CyberArmy::WWW::Request->instance(shift);
	$location = $r->location;
	$wiki = CyberArmy::Wiki->new(
		$r->subprocess_env('Wiki')
	) or return 404;
	
	my $path = $r->uri;
	CyberArmy::WWW::Utils::escapeHtml($path);
	my $node = ($path =~ /$location\/(\w+)/) ?  $1 : $wiki->{'frontnode'};

	my $params = $r->getParams({escapehtml => 1, multi => { diff => 1 }});
	$params->{'action'} ||= 'view';
	if($wiki->{'access_group'})
	{
		my $user = CyberArmy::WWW::Request::User->instance();
		if(!$user || ((!$user->CheckGroupList($wiki->{'access_group'})) && 
			(!$user->CheckGroupList($wiki->{'edit_group'})) && (!$user->CheckGroupList($wiki->{'admin_group'}))))
		{
			return 403;
		}
	}
	if ($params->{'action'} eq 'view') { 
		&viewNode($node,$params->{'revision'})
	} elsif ($params->{'action'} eq 'related') { 
		&viewRelated($node)
	} elsif ($params->{'action'} eq 'edit') { 
		&editNode($node,$params->{'revision'})
	} elsif ($params->{'action'} eq 'history') { 
		&viewNodeHistory($node,$params->{'diff'});
	} elsif ($params->{'action'} eq 'search') {
		&searchNodes($params->{'search'}); 
	} elsif ($params->{'action'} eq 'changes') { 
		&viewChanges($params->{'period'}); 
	} else { return 404 }
	
	return 0;
}

sub _info {
	my %info =();
	$info{'frontnode'} = $wiki->{frontnode};
	$info{'location'} = $location;
	$info{'version'} = $CyberArmy::Wiki::VERSION;
	$info{'title'} = $wiki->{title};
	return \%info;
}

sub viewNode {
	my ($name,$revision) = @_;
	my %template = ();

	my $node = $wiki->getNode($name,$revision);
	if ($node->{'name'}) {
		my $body = CyberArmy::Wiki::Format::Html(
			$node->{'content'},$wiki->getLinks('from',$name));
		my $user = CyberArmy::WWW::Request::User->instance();
		if ($user && ($wiki->{'edit_group'}) && ($user->CheckGroupList($wiki->{'edit_group'}))) {
		$template{'can_edit'} = 'true';
		}
		$template{'body'} = $body;
		$template{'created'} = $node->{created};
		$template{'author'} = $node->{author};
		$template{'info'} = _info();
		$template{'wikititle'} = $wiki->{'title'}.': '.$node->{'name'};
		my $r = CyberArmy::WWW::Request->instance();
		$r->content_type('text/html');
		$r->printTemplate('wiki/view.tmpl',{ %template });
	} else { &editNode($name) }
}

sub viewRelated {
	my ($name) = @_;
	my %template = ();

	my $node = $wiki->getNode($name);
	if ($node->{'name'}) {
		$template{'name'} = $name;
		my @related;
		foreach (sort(keys %{$wiki->getLinks('to',$name)})) {
			push @related,$_;
		}
		$template{'related'} = \@related;
		$template{'wikititle'} = 'Related to '.$node->{'name'};
		$template{'info'} = _info();
                my $r = CyberArmy::WWW::Request->instance();
                $r->content_type('text/html');
                $r->printTemplate('wiki/viewrelated.tmpl',{ %template });
	} else { exit(404) }
}

sub searchNodes {
	my ($search) = @_;
	my %template = ();

	$template{'search'} = $search;
	$template{'info'} = _info();
	$template{'wikititle'} = 'Search';

	if ($search && length $search >= 4) {
		
		my $results = $wiki->searchNodes($search);
		$template{'results'} = $results;
	}
        my $r = CyberArmy::WWW::Request->instance();
        $r->content_type('text/html');
        $r->printTemplate('wiki/search.tmpl',{ %template });
}

sub editNode {
	my ($name,$revision) = @_;
	my $r = CyberArmy::WWW::Request->instance;
	my $user = CyberArmy::WWW::Request::User->instance();
	my %template = ();	
	
	if ($r->method eq 'POST') {
 		$user or exit(403);
		if ($wiki->{'edit_group'}) {
			exit(403) unless $user->CheckGroupList($wiki->{'edit_group'});
		}
		my $r = CyberArmy::WWW::Request->instance;
		my $posted = $r->getParams({from=>'posted', escapehtml => 1});
		
		if ($posted->{'action'} eq 'Save') {
			$posted->{'content'} =~ s/\s{8}/\t/g if $posted->{'space2tab'};
			
			my $node = $wiki->getNode($name);
			
			if (Digest::MD5::md5_hex($node->{'content'} || '') ## content
					ne Digest::MD5::md5_hex($posted->{'content'})) { ## modified
				$wiki->setNode($name,$user->nickname,
					\$posted->{'content'},$posted->{'log'}) || exit(500);
			}
			
			$r->redirectTo($location.'/'.$name);
		} elsif ($posted->{'action'} eq 'Delete') {
			
			$user->CheckGroupList($wiki->{'admin_group'}) or exit(403);
			$wiki->deleteNode($name);
			$r->redirectTo($location.'/'.$wiki->{'frontnode'});
			
		} elsif ($posted->{'action'} eq 'Preview') {
			$posted->{'content'} || exit(412);
			
			my $body = $posted->{'content'};
			$body=~ s/\s{8}/\t/g if $posted->{'space2tab'};
			$body = CyberArmy::Wiki::Format::Html(
				$body,$wiki->getLinks($name));
			
			$template{'body'} = $body;
			$template{'content'} = $posted->{'content'};
			$template{'space2tab'} = $posted->{space2tab} ? $posted->{space2tab}:'';
			$template{'log'} = $posted->{'log'};
			$template{'wikititle'} = $wiki->{'title'}.': '.$name;
			$template{'info'} = _info();
        	        my $r = CyberArmy::WWW::Request->instance();
	                $r->content_type('text/html');
                	$r->printTemplate('wiki/preview.tmpl',{ %template });
		} else { exit(404) }

	} else {
		my $node = $wiki->getNode($name,$revision);
		$template{'node_content'} = $node->{'content'} ? $node->{'content'}:'';
		$template{'node_created'} = $node->{'created'};
		$template{'crev'} = $revision ? $revision : undef;
		$template{'name'} = $name;
		$template{'revisions'} = \@{$wiki->getNodeRevisionsList($name,1)};

		my $delete = ($wiki->{'admin_group'} && $user)
			? (($user->CheckGroupList($wiki->{'admin_group'}) 
			? 'true' : undef)) : undef;
		$template{'delete'} = $delete;
		$template{'title'} = 'Edit '.$name;
		$template{'info'} = _info();
	        my $r = CyberArmy::WWW::Request->instance();
        	$r->content_type('text/html');
        	$r->printTemplate('wiki/editnode.tmpl',{ %template });
	}
}

sub viewChanges {
	my ($time) = @_;
	$time ||= 7;
	my %template = ();
	$template{'time'} = $time;
	$template{'logs'} = \@{$wiki->getNodeRevisionsList($time)};
	$template{'wikititle'} =  'Recent Changes';
	$template{'info'} = _info();
	my $r = CyberArmy::WWW::Request->instance();
	$r->content_type('text/html');
	$r->printTemplate('wiki/changelog.tmpl',{ %template });
}

sub viewNodeHistory {
	my ($name,$diff) = @_;
	my %template = ();
	if ($diff) {
		if (scalar(@$diff) == 2) {
			my @array;
			@$diff = sort @$diff;
			my $one = $wiki->getNode($name,shift@$diff);
			my $two = $wiki->getNode($name,shift@$diff);
			
			my $diff = Algorithm::Diff::diff(
				[split"\n",$one->{'content'}],
				[split"\n",$two->{'content'}]);
				
			my %colors = ( +'-' => '#FF0000', +'+' => '#66CC99' );
 			$template{'name'} = ${name};
			$template{'colors_-'} = $colors{'-'};
			$template{'colors_+'} = $colors{'+'};
			$template{'one_created'} = $one->{created};
			$template{'two_created'} = $two->{created};

			foreach my $line (@$diff) {
				foreach (@$line) {
					push @array,({
						bgcolor => $colors{$_->[0]},
						line => "#$_->[1]",
						diff => $_->[2]});
				}
			}
			$template{'diff'} = \@array;
		        $template{'wikititle'} = 'Diff '.$name;
		        $template{'info'} = _info();
	        	my $r = CyberArmy::WWW::Request->instance();
        		$r->content_type('text/html');
		        $r->printTemplate('wiki/history.tmpl',{ %template });
		} else { exit(412) }
	} else {
		my @array;
		$template{'name'} = ${name};
		$template{'revisions'} = \@{$wiki->getNodeRevisionsList($name)};
		$template{'wikititle'} = $name."'s history";
		$template{'info'} = _info();
	        my $r = CyberArmy::WWW::Request->instance();
        	$r->content_type('text/html');
	        $r->printTemplate('wiki/history.tmpl',{ %template });
	}
	
}

package CyberArmy::Wiki::Format;

# using qwiki's formatters
#
#    Permission to copy, use, modify, sell and distribute this software
#    is granted provided this copyright notice appears in all copies.
#    Copyright (C) 2000 Bill Kelly / Full Body Groove.  All Rights Reserved.
#    This software is released in the hope it will be useful, but please
#    understand you use it at your own risk.
#    THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
#    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
#    OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#

sub Html {
    my ($plain,$links) = @_;

    my %listTypeOpen =  ( '*' => "<ul>",  '1.' => "<ol>" );
    my %listTypeClose = ( '*' => "</ul>", '1.' => "</ol>" ); 
    my %paraTypeOpen =  ( 'p' => "<p>",  'pre' => "<pre>" );
    my %paraTypeClose = ( 'p' => "</p>", 'pre' => "</pre>" );
    my $GT = '(?:<[^>]*>)*';  # "gobble tags"

    my $line = "";
    my $linePlain = "";
    my $html = "";
    my $prevTextInterrupted = 0;
    my $blankCnt = 0;
    my @blankBuf = ();
    my $blankHasDiff = 0;
    my $inPara = "";
    my @listLvl = ();

    my $_isBlank = sub {
        return $linePlain =~ /^\s*$/s;
    };
    my $_isDivider = sub {
        return $linePlain =~ /^----/;
    };
    my $_isMono = sub {
        return $linePlain =~ /^[ ]+/;
    };
    my $_isItem = sub {
        return $linePlain =~ /^(\t+)([*]|1\.)/s;
    };
    my $_closePara = sub {
        return unless ($inPara);
        $prevTextInterrupted = 1;
        $html .= $paraTypeClose{$inPara}."\n";
        $inPara = "";
    };
    my $_openPara = sub {
        my ($pType) = @_;
        return "" if ($inPara eq "$pType");
        &$_closePara;
        $prevTextInterrupted = 1;
        $html .= $paraTypeOpen{$pType}."\n";
        return $inPara = $pType;
    };
    my $_goListLevel = sub {
        my ($tabs, $listType) = @_;
        my $itemLvl = length($tabs);
 # print STDERR ("_goListLevel: ($itemLvl) (", defined($listType)? $listType: "undef", ") (@listLvl)\n");
        while (scalar(@listLvl) > $itemLvl  ||
               ($itemLvl > 0 && (scalar(@listLvl) == $itemLvl) &&
                ($listLvl[$itemLvl-1] ne $listType))) {
            $prevTextInterrupted = 1;
            $html .= $listTypeClose{ pop(@listLvl) };
        }
        while (scalar(@listLvl) < $itemLvl) {
            push(@listLvl, $listType);
            $prevTextInterrupted = 1;
            $html .= $listTypeOpen{ $listType };
        }
    };

    my $prevWasText = 0;
    while ($plain =~ /.*?(\n|$)/gs) {
        $linePlain = $line = $&;
        $linePlain =~ s/($GT)//gso;
	##WIKI NAMED ANCHOR PATCH: (damn the other was bbcode only!!)
	##To make this work 'x6ANCHOR'x6 (no spaces between ' and anchor)
	##Icydemon => Noob);
	$line =~ s{(&#39;){6,6}(.*?)(&#39;){6,6}}{<a name="$2"></a>}gs;
        $line =~ s{(&#39;){5,5}(.*?)(&#39;){5,5}}{<b><i>$2</i></b>}gs;
        $line =~ s{(&#39;){3,3}(.*?)(&#39;){3,3}}{<b>$2</b>}gs;
        $line =~ s{(&#39;){2,2}(.*?)(&#39;){2,2}}{<i>$2</i>}gs;
 
        if (&$_isBlank) {
            $blankCnt++;
            if ($line =~ /<!--qwdiff-->/s) {
                my $sp = ($prevWasText)? "" : "&nbsp;";
                $line =~ s/<!--qwdiff_end-->.*?$/$sp<br>$&/s;
                $blankHasDiff = 1;
                push(@blankBuf, $line);
            }
            else {
                push(@blankBuf, "<br>\n");
            }
            $line = "";
            $prevWasText = 0;
        }
        else {
            my $lineWasText = 0;
            if (&$_isItem) {
                &$_closePara;
                $prevTextInterrupted = 1;
                $blankCnt -= 1;  # added because allowing linefeed seemed nicer here in EditText
              # $line =~ s/^(\t+)([*]|1\.)/&$_goListLevel($1, $2); "<li>"/e;
                $line =~ s/^($GT)(\t+)([*]|1\.)/&$_goListLevel($2, $3); "$1<li>"/eo;
            }
            else {
                &$_goListLevel("");
                if (&$_isDivider) {
                    $blankCnt++ unless ($prevTextInterrupted);
                    $prevTextInterrupted = 1;  # (hr has implicit br)
                 #   my $hr = ($line =~ /<!--qwdiff-->/s)?
                 #               "&nbsp;<hr>&nbsp;\n" : "<hr>\n";
                    $line =~ s/-{4,}/<hr>/s;
                    my $tags = "";
                    while ($line =~ /($GT)/gso) {
                        $tags .= $1;
                    }
                    $line = $tags;
                }
                else {
                    my $newPara = &$_openPara( (&$_isMono)? 'pre': 'p');
                    $blankCnt-- if ($newPara);  # new para has implicit br
                    $lineWasText = 1;
                }
            }
            my $numBr = (($blankCnt < 1) || $prevTextInterrupted)?
                                                $blankCnt : ($blankCnt + 1);
            if ($blankHasDiff) {
                $html .=  join('', @blankBuf);
            }
            else {
                $html .= ("<br>\n" x $numBr) if ($numBr > 0);
            }
            $blankCnt = 0;
            @blankBuf = ();
            $blankHasDiff = 0;
            $prevTextInterrupted = 0 if ($lineWasText);
            $prevWasText = $lineWasText;
        }

        markupWikiLinks(\$line,$links);
        stripWikiMetaCharsInHtml(\$line);
        $html .= $line;
    }
    &$_goListLevel("");
    &$_closePara;
    $html .= ("<br>\n" x ($blankCnt-1)) if (($blankCnt-1) > 0);

    return $html;
}

sub markupWikiLinks {
    my ($pageTextRef,$links) = @_;

    {
        local($^W) = 0;  # warnings off temporarily - the undef values are intentionally used below
        # \x60 below is back-tick (`)
        my $sub;
        $$pageTextRef =~ s{\b(mailto|https|http|ftp|wiki):}{lc($&)}gsei;
        #added wiki:page[alias name]  option, to display the alias rather than just the page name
        $$pageTextRef =~ s` (?: \b((?:[A-Z][a-z]*){2,})\b ) |
                            ( \b mailto:[^\s]+?[@][^\s)'\x60<\[]+\[.*?\] ) |
                            ( \b mailto:[^\s]+?[@][^\s)'\x60<]+ ) |
                            ( \b https:[^\s'\x60<\[]+\[.*?\] ) |
                            ( \b https:[^\s'\x60<]+ ) |
                            ( \b http:[^\s'\x60<\[]+\[.*?\] ) |
                            ( \b http:[^\s'\x60<]+ ) |
                            ( \b ftp:[^\s'\x60<\[]+\[.*?\] ) |
                            ( \b ftp:[^\s'\x60<]+ ) |
                            ( \b wiki:[^\s\[]+\[.*?\] ) |
                            ( \b wiki:[^\s]+ )
                          ` ($1)? wikiLink($1,$links): markupUrl("$2$3$4$5$6$7$8$9$10$11");
                          `gsex;
    }
}

sub markupUrl {
    my ($url) = @_;
    if ($url =~ m{^http:.*\.(gif|jpg|jpeg|png)$}i) {
      # $url =~ s{^http:([^/])}{$1};  # strip leading http: for relative urls
        $url =~ s{^http:([^/])}{$location/$1};
        return qq{<img src="$url">};
    }    
    elsif ($url =~ m{^((http|https|mailto|ftp):([^[]*))(\[(.*?)])$}i) {
        my $pageName = $1;
        my $alias = $5;
        $url =~ s{^http:([^/])}{$location/$1};
      	$pageName = $alias if ($alias);
      	return qq{<a href="$1">$pageName</a>};
        	
    }
    elsif ($url =~ m{^wiki:(.*?)(\[(.*?)\]|)$}i) {
        my $pageName = $1;
        my $alias = $3;
        $url = "$location/$pageName";
      	$pageName = $alias if ($alias);
        return qq{<a href="$url">$pageName</a>};
    }
    else {
        my $dispUrl = $url;
      # $url =~ s{^http:([^/])}{$1};  # strip leading http: for relative urls
        $url =~ s{^http:([^/])}{$location/$1};
		return qq{<a href="$url">$dispUrl</a>}; # break out of frames
    }
}

sub stripWikiMetaCharsInHtml {
    my ($textRef) = @_;
    
    {
        local($^W) = 0;  # warnings off temporarily - the undef values are intentionally used below
        my ($text, $tag);
        # Don't modify content between html < tags >.
        $$textRef =~ s{ ([^<]*) (<[^>]*> | $ ) }
                      { ($text, $tag) = ($1, $2);
                        stripWikiMetaChars(\$text);
                        "$text$tag";
                      }gsex;
    }
}

sub wikiLink {
    my ($name,$links) = @_;

    my $strippedName = $name;
    stripWikiMetaChars(\$strippedName);
    if (exists($links->{$strippedName})) {
        # If name exists in database, mark it up (even if non-strict WikiName)
		return "<a href=\"$location/$strippedName\">$name</a>";
    }
    else {
        # Only automatically provide edit links for strict WikiNames.
        # Names like CLanguage or BASIC must be created manually (via
        # the edit URL), but will then be marked up once they exist
        # in the database.
        return (validStrictWikiName($strippedName))?
			"$name<sup><a href=\"$location/$strippedName?action=edit\">?</a></sup>" : $name;
    }
}

sub stripWikiMetaChars {
    my ($textRef) = @_;
    $$textRef =~ s/`(`*)/$1/gse;  # gobble one back-tick (WikiName`s handler)
}

sub validStrictWikiName {
    my ($name) = @_;
    return $name =~ /^(?:[A-Z][a-z]+){2,}$/s;  # (strict AbCd names)
}

1;
