#!/usr/bin/perl -wW
package CyberArmy::WWW::Services;

use strict;
use CyberArmy::Message;
use CyberArmy::Mail;
use CyberArmy::User;
use CyberArmy::Utils;
use CyberArmy::Banner;
use CyberArmy::Remote::Feed;
use CyberArmy::Template;
use CyberArmy::Services::Mail;
use CyberArmy::WWW::Request;

$CyberArmy::WWW::Services::VERSION = '0.2';

sub handler {
	my $r = CyberArmy::WWW::Request->instance(shift);
	
	my $path = $r->path_info;
	CyberArmy::WWW::Utils::escapeHtml($path);
	my (undef,$node) = split /\// , $path;
	
	if ($node eq 'banners') { &handleBanners }
	elsif ($node eq 'mail') { &handleMail }
	elsif ($node eq 'feeds') { &handleFeeds}
	else { $r->printTemplate('services/services.tmpl') }
}

sub handleBanners {
	my $r = CyberArmy::WWW::Request->instance;
	my $params = $r->getParams({ escapehtml => 1 });
	$params->{'action'} ||='';
	
	if ($params->{'action'} eq 'statistics') {
		$params->{'year'} ||= '';
		$params->{'month'} ||= '';
		$params->{'day'} ||= '';
		
		my $stats = CyberArmy::Banner->getStatistics($params);
		$r->printTemplate(
			'services/banners/statistics.tmpl',
			{ params => $params, stats => $stats }
		);
	} elsif ($params->{'action'} eq 'admin') {
		$r->printTemplate(
			'services/banners/admin.tmpl',
			{ banners => CyberArmy::Banner->list() }
		);
	} elsif ($params->{'action'} eq 'edit') {
		if ($r->method eq 'POST') {
			my $posted = $r->getParams({from=>'posted',escapehtml => 1});
			
			if ($posted->{'action'} eq 'delete') {
				my $banner = CyberArmy::Banner
					->new({id => $posted->{'id'}}) or exit 404;
				$banner->remove;
				$r->redirectTo('/services/banners/?action=admin');
				
			} elsif($posted->{'action'} eq 'update')  {
				my @images; push(@images, $_) foreach(split/[\n\r]+/,$posted->{'images'});
				
				my $id = CyberArmy::Banner->add({ id => $posted->{'id'},
					caption => $posted->{'caption'}, url => $posted->{'url'},
					priority => $posted->{'priority'}, images => \@images,
					credits => $posted->{'credits'}
				}) || '';
				
				$r->redirectTo('/services/banners/?action=edit&banner='.$id);
			} else { exit 412 }
		} else {
			my $banner = ! $params->{'banner'} ? { } :
				CyberArmy::Banner->new({id =>  $params->{'banner'}});
			$r->printTemplate('services/banners/edit.tmpl',{ banner => $banner });
		}
	} else { $r->printTemplate('services/banners/menu.tmpl'); }
}

sub handleMail {
	my $r = CyberArmy::WWW::Request->instance;
	my $params = $r->getParams({ escapehtml => 1 });
	$params->{'action'} ||='';

	if ($params->{'id'}) {
		my $mail = CyberArmy::Services::Mail->new($params->{'id'});
		
		my $info = $mail->isSetup();
		exit(403) if $info and $info->{'system'} eq 'Y';
		
		if ($r->method eq 'POST') {
			my $posted = $r->getParams({from=>'posted', escapehtml => 1});
			$posted->{'action'} ||= 'update';

			if ($posted->{'action'} eq 'update') {
				$posted->{'quota'} = $posted->{'custom_quota'} ?
					$posted->{'custom_quota'} : ($posted->{'quota'} || 0);
	
				if ($posted->{'expiry'}) {
					if ($posted->{'expiry'} eq 'add_year') {
						$posted->{'expiry'} = '+1 YEAR';
					} elsif ($posted->{'expiry'} eq 'remove_year') {
						$posted->{'expiry'} = '-1 YEAR';
					} elsif ($posted->{'expiry'} eq 'add_month') {
						$posted->{'expiry'} = '+1 MONTH';
					} elsif ($posted->{'expiry'} eq 'remove_month') {
						$posted->{'expiry'} = '-1 MONTH';
					} else {$posted->{'expiry'} = 0 }
				} else { undef $posted->{'expiry'} }
	
				my $password = $posted->{'password'} ? ## reset the password
					CyberArmy::Utils::RandomAlphaNum(8) : '';
				
				my ($mail_user,$mail_caid); $mail_user = CyberArmy::User
					->new(nickname => $posted->{'user'}) if $posted->{'user'};
				$mail_caid = $mail_user->caID if $mail_user;
				
				$mail->setAsMailbox({
					caid => $mail_caid,
					quota => $posted->{'quota'},
					limit => $posted->{'expiry'},
					password => $password
				});
				
				$mail->setAsForward($info->{'forward'}) if $info->{'forward'};
				
				if ($posted->{'password'} && $mail_user) {
					my $user = CyberArmy::WWW::Request::User->instance;
					CyberArmy::Template->instance->process(
						'services/mail/reset_message.tmpl',
						{ email => $mail->{'id'}, 
							password => $password }, \my $body );

					CyberArmy::Message->new(
						caID => [$mail_user->caID], body => $body,
						user => $user, subject => 'Mail password reset'
					);	
				}
				
				unless ($info) {
					CyberArmy::Template->instance->process(
						'services/mail/welcome_email.tmpl',
						{ email => $mail->{'id'} }, \my $body );
					
					CyberArmy::Mail::Send(
						from => 'postmaster',subject => 'ping!',
						to => [$mail->{'id'}],body => $body,
					) or exit (500);
				}

			} else { $mail->purge() }
			
			$mail->{'id'} =~ s/@/%40/;
			$r->redirectTo($r->uri . '?id=' . $mail->{'id'});

		} else {
			my $user = CyberArmy::User->new(caID => $info->{'caid'});
			$r->printTemplate(
				'services/mail/admin.tmpl',
				{ mail => $mail, user => $user });
		}
	} else { $r->printTemplate('services/mail/menu.tmpl'); }
}

####
#sub handleFeeds
#
#Used to view, add, edit and delete feeds.
##
sub handleFeeds{
	my $r = CyberArmy::WWW::Request->instance;
   my $params = $r->getParams({from => 'posted', escapehtml => 1});
	$params->{'action'} ||='';
	
	if ($params->{'action'} eq 'create') {
			#details for new feed added
			if(($params->{'name'} && $params->{'name'} ne '') 
				&& ($params->{'url'} && $params->{'url'} ne '') 
				&& ($params->{'type'} && $params->{'type'} ne '') 
				&& ($params->{'remote'} && $params->{'remote'} ne '') ){
				my $remote = $params->{'remote'} eq 'Y' ? 'Y' : 'N';
				
				CyberArmy::Remote::Feed->add($params->{'name'}, $params->{'type'}, $params->{'url'}, $remote);
				
			}
			else 
			{
				#something was missing, indicate error
				
				#get feeds
				my $feeds = CyberArmy::Remote::Feed::List->getFeedList();
				$r->printTemplate('services/feeds/list.tmpl',
					{   feeds => $feeds,
						name=> $params->{'name'} ? $params->{'name'} : undef, 
						type=> $params->{'type'} ? $params->{'type'} : undef,
						url=> $params->{'url'} ? $params->{'url'} : undef,
						remote=> $params->{'remote'} ? $params->{'remote'} : undef,
						paramsMissing => $params->{'submit'} ? 1 : undef
					});
				return;
			}
	} elsif ($params->{'action'} eq 'edit') {
		# send to edit page
		if($params->{'id'} && $params->{'id'} ne '') 
		{
				my $feed = CyberArmy::Remote::Feed::List->getSpecificFeedList($params->{'id'});
				$r->printTemplate('services/feeds/edit.tmpl',
					{	id=> $feed->{id},
						name=> $feed->{name}, 
						type=> $feed->{type},
						url=> $feed->{url},
						remote=> $feed->{remote}
					});
					return;
		}
		
	} elsif ($params->{'action'} eq 'update') {
		#actually do an update/edit
		if(($params->{'id'} && $params->{'id'} ne '')
		&& ($params->{'name'} && $params->{'name'} ne '') 
				&& ($params->{'url'} && $params->{'url'} ne '') 
				&& ($params->{'type'} && $params->{'type'} ne '') 
				&& ($params->{'remote'} && $params->{'remote'} ne '') )
		{
				my $remote = $params->{'remote'} eq 'Y' ? 'Y' : 'N';
				CyberArmy::Remote::Feed->update($params->{'id'}, $params->{'name'}, $params->{'type'}, $params->{'url'}, $remote);
				
		}
		else 
		{
			#something was missing, indicate error				
			#get feeds
			if(($params->{'id'} && $params->{'id'} ne ''))
			{
				my $feeds = CyberArmy::Remote::Feed::List->getFeedList();
				$r->printTemplate('services/feeds/edit.tmpl',
					{   feeds => $feeds,
						id=> $params->{'id'}, 
						name=> $params->{'name'} ? $params->{'name'} : undef, 
						type=> $params->{'type'} ? $params->{'type'} : undef,
						url=> $params->{'url'} ? $params->{'url'} : undef,
						remote=> $params->{'remote'} ? $params->{'remote'} : undef,
						paramsMissing => $params->{'submit'} ? 1 : undef
					});
				return;
			}
		}
		
	} elsif ($params->{'action'} eq 'delete') {
		if($params->{'id'} && $params->{'id'} ne '')
		{
			CyberArmy::Remote::Feed->delete($params->{'id'});
		}
	}

	#get feeds
	my $feeds = CyberArmy::Remote::Feed::List->getFeedList();
	$r->printTemplate( 'services/feeds/list.tmpl', { feeds => $feeds } );
}

1;
