#!perl -wW
package CyberArmy::WWW::Request;

#
# Apache wrapper + goodies
#

$CyberArmy::WWW::Request::VERSION = '0.5.5';

use strict;
use Apache ();
use Apache::Singleton::Request ();
use base qw( Apache::Singleton::Request Apache );
use Encode ();

use CyberArmy::Template;

sub new {
	bless( Apache->request($_[1]), $_[0]);
}

sub _new_instance { &new }

sub printTemplate {
	my $tt2 = CyberArmy::Template->instance;
	
	if ($tt2->process($_[1],$_[2],\(my $html_out))) {
		$_[0]->content_type 
			or $_[0]->content_type('text/html');
		$_[0]->send_http_header();
		return if $_[0]->header_only;
		$_[0]->print($html_out);
	} else {
		$_[0]->log_error('[Template] '.$tt2->error);
		exit(500);
	}
}

sub errorTemplate {
	my ($code,$template,$args) = ('','',{});

	if ($_[3]) {
		$code = $_[1]; $template = $_[2]; $args = $_[3];
	} elsif ($_[2]) {
			if (ref $_[2]) { $template = $_[1]; $args = $_[2]; }
			else { $code = $_[1]; $template = $_[2] }
	}	else { 
		if ($_[1] =~ /^\d+$/) { $code=$_[1] }
		else { $template=$_[1] }
	}
	if ($code) {
		$_[0]->pnotes(error_tmpl => {template => $template, args => $args} );
		$_[0]->custom_response($code, '/error/');
	} else { $_[0]->printTemplate($template,$args) }

	$_[0]->exit($code or -1);
}


sub multipleChoices {
	my $r = shift;

	my $uri = shift; foreach (@$uri) {
		## note that specifying a location header
		## might trigger automatic redirection
		$r->err_headers_out->add( 'Location' => $_ );
	}

	my $html_out; CyberArmy::Template->instance
		->process(shift, shift, \($html_out))
			&& $r->custom_response(300, $html_out);

	$r->exit( 300 );
}

sub redirectTo {
	my $r = shift;

	$r->err_headers_out->add(
		'Location' => (shift() or '/'));

	# From RFC 2616: 10.3.4 303 See Other
	#   The response to the request can be found under a different URI and
	#   SHOULD be retrieved using a GET method on that resource. This method
	#   exists primarily to allow the output of a POST-activated script to
	#   redirect the user agent to a selected resource. The new URI is not a
	#   substitute reference for the originally requested resource. The 303
	#   response MUST NOT be cached, but the response to the second
	#   (redirected) request might be cacheable.

	$r->exit( 
		($r->method eq 'POST' 
			&& $r->protocol eq 'HTTP/1.1') ? 303 : 302
	);
}

sub getParams {
	## get params from different http methods and cookies
	
	$_[1]->{'from'} ||= 'args';
	
	if (my $params = $_[0]->pnotes($_[1]->{'from'})) {
		return $params;
	}
	
	my %params;

	if ($_[1]->{'from'} eq 'args') {
		(my $args = $_[0]->args) or return \%params;
		foreach (split /&/ , Apache::Util::unescape_uri_info($args)) {
			my ($name,$value) = split /=/, $_ , 2;
			CyberArmy::WWW::Utils::escapeHtml($name,$value) 
				if $_[1]->{'escapehtml'};

			if ($_[1]->{'multi'}->{$name}) { 
				push @{$params{$name}} , $value 
			} else { $params{$name} = $value }
    	}
	} elsif ($_[1]->{'from'} eq 'posted' && ($_[0]->method eq 'POST')) {
		foreach (split /&/, (my $args = $_[0]->content)) {
			my ($name,$value) = split /=/ , 
				Apache::Util::unescape_uri_info($_) , 2;
			CyberArmy::WWW::Utils::escapeHtml($name,$value) 
				if $_[1]->{'escapehtml'};

			if ($_[1]->{'multi'}->{$name}) { 
				push @{$params{$name}}, $value 
			} else { $params{$name} = $value }
		}
	} elsif ( $_[1]->{'from'} eq 'cookies' 
		&& (my $cookies = $_[0]->header_in('Cookie'))) {
		%params = map { 
			CyberArmy::WWW::Utils::escapeHtml($_)
				if $_[1]->{'escapehtml'}; split /=/, $_ , 2 
		} (split /; /, $cookies);
	}
	
	$_[0]->pnotes($_[1]->{'from'} => \%params);

	return \%params;
}

sub escapeHtml { 
  ## Escape HTML entities, using Apache::Util::escape_html
  foreach (@_) { 
    if (my $ref = ref) {
      foreach (@$_) { $_ = HTML::Entities::encode_entities(HTML::Entities::decode_entities(Encode::decode_utf8($_))) }
    } else { $_ = HTML::Entities::encode_entities(HTML::Entities::decode_entities(Encode::decode_utf8($_))) }
  }
}

sub checkReferer {
	# hack for chawmp's multihomed setup
	return 1 if ($ENV{NO_REFERER_CHECK});

	my $self = shift;
	
	(my $referer = $self->header_in('Referer'))
		or return undef; ## no referer, don't bother checking

	my $name = $self->get_server_name;
	foreach ($#_ >= 0 ? @_ : '/') {
		$referer =~ m/^http(?:s)*:\/\/$name(?::\d+)*$_/ 
			or return undef;
	}

	1;
}

sub checkSSLMode {
	## WARNING
	## this is *NOT* portable under all ENVs
	## it doesn't work with apache13+mod_ssl
	$_[0]->header_in('X-SSL-cipher') ? 1 : 0;	
}

sub getServerLink {
	my ($host,$port) = split(/:/,$_[0]->header_in('Host'));
	if ($_[0]->checkSSLMode()) {
		$port ||= 443;
		$host = 'https://'.$host;
		$host = $host.':'.$port 
			unless ($port == 443)
	} else {
		$port ||= 80;
		$host = 'http://'.$host;
		$host = $host.':'.$port 
			unless ($port == 80)
	}

	return $host;
}

package CyberArmy::WWW::Utils;

use HTML::Entities ();
use Encode ();

sub escapeHtml { 
  ## Escape HTML entities, using Apache::Util::escape_html
  foreach (@_) { 
    if (my $ref = ref) {
      foreach (@$_) { $_ = HTML::Entities::encode_entities(Encode::decode_utf8($_)) }
    } else { $_ = HTML::Entities::encode_entities(Encode::decode_utf8($_)) }
  }
}


sub encodeHtml {
	my $encoded;
    foreach (split //, $_[0]) {
		$encoded .= '&#'.ord().';'
	}
	return $encoded;
}

package CyberArmy::WWW::Request::User;

use CyberArmy::User;
use Apache::Singleton::Request;
use base qw(Apache::Singleton::Request CyberArmy::User);

sub _new_instance {
	my $r = CyberArmy::WWW::Request->instance;
	
	my $auto_select = $r->dir_config('dinah_auto_select');
	my $select = $r->dir_config('SelectFields');
	$select = ! $select ? $auto_select: $auto_select.','.$select;

	my $user;
	
	$r->auth_name("CyberArmy"); ## required by get_basic_auth_pw()
	#$r->auth_name("Basic");
	my ($basic, $pass) = $r->get_basic_auth_pw();
	my $cookies = $r->getParams({from=>'cookies'});

	if ($basic == 0) { ## means we got a valid basic auth request
		## here we basically let the user(agent) initiate any basic 
		## authentification dialog, instead of forcing him with a 401
		$user = __PACKAGE__->new(
			nickname => $r->connection->user(), select => $select
		);

		## the user/pass isn't valid, we will spit out a standard 401
		## we need to make a difference between failed auth and failed
		## access, especially for the logging output, in this case
		$r->exit(401) unless defined($user) && $user->Passhash($pass);

		## also, we need to make sure he isn't banned
		$r->exit(403) if $user->IsBanned(); ## notice the 403

		## even outside the realm of a session, we need to monitor IPs
		$user->LogIP($r->connection->remote_ip());

	} elsif ($cookies->{'session'}) { ## check for a regular session
		$user = __PACKAGE__->new(
			session_id => $cookies->{'session'}, select => $select
		) or do {
			## the client had a session cookie, but it doesn't look valid
			## we remove it, so subsequent requests don't have to process it
			my $cookies = $r->pnotes('cookies');
			delete $cookies->{'session'};
			$r->pnotes(cookies => $cookies);

			## and instruct the browser to kill it, assuming it's a good boy
			$r->err_headers_out->add('Set-Cookie' => 'session'.
				'=expired; path=/; expires=Tue, 8-Sep-2002 13:37:33 GMT');
			return undef;
		};

		my $request_time = $r->request_time;

		if ($user->HasValidSessionTime($request_time)) {

			if ($user->session_mode eq 'secured') { ## safe session
				if (($r->connection->remote_ip ne $user->session_ip)) {
					$user->logOut();
					$r->errorTemplate(403,
						'errors/safe_session_error.tmpl',{user=>$user})
				}
			} else {
				## for non-safe sessions, we may need to log after every request
				$user->LogIP($r->connection->remote_ip);
			}
			
			if ($user->IsBanned()) { ## verify he isn't banned
				$user->logOut();
				$r->errorTemplate(403,'errors/banned.tmpl',{user=>$user});
			}

			$r->connection->user( $user->nickname );

			if ($user->session_ltime < 
					($request_time - 
						$r->dir_config('dinah_session_update'))) {
				$r->push_handlers('PerlCleanupHandler',
					sub { $user->Update( session_ltime => $request_time ) }
				);
			}

		} else {
			$user->logOut();
			$r->multipleChoices([],'errors/session_expired.tmpl',
				{ user => $user,  location => $r->uri });
			return undef;
		}

	} else { return undef }

	## the following is just being paranoid about caching proxies
	$r->header_out( 'Pragma' => 'no-cache' );

	return $user;
}

sub logIn {
	my $r = CyberArmy::WWW::Request->instance;

	my $cookies = $r->getParams({from=>'cookies'});
	## set the defaults
	if ($cookies->{'login_mode'}) {
		if ($cookies->{'login_mode'} ne 'secured'
			and $cookies->{'login_mode'} ne 'normal') {
			$cookies->{'login_mode'} = 'normal';
		}
	} else { $cookies->{'login_mode'} = 'secured'; }

	if ($cookies->{'login_timeout'}) {
		if ($cookies->{'login_timeout'} < 1) {
			$cookies->{'login_timeout'} = 1;
		} elsif ($cookies->{'login_timeout'} > 24) {
			$cookies->{'login_timeout'} = 24;
		}
	} else { $cookies->{'login_timeout'} = 1 }

	my $request_time = $r->request_time;	
	my $session_id = $request_time.CyberArmy::Utils::RandomAlphaNum(5);
	
	## activate the new session
	$_[1]->Update(
		session_id => $session_id,
		session_ip => $r->connection->remote_ip,
		session_timeout => $cookies->{'login_timeout'},
		session_mode => $cookies->{'login_mode'},
		session_time => $request_time,
		session_ltime => $request_time,
	) or exit (500);

	my $ssl = $r->checkSSLMode() ? 'secure' : '';

	$r->err_headers_out->add(
		'Set-Cookie' => 'session='.$session_id.'; path=/;'
		.' expires=Tue, 1-Jan-2010 13:37:33 GMT; '.$ssl
	);

	$ssl ?
		$r->err_headers_out->add(
			'Set-Cookie' => 'ssl_mode=yes; path=/;'
			.'expires=Tue, 1-Jan-2010 13:37:33 GMT'
		):
		$r->err_headers_out->add(
			'Set-Cookie' => 'ssl_mode=yes; path=/;'
			.'expires=Tue, 1-Jan-1970 13:37:33 GMT'
		);

	## non-safe sessions are logged after every request anyway, so no need to do so here
	$_[1]->LogIP($r->connection->remote_ip) if ($_[1]->session_mode eq 'secured');
}

sub logOut {
	$_[0]->Update(session_id=> undef );
	my $r = CyberArmy::WWW::Request->instance;
	
	$r->err_headers_out->add('Set-Cookie' => 'session'.
		'=logout; path=/; expires=Tue, 8-Sep-2002 13:37:33 GMT');

	my $cookies = $r->pnotes('cookies');
	delete $cookies->{'session'};
	$r->pnotes(cookies => $cookies);
}

package CyberArmy::WWW::Request::Access;

use CyberArmy::Database;

sub handler {
	my $r = CyberArmy::WWW::Request->instance( shift );

	if (my $db = CyberArmy::Database->instance()) {
		$db->ping or return 503;
	} else { return 503 }
	
	my $auth = $r->dir_config('Auth');
	return -1 if ($auth && $auth eq 'Off');

	my $cookies = $r->getParams({from=>'cookies',escapehtml=>1});
	## WARNING: In ssl_mode, port isn't taken into consideration
	if ($cookies->{'ssl_mode'} and not $r->checkSSLMode()) {
		my $qs = $r->query_string() || '';
		$qs = '?'.$qs if ($qs);
		$r->redirectTo('https://'.$r->get_server_name().$r->uri().$qs);
	}
	
	my $user = CyberArmy::WWW::Request::User->instance();

	if ($auth) {
		$r->redirectTo('/login/') unless $user;
		my @groups = $r->dir_config->get($auth);
			return 
			( ($auth eq 'Require' && !$user->IsInGroup(@groups))
				or ($auth eq 'Allow' && ! $user->IsInGroupList(@groups))
			) ? 403 : 0;
	} else { return 0 }
}
1;
