#!/usr/bin/perl -wW
package CyberArmy::WWW::Repository;

use strict;
use XML::RSS ();
use Digest::MD5 ();
use Apache::Util ();
use Apache::Request ();

use CyberArmy::Message;
use CyberArmy::User;
use CyberArmy::Template;
use CyberArmy::Repository;

$CyberArmy::WWW::Repository::VERSION = '0.3';

sub handler {
	return 503 unless ($ENV{'FILES_ROOT'});
	my $r = shift;
	my (undef,undef,@path) = split(/\//,$r->uri);
	CyberArmy::WWW::Utils::escapeHtml(@path);
	my $req = $path[0] || '';
	if ($req eq 'pub') {
		## requesting the actual file
		my $private = ((((stat($ENV{'FILES_ROOT'}.
			$path[-2]))[2]) or return 404 ) == 16872);
		## for "private" files, forbidden
		## unless it's an internal_redirect
		## else, just give the request back
		## to apache itself.
		return ($private && $r->is_initial_req) ? 403 : -1
		## in production pub/ will be
		## overriden by pound/thttpd
	} elsif ($req eq 'search') { search_handler(@path) }
	elsif ($req eq 'latest') { latest_handler(@path) }
	else { ## the virtual repository file system
		($r->uri =~ /\/$/) 
			? directory_handler(@path) : file_handler(@path);
	}

	return 0;
}

sub search_handler {
	my (@path) = @_;
	my $r = CyberArmy::WWW::Request->instance();

	my $param = $r->getParams({ escapehtml => 'yes' });
	exit (412) if (not $param->{'q'})
		or (length $param->{'q'} < 2);

	## TODO: hackish, needs to be rewritten
	my $files = CyberArmy::Repository::Search->new(
		{ map { $_=>'%'.$param->{'q'}.'%' } qw(name type description) },
		{
			logic => ' OR ', limit => 30, cmpop => ' like ', 
			join_user => 1, join_dir => 1, order_by => 'name',
			where =>	" AND repository_file.status = 'listed' ".
						' AND read_access IS NULL',
		}
	);
	
	$r->printTemplate(
		'repository/repository.tmpl', 
			{ files => $files, path=>\@path, 
				directory => { name => 'search' }});
}

sub latest_handler {
	my (@path) = @_;
	
	my $r = CyberArmy::WWW::Request->instance();
	my $param = $r->getParams({ escapehtml => 'yes' });
	($param->{'limit'} && $param->{'limit'} 
		=~ /^\d+$/) or $param->{'limit'} = 10; 
	
	my $files = CyberArmy::Repository::Search->new(
		{ status => 'listed' },{
			where => ' AND read_access IS NULL ',
			limit => $param->{'limit'}, join_dir => 'yes',
			join_user => 'yes', order_by => 'upload_date DESC', 
		}
	);	
	
	if ($path[1]) {
		if ($path[1] eq 'repository.rss') {
			my $server = $r->getServerLink();
			my $link = $server.$r->location().'/';
			my $rss = new XML::RSS (version => '1.0');
			$rss->channel(
				title		=> 'CyberArmy Files Repository',
				link		=> $link,
				description	=> 'Latest public cyberarmy files repository additions',
				dc			=> {
					publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
					language	=> 'en-US'
				},
				syn			=> {
					updatePeriod     => 'daily',
					updateFrequency  => 1,
				}
			);
				
			$rss->image(
				title		=> 'CyberArmy',
				url			=> $server.'/includes/header/cyberarmy.gif',
				link		=> $link,
			);

			$rss->textinput(
				title		=> 'Search',
				description	=> 'Search the CyberArmy Repository',
				name		=> 'q',
				link		=> $link.'search/',
			);

				
			while (my $file = $files->fetch()) {
				$rss->add_item(
					title		=> $file->{'description'},
					link		=> $link. $file->{'path'}. '/'.
									$file->{'name'}. '?action=view',
					dc			=> {
						creator	=> $file->{'uploaded_by_nickname'},
						subject	=> $file->{'path'},
						date	=> $file->{'upload_date'}.' -0000',
					}
				);
			}
			
			$r->content_type('application/xml');
			$r->print( $rss->as_string );
			
		} else { exit 404 }
	} else {
		$r->printTemplate(
			'repository/repository.tmpl', {
				path=>\@path, files => $files,
				directory => { name => 'latest' },
			}
		);
	}
}

sub directory_handler {
	my (@path) = @_;
	my $r = CyberArmy::WWW::Request->instance();
	my $id = $path[0] ? {path => \@path} : {id => 0};
	my $directory = CyberArmy::Repository::Category->new($id);
	exit(404) unless ($directory || 
		(defined $id->{'id'} && $id->{'id'} == 0));
	$r->redirectTo($r->uri . '/') unless ($r->uri =~ /\/$/);
	my $user = CyberArmy::WWW::Request::User->instance();
	if ($directory && $directory->{'read_access'}) {
		exit(403) unless $user && 
		$user->CheckGroupList(
			"$directory->{admin_access};".
			"$directory->{read_access};" .
			"$directory->{review_access}"
		);
	}

	my $param = $r->getParams({ escapehtml => 'yes' });
	if (not $param->{'action'}) {
		my $directories = $directory ? $directory->getListOf() : 
			CyberArmy::Repository::Category->getListOf();

		#Get order to sort files by
		my $orderby = 'upload_date'; #default order
		my $posted = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
			$r->getParams({from=>'posted', escapehtml => 1} );

		if($posted->{'column'} && $posted->{'order'}){ #set user defined order

			if((grep $_ eq $posted->{'column'}, ('upload_date', 'name', 'size')) && (grep $_ eq $posted->{'order'}, ('asc', 'desc'))){
				$orderby = $posted->{'column'}.' '.$posted->{'order'};
			}
		}
		my $files = CyberArmy::Repository::Search->new(
			{ category_id => $directory->{'id'},
			status => 'listed'},{ join_user => 'yes',
			order_by => $orderby});
		$r->content_type('text/html');
		$r->printTemplate(
			'repository/repository.tmpl',{
			path => \@path, files => $files,
			directory => $directory,
			directories => $directories,
			order => $posted->{'order'} ? $posted->{'order'} : 'asc',
			column => $posted->{'column'} ? $posted->{'column'} : 'upload_date'
		});
	} elsif ($param->{'action'} eq 'upload') {
		if ($directory->{'upload_access'}) {
			$r->errorTemplate(403,'repository/closed_error.tmpl')
				unless $user && 
					$user->CheckGroupList($directory->{'upload_access'});
		}
		$directory->{'status'} ||= 'closed';
		$r->errorTemplate(403,'repository/closed_error.tmpl')
			unless ($directory->{'status'} eq 'open');
		exit(403) unless $user; ## no anonymous uploads

		if ($r->method eq 'POST') {
			my $apr = Apache::Request->new(shift,
				POST_MAX => $r->dir_config('UPLOAD_LIMIT'),
				TEMP_DIR => $ENV{'FILES_ROOT'},
			);

			my $parse = $apr->parse();
			my $upload = $apr->upload or exit($parse);

			CyberArmy::WWW::Utils::escapeHtml(
				(my $description = $apr->param('description')),
				(my $filename = (split(/(\\|\/)/,$upload->filename))[-1])
			); exit(412) unless ( $filename && $description );

			my $md5 = Digest::MD5->new();
			$md5->addfile($upload->fh());
			my $digest = $md5->hexdigest();
			
			if (my $check_if_exists = CyberArmy::Repository::Search
				->new({md5_digest => $digest},{join_dir=>'yes'})) {
				if (my $file = $check_if_exists->fetch) {
					$r->errorTemplate(403,
						'repository/upload_exists.tmpl',{ file => $file });
				}
			}
			
			my $log = $r->log;
			
			my $dirumask = umask; umask(0077);
			my $upload_dir = $ENV{'FILES_ROOT'}.$digest.'/';
			unless (mkdir $upload_dir) {
				$log->error($upload_dir.': '.$!); exit(503);
			} umask($dirumask);


			unless ($upload->link($upload_dir.$filename)) {
				$log->error('Uploading to '.
						$upload_dir.$filename. ': '.$!);
				rmdir $upload_dir; exit(503);
			} chmod (0644,$upload_dir.$filename);

			CyberArmy::Repository->new({
				category_id => $directory->{'id'},
				name => $filename,
				size => $upload->size,
				md5_digest => $digest,
				type => $upload->type,
				uploaded_by_caid => $user->caID,
				description => $description
			}) or exit(500);

			$r->printTemplate(
				'repository/upload_ok.tmpl',{
				upload => $upload,
				digest => $digest
			});

		} else {
			$r->printTemplate(
				'repository/upload.tmpl',{
				directory => $directory,
			});
		}
	} elsif ($param->{'action'} eq 'review') {
		$directory->{'review_access'}	= 
		$directory->{'admin_access'}	= 
			'repo_master;staff' unless $directory;
		if ($directory->{'review_access'}) {
			exit(403) unless $user
			&& $user->CheckGroupList(
				"$directory->{review_access};"
				."$directory->{admin_access}"
			);
		}
		if ($r->method eq 'POST') {
			my $posted = $r->getParams({
				from => 'posted',
				escapehtml => 'yes',
				multi => {file_id=>1}
			});
			exit(412) unless $posted->{'file_id'};
			$posted->{'review'} ||= 'accept';
			if ($posted->{'review'} eq 'accept') {
				my $log = $r->log(); my @list;
				foreach (@{$posted->{'file_id'}}) {
					my $file = CyberArmy::Repository->new({id=>$_});
					
					if ($file->update({status => 'listed'})) {
						push @list, $ENV{'FILES_ROOT'}
								. '/' . $file->{'md5_digest'};

						if ($posted->{'notify'}) {
							CyberArmy::Template->instance->process(
								'repository/notify_review.tmpl',{
									file => $file, review => 'accepted'
							},\my $body);
							CyberArmy::Message->new(
								caID => [$file->{'uploaded_by_caid'}],
								user => $user, body => $body,
								subject => $file->{'name'}
							);
						}
					}
				}
				chmod(($posted->{'read_access'} ? 0700 : 0750),@list);
				$r->redirectTo('./');
			} elsif ($posted->{'review'} eq 'reject') {
				if ($posted->{'notify'} && !$posted->{'message'}) {
					$r->content_type('text/html'); $r->printTemplate(
						'repository/notify_why_rejected.tmpl',
						{ files_to_reject => $posted->{'file_id'}});
				} else {
					foreach (@{$posted->{'file_id'}}) {
						my $file = CyberArmy::Repository->new({id=>$_});
						
						if ($posted->{'notify'}) {
							CyberArmy::Template->instance->process(
								'repository/notify_review.tmpl',{
									file => $file, review => 'rejected',
									message => $posted->{'message'}
							},\my $body);
							CyberArmy::Message->new(
								caID => [$file->{'uploaded_by_caid'}],
								user => $user, body => $body,
								subject => $file->{'name'}
							);
						}
						
						unlink ($ENV{'FILES_ROOT'}.'/'.$file->{'md5_digest'}
							.'/'.$file->{'name'}) && $file->remove &&
						rmdir ($ENV{'FILES_ROOT'}.'/'.$file->{'md5_digest'});
					}
					$r->redirectTo('./');
				}
			} else { exit (412) }
		} else {
			$param->{'review'} ||= 'pending';
			my $args = { status => $param->{'review'} };
			$args->{'category_id'} = $directory->{'id'} if $directory->{'id'};
			my $files = CyberArmy::Repository::Search
				->new($args,{join_user => 1,join_dir => 'yes', order_by => 'name'});
			$r->content_type('text/html');
			$r->printTemplate(
				'repository/review.tmpl',{
					files => $files,
					path => \@path,
					review => $param->{'review'}
			});
		}
	} elsif ($param->{'action'} eq 'admin') {
		$directory->{'admin_access'} = 'repo_master;staff'
			unless $directory;
		if ($directory && $directory->{'admin_access'}) {
			exit(403) unless $user
			&& $user->CheckGroupList(
				$directory->{'admin_access'}
			);
		};

		if (not $param->{'admin'}) {
			my $directories = $directory->{'id'} ? $directory->getListOf()
				: CyberArmy::Repository::Category->getListOf();
			$r->content_type('text/html');
			$r->printTemplate(
				'repository/admin.tmpl',{
				path =>\@path,
				directories => $directories
			});
		} elsif ($param->{'admin'} eq 'edit') {
			$param->{'edit'} or exit(412);
			my $edit_dir = CyberArmy::Repository::Category
				->new({ id => $param->{'edit'} }) or exit(404);
			exit(404) unless ($edit_dir->{'parent'} 
				== ($directory->{'id'} || $id->{'id'}));

			if ($r->method eq 'POST') {
				$r->checkReferer(
					Apache::Util::escape_uri($r->uri)) or exit(412);
				my $posted = $r->getParams({
					from => 'posted', escapehtml => 'yes' });
				my $action = $posted->{'action'} || 'update';
				foreach (qw(id action path parent)) { delete $posted->{$_} }
				if ($action eq 'delete') {
					if ($edit_dir->remove()) {
						$r->redirectTo('./?action=admin');
					} else {
						$r->content_type('text/html');
						$r->printTemplate('repository/remove_error.tmpl');
					}
					return;
				} else { $edit_dir->update($posted) }

				if (($edit_dir->{'read_access'} || '')
						ne ($posted->{'read_access'} || '')) {
					my $files_to_change = 
						CyberArmy::Repository::Search
							->new({category_id => $edit_dir->{'id'}});

					my ($file,@list);
					while ($file = $files_to_change->fetch()) {
						next if $file->{'status'} eq 'pending';
						push @list, $ENV{'FILES_ROOT'}.$file->{'md5_digest'};
					}
					chmod(($posted->{'read_access'} ? 0700 : 0750),@list);
				}

				$r->redirectTo(
					'./?action=admin&admin=edit&edit='.$edit_dir->{'id'}
				);
			} else {
				my @fields = keys %$edit_dir;
				$r->content_type('text/html');
				$r->printTemplate(
					'repository/edit.tmpl',{
					fields => \@fields,
					directory => $edit_dir
				});
			}
		} elsif ($param->{'admin'} eq 'new') {
			$r->checkReferer(
				Apache::Util::escape_uri($r->uri)) or exit(412);
			exit(412) unless $param->{'new'};
			my $prop = {
				name => $param->{'new'},
				parent => $directory->{'id'}
			};
			if ($directory->{'admin_access'}) {
				$prop->{'upload_access'}	=
				$prop->{'review_access'}	=
				$prop->{'admin_access'}		= 
				$directory->{'admin_access'}
			}

			$prop->{'read_access'} =
				$directory->{'read_access'};

			my $new = CyberArmy::Repository::Category->new($prop)
				or $r->redirectTo('./?action=admin');
			$r->redirectTo('./?action=admin&admin=edit&edit='.$new->{'id'});
		} else { exit (404) }
	} else { exit (404) }
}

sub file_handler {
	my (@path) = @_;
	if (my $file = CyberArmy::Repository->new({path=>\@path})) {
		my $r = CyberArmy::WWW::Request->instance();
		my $directory = CyberArmy::Repository::Category
			->new({id=>$file->{'category_id'}});
		my $user = CyberArmy::WWW::Request::User->instance();

		if ($directory->{'read_access'}) {
			exit(403) unless $user && 
				$user->CheckGroupList($directory->{'read_access'});
		};
		
		if ($file->{'status'} eq 'pending') {
			exit(403) unless $user && 
				$user->CheckGroupList(
					"$directory->{admin_access};".
					"$directory->{review_access}"
				);
		}
	
		my $param = $r->getParams({escapehtml => 'yes'});

		if (not $param->{'action'}) {
			if ($r->checkReferer() || !$r->header_in('Referer')) {
				if ($directory->{'read_access'} || 
						$file->{'status'} eq 'pending') {
					$r->internal_redirect('/files/pub/'.
						$file->{'md5_digest'}.'/'.$file->{'name'});
				} else {
					$r->redirectTo('/files/pub/'.
						$file->{'md5_digest'}.'/'.$file->{'name'});
				}	
			} else { $r->redirectTo($r->uri. '?action=view') }

		} elsif ($param->{'action'} eq 'view') {
			$r->content_type('text/html');
			$r->printTemplate(
				'repository/view.tmpl',{
				file => $file,
				path => join('/',@path),
				user => CyberArmy::User
					->new(caID => $file->{'uploaded_by_caid'})
			});
			
		} elsif ($param->{'action'} eq 'review') {
			my @fields = qw(name type category_id description);
			if ($r->method eq 'POST') {
				$r->checkReferer(
					Apache::Util::escape_uri($r->uri)) or exit(412);
				my $posted = $r->getParams({
					from => 'posted', escapehtml => 'yes' });
				$posted->{'action'} ||= 'update';
				if ($posted->{'action'} eq 'update') {
					my %updates = map{($_=>$posted->{$_})}@fields;
					$file->update(\%updates) or exit(412);

					## changing dir/category
					if ($file->{'category_id'} != 
							($updates{'category_id'} || 0 )) {
						my $new_dir = CyberArmy::Repository::Category
							->new({id=>$updates{'category_id'}}) or exit (404);
						exit(403) unless  $user->CheckGroupList(
							"$new_dir->{admin_access};".
							"$new_dir->{review_access}"
						);
						if ($file->{'status'} ne 'pending') {
							chmod $new_dir->{'read_access'} ? 0700 : 0750, 
								$ENV{'FILES_ROOT'}.'/'.$file->{'md5_digest'}
						}
						$file->{'category_id'} = $updates{'category_id'};
					}

					if ($posted->{'name'} ne $file->{'name'}) {
						my $dir = $ENV{'FILES_ROOT'}
								.'/'.$file->{'md5_digest'}.'/';	
						if (link($dir.$file->{'name'},
								$dir.$posted->{'name'})) {
							unlink $dir.$file->{'name'};
						} else { exit(500) }
						$file->{'name'} = $posted->{'name'};
					}
				} else { exit 412 }
				$r->redirectTo('/files/'.join('/',$file->getPath).'?'.$r->args);
			} else {
				my $uploader = 
				$r->printTemplate(
					'repository/review_file.tmpl',{
					file => $file, fields => \@fields, 
					user => CyberArmy::User
						->new(caID => $file->{'uploaded_by_caid'})
				});
			}
		} else { exit 404 }
	} else { &directory_handler }
}
