#!/usr/bin/perl -W
package CyberArmy::WWW::Projects;

use strict;
use CyberArmy::User;
use CyberArmy::Utils;
use CyberArmy::Brigades;
use CyberArmy::Template;
use CyberArmy::Message;
use CyberArmy::Projects;
use CyberArmy::Forum;
use CyberArmy::Library;
use CyberArmy::BBCode;

$CyberArmy::WWW::Projects::VERSION = '0.1';

sub handler {
	my $r = CyberArmy::WWW::Request->instance( shift );

	# Where are we going and what are we doing?
	my $path = $r->path_info;
	CyberArmy::WWW::Utils::escapeHtml($path);
	my (undef,$project,@args) = split /\// , $path;
	$project = $project && $project =~ /^(.*)\.html$/ ? $1 : $project;
	
	if(!$project) {	&ProjectMenu(); }
	elsif ($project eq "admin") { 
		if ($r->method eq 'POST') {
			&AdminPost();
		} else { &ShowAdmin(); }
	}
	elsif ($project eq "completed") { 
		&ProjectMenu(1);
	}
	elsif ($project eq "search") { 
		if ($r->method eq 'POST') {
			&Search(0);
		} else { &ProjectMenu(0); }
	}
	elsif ($project eq "searchCompleted" || $project eq "searchcompleted") { 
		if ($r->method eq 'POST') {
			&Search(1);
		} else { &ProjectMenu(1); }
	}
	elsif ($project && !@args) { &ShowProject($project) }
	elsif ( $args[0] eq 'join' ) { 	&JoinProject($project); }
	elsif ( $args[0] eq 'discharge' ) { 	&LeaveProject($project); }
	elsif ($args[0] eq "lastlogins") { &lastlogins($project) }
	elsif ( $args[0] eq 'command' ) { 
		if ($r->method eq 'POST') {
			&UpdateProject($project);
		} else { &AdminProject($project); }
	}
	elsif ( $args[0] eq 'newsadmin' ) { 
		if ($r->method eq 'POST') {
			&AdminNews($project)
		} else { &NewsForm($project) }
	}
	elsif ( $args[0] eq 'articles' ) { 
		&ProjectArticles($project)
	}
	else { &ProjectMenu(); }

}

sub Search {	
	my ($completed) = @_;
	my $r = CyberArmy::WWW::Request->instance;
	my $posted = $r->getParams({from=>'posted',escapehtml=>1});
	my $s = $posted->{search};
	$s =~ s/\\\*//gi;
	$s =~ s/\\.//gi;
	my @words = split(/ /,$s);
	&ProjectMenu($completed,@words);

	# This search function merely highlights the words searched for, rather than
	# only listing projects with 'x' in. This saves us going near the database
	# with raw input, also minimises stupidity input, i.e 64k characters etc..
}

sub AdminPost {
	my $r = CyberArmy::WWW::Request->instance;
	my $posted = $r->getParams({from=>'posted',escapehtml=>1});

	my $user = CyberArmy::WWW::Request::User->instance or return 403;
	return 403 unless $user->IsInGroupList('commander','proj_coord');

	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From project_categories order by category_name asc"
			);
	$categories->execute();
	if ($posted->{action_type} eq "del") {
		my $ok;
		return 403 if ($posted->{delid} eq "misc");
		while (my $row = $categories->fetchrow_hashref()) { 
			if ($posted->{delid} eq $row->{category}) {
				if ($posted->{"conf".$row->{category}}) {
					$ok = 1;
				}
				else { return 412 }
			}
		}
		if ($ok == 1) {
			# Move all the relevant projects to Misc type.
			$CyberArmy::Utils::Database::dbh->do(
				"UPDATE projects set proj_type = 'misc' where proj_type = ?",
				undef,$posted->{delid}
			);
			# Rm -rf the category from the list table..
			$CyberArmy::Utils::Database::dbh->do(
				"delete from project_categories where category = ?",
				undef,$posted->{delid}
			);
		}
		else { return 404 } # wtf?
	}
	elsif ($posted->{action_type} eq "rename") {
		if ($posted->{rnid} eq "misc") { return 403 }
		return 403 unless ($posted->{catdisplay} ne "");
		$CyberArmy::Utils::Database::dbh->do(
			"UPDATE project_categories set category_name = ? where category = ?",
			undef,$posted->{catdisplay},$posted->{rnid}
		);
	}
	elsif ($posted->{action_type} eq "new") {
		return 412 unless 
			(($posted->{catname} ne "") && ($posted->{catdisplay} ne ""));
		# Ok i guess...
		# Insert the jobby into the thingy..
		$CyberArmy::Utils::Database::dbh->do(
				"Insert into project_categories values (?,?)",
				undef,$posted->{catname},$posted->{catdisplay}
		);
	}
	CyberArmy::WWW::Request->instance->redirectTo('/projects/admin');
}

sub ShowAdmin {
	my $user = CyberArmy::WWW::Request::User->instance or return 403;
	return 403 unless $user->IsInGroupList('commander','proj_coord');

	my @cats;
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From project_categories order by category_name asc"
			);
	$categories->execute();
	while (my $row = $categories->fetchrow_hashref()) { push(@cats,$row); }
	my $categorys = \@cats;

	my $logs = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * from log_projects order by adate desc limit 50"
	);
	$logs->execute();
	my @ress;
	while (my $res = $logs->fetchrow_hashref()) { push(@ress,$res); }

	my $ref = \@ress;
	my $r = CyberArmy::WWW::Request->instance();

	$r->content_type('text/html');
	$r->printTemplate(
		'projects/projc.tmpl', {
			title => "CyberArmy Projects Admin",
			cats => $categorys,
			logs => $ref
		}
	);
	
	return 0;

}


sub ProjectMenu {
	my ($completed,@words) = @_;
	my $user = CyberArmy::WWW::Request::User->instance;

	# Get the old categories for menus..
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From project_categories order by category_name asc"
			);
	$categories->execute();
	my $categoryList = $categories->fetchall_arrayref({});

	my $brigade = ($user) ? $user->brigade: "";
	my $admin = (($user) && $user->IsInGroup('proj_coord')) ? qq~ [<a href="/projects/admin">Admin</a>] ~ : "";
	my $create = (($user) && $user->IsInGroup('x'.$brigade)) ? qq~ [<a href="/brigades/$brigade/adminproject/new">Create New Project</a>] ~ : "";

	#Do not list completed projects here (ie status!=3)
	my $projectList = &getProjects($completed,@words);


	# That didn't fuck up? Wahey! First time for everything eh? Lets go!
	my $r = CyberArmy::WWW::Request->instance();

	$r->content_type('text/html');
	if($completed  && $completed==1)
	{
		$r->printTemplate(
			'projects/completed.tmpl', {
				title => "CyberArmy Projects :: Completed",
				categories => $categoryList,
				projects => $projectList,
				admin => $admin,
				create => $create
			}
		);		
		return 0;
	
	}
	
	$r->printTemplate(
		'projects/index.tmpl', {
			title => "CyberArmy Projects",
			categories => $categoryList,
			projects => $projectList,
			admin => $admin,
			create => $create
		}
	);
	
	return 0;
}



#######################
# sub getProjects
#
# Returns list of projects, by default this will exclude completed projects
#
# Parameters:
#	completed   - 0 to exclude completed projects, 1 to list only completed projects
#	words		- Words to highlight (used for searches)
###
sub getProjects {
	my ($completed,@words) = @_;
    
	my $r = CyberArmy::WWW::Request->instance();	
	my $user = CyberArmy::WWW::Request::User->instance;
    #By default do not list completed projects
	my $status = "projects.proj_status!=3";
	if($completed == 1){
   		 #List only completed projects
		$status = "projects.proj_status=3";
	}
	my $projects = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT * From projects left join brigades on 
		 projects.proj_brigade_parent = brigade_id  WHERE ".$status." order by proj_name asc"
		);
	$projects->execute();
	my $projectList = $projects->fetchall_arrayref({});
	foreach my $project (@$projectList) {
		$project->{desc} = $project->{proj_desc};
		$project->{full_desc} = $project->{desc};
		#$desc =~ s#$_#<span style="background-color: \#FF9900">$&</span>#gi foreach (@words);
		
#		foreach (@words) {
			# It would have been cool to accept regex input here, but it's too open to abuse..
			# Therefore, we do a 'bit' of extra input validation.
#			$_ =~ s/\.//gi;
#			$_ =~ s/\\\*//gi;
			$project->{desc} =~ s#\Q$_\E#<span style="background-color: \#FF9900">$&</span>#gi foreach (@words);
#		}

		$project->{desc} =~ s/^(.{150}).*$/$1\.\.\./i if (!@words);
		CyberArmy::BBCode::apply($project->{desc});
		$project->{ok} = 0; #Classified project?
		my $mark = "";
		if (($project->{proj_access} eq "C") || ($project->{proj_q} eq "y")) {
			if (($user) && (($user->IsInGroup("x".$project->{proj_brigade_parent})) || 
				($user->IsInGroup($project->{'proj_group'})) ||
				($user->IsInGroup('proj_coord')) ||
				($user->CheckGroupList($project->{proj_invite})))) 
			{
				$project->{ok} = 1;
				$project->{mark} = ($project->{proj_access} eq "C") ? "<b>[Classified]</b> " : "<b>[Queued]</b> "; 	
			}
		}
		elsif ($project->{proj_access} eq "I") { # Internal project
			if (($user) && (($user->IsInGroup($project->{proj_group})) ||
				($user->IsInBrigade($project->{proj_brigade_parent})) ||
				($user->IsInGroup('proj_coord')) ||
				($user->CheckGroupList($project->{proj_invite}))))
			{
				$project->{ok} = 1;
			}
		} 
		else { 
			$project->{ok} = 1;
		}

		
	}
	
	return $projectList;


}


sub UpdateProject {
	my ($pnum) = (shift);

	my $project = CyberArmy::Projects->new($pnum,
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
			'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_img,proj_leader_access'
	) or return 404;
	
	my $parent = $project->brigade_parent;

	my $user = CyberArmy::WWW::Request::User->instance or return 403;
	return 403 unless $user->IsInGroupList(
		$project->group.'=3','proj_coord','x'.$parent
	);
	

	my $r = CyberArmy::WWW::Request->instance;
	my $posted = $r->getParams({from=>'posted',escapehtml=>1});

	if ($posted->{action_type} eq "update") { #update the invite/status/img
		my $img = $posted->{imgurl};
		# Interesting issue brought up by Ikioi here...
		# if you make the imgurl /my/logout.. when loaded.. it does just log you out.
		# Heres some regexs to hopefully stop shit like that from occuring..

		$img =~ s/^http:\/\/|\(|\)|://g;
		if ($img !~ m/(png|gif|jp(e?)g|bmp)$/i) { return 403 }
#			if ($img =~ m/cyberarmy.net/i) { ProjectsError("Sorry, no locally hosted images"); }
		if ($img =~ m/(\/logout|\/command)/i) { return 403 }	
		$img =~ s/\.\.//gi; # traversal handled?


		$CyberArmy::Utils::Database::dbh->do(
			"update projects set proj_invite = ?, proj_status = ?, proj_img = ?, proj_leader_access = ? ". 
			"where proj_group = ?",
			undef,$posted->{invite},$posted->{status},$img,$posted->{leader_access},$project->group
		);
		# Log what we've done..
		my $log = (($posted->{log} ne "") && ($posted->{log} ne "Log")) ? "Project Admin Options Updated :: ".$posted->{log} : "Project Admin Options Updated";
		$project->Log( on => "project",
				   type => "project_updated",
				   msg => $log,
				   action_by => $user->showname,
				   action_by_id => $user->caID
		);
	}
	elsif ($posted->{action_type} eq "proper_admin") { # proj_coord only
		return 403 unless ($user->IsInGroup("proj_coord"));
		
		unless (($posted->{brigade_parent}) || ($posted->{leader_nickname})) {
			return 412;
		}
		if (($posted->{brigade_parent}) && ($posted->{brigade_parent} ne "")) {
			$CyberArmy::Utils::Database::dbh->do(
			"UPDATE projects set proj_brigade_parent = ? where proj_group = ?",
			undef,$posted->{brigade_parent},$project->group
		);
		}
		if (($posted->{leader_nickname}) && ($posted->{leader_nickname} ne "")) {
			my $newguy = CyberArmy::User->new(
				nickname => $posted->{leader_nickname}
			) or return 404;

			my $oldguy = $project->GetLeader();
			$oldguy->Update(delfromgroup => $project->group );
			$newguy->Update(addtogroup => $project->group. '=3' );
		}
	}
	elsif ($posted->{action_type} eq "del") { #delete a user

		return 412 unless ($posted->{delid} ne "");
		return 403 if ($project->GetLeader()->caID eq $posted->{delid});

#		$CyberArmy::Utils::Database::dbh->do(
#			"delete from projects_users where proj_id = ? and proj_user_id = ?",
#			undef,$pnum,$posted->{delid}
#		) or ProjectsError("No such user?");
	
		my $loser = CyberArmy::User->new(
			caID => $posted->{delid}
		) or return 404;
		$loser->Update( delfromgroup => $project->group ) or return 500;


                my $list = $loser->todoList();
                
		$list->del(undef,$project->group); #id is left undef to indicate del all ToDos of this project

		$project->Log( on => "project",
				   type => "user_removed",
				   msg => 'Removed '.$loser->showname.' from project',
				   action_by => $user->showname,
				   action_by_id => $user->caID
		);
		
		$loser->Log( 
			type => 'modified', logby => $user->nickname,
			logbycaID => $user->caID, 
			msg => 'Removed from group '.$project->group
		);

		# Inform user they were removed from the project
		CyberArmy::Message->new( 
			caID => [$loser->caID], 
			user => $user,
			subject => 'Removed From '.$project->name,
			body => 'You have been removed from  [b][url=/projects/'.
				$project->group.']'.$project->name. '[/url][/b].');
	}
       	elsif ($posted->{action_type} eq "todo") {
                return 412 unless (($posted->{userid}) && ($posted->{todomsg}));
                my $sendto = CyberArmy::User->new(
					caID => $posted->{userid} 
				) or return 404;
                return 403 unless ($sendto->IsInGroup($project->group));

                my $list = $sendto->todoList();
                $list->add($posted->{todomsg}, $project->group);
		# Log it
		$project->Log( on => "project",
			type => "todo_added",
			msg => 'Added Task to '.$sendto->showname.'\'s ToDo list',
			action_by => $user->showname,
			action_by_id => $user->caID
			);
        }
	elsif ($posted->{action_type} eq "todoAll") {
		return 412 unless ($posted->{todomsg});

		my $users = $project->GetUsers();
		foreach (@$users){

			my $user = CyberArmy::User->new(
				caID => $_->caID
			) or return 500;
			my $list = $user->todoList();
			$list->add($posted->{todomsg}, $project->group);

		}

		# Log it
		$project->Log( on => "project",
			type => "todo_added",
			msg => 'Added Task to to all members',
			action_by => $user->showname,
			action_by_id => $user->caID
			);

        }
       	elsif ($posted->{action_type} eq "todoDel") {
			return 412 unless ($posted->{todoid} && $posted->{userid});
			my $sendto = CyberArmy::User->new(
				caID => $posted->{userid}
			) or return 404;                

                my $list = $sendto->todoList();
                
			return 404 unless $list->del($posted->{todoid},$project->group);

		# Log it
		$project->Log( on => "project",
			type => "todo_deleted",
			msg => 'Deleted Task from '.$sendto->showname.'\'s ToDo list',
			action_by => $user->showname,
			action_by_id => $user->caID
			);
        }
	elsif ($posted->{forum_request}) {
		if(!$posted->{reason} || $posted->{reason} eq ""){
			my $r = CyberArmy::WWW::Request->instance();
			$r->content_type('text/html');
			$r->printTemplate('projects/request.tmpl');
			return;
		}
		else{
			my $body = 'Please make a forum for Project <a href="/projects/'.$project->group.'">'.$project->name.'</a><br /><br /><b>Reason:</b><br /><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="80%" bgcolor="#364C7D"><tr><td width="100%">'.$posted->{reason}.'</td></tr></table></center>';
			my $forum = CyberArmy::Forum->new( id=> "projectadm" );
			$forum->PostMessage( author => $user->showname,
					author_caID => $user->caID,
					author_rank => "Leader",
					subject => "Forum Request",
					body => $body
			);
		}
	}
	# all done :)
	CyberArmy::WWW::Request->instance->redirectTo('/projects/'.$project->group .'/command');
}		
		
sub JoinProject {
	my ($project) = (@_);
	my $user = CyberArmy::WWW::Request::User->instance;
	return 403 unless ($user);
 
	my $po = CyberArmy::Projects->new($project,
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
			'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_q'
		) or return 404;
#	warn "PO: $po->invite";

	return 403 if ($po->q eq "y");

	if (
	   ($user->IsInGroup($po->group)) && 
	   ($user->IsInGroup($po->group) != '0E0')
	) { return 403 }

	return 403 unless (
		($user->CheckGroupList($po->invite)) || ($po->invite eq "open")
	);

	# rm from project list afterwards...
	my $invite = $po->invite;
	my $nickname = $user->nickname;
	
	$invite =~ s/~$nickname(;?)//gi;

	$CyberArmy::Utils::Database::dbh->do(
	"update projects set proj_invite = ? where proj_id = ?",undef,$invite,$project);

	# Otherwise we're ok?
	$user->Update( addtogroup => $po->group);
	
	

	if (!$user->IsInGroup('trooper')) {
			# We have a new recruit joining a brigade -> promote him.
			$user->Update( addtogroup => 'trooper' );
			# cMS Him

			# Safe to make a CinC object?
			my $cinc = CyberArmy::Brigades->new(1)->Commander();
			CyberArmy::Message->new(
				caID => [$user->caID],
				user => $cinc,
				subject => 'Congratulations! '.$user->showname,
				body => qq~ Congratulations! Joining a project and committing some of your time to CyberArmy is a big step, for this you have been promoted to Trooper. Further promotions will rely on you achieving the tasks set for you by your project leader, or working for a brigade - the higher the rank, the more time it will take you to get there, but the larger the rewards and respect.
					\nGood luck and congratulations again!
					\nCommander in Chief ~.$cinc->showname
			);

			$user->Log(
				type => 'promoted',
				msg => 'auto-promo to trooper',
				logby => $cinc->showname,
				logbycaID => $cinc->caID
			);
			undef $cinc;
	}


	# Log the fun:
	$po->Log( on => "project",
		  type => "user_joined",
		  msg => "user joined project",
		  action_by => $user->nickname,
		  action_by_id => $user->caID
		);

	# Inform the project leader
	my $leader = $po->GetLeader();
	CyberArmy::Message->new( 
		caID => [$leader->caID], 
		user => $user,
		subject => 'New Member In '.$po->name,
		body => $user->showname. ' joined the project  [b][url=/projects/'.
				$po->group.']'.$po->name.'[/url][/b].');
				
	$user->Log( 
		type => 'modified', logby => $leader->nickname,
		logbycaID => $leader->caID, 
		msg => 'Added to group '.$po->group
	);

	# And we're done! :D
	CyberArmy::WWW::Request->instance->redirectTo('/projects/'.$po->group);		
}


sub LeaveProject {
	my ($project) = (@_);
	my $user = CyberArmy::WWW::Request::User->instance;
	Apache::exit(403) unless ($user);
 
	my $po = CyberArmy::Projects->new($project,
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
			'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_q'
		) or return 404;


	if (!($user->IsInGroup($po->group))) { return 403 }
	if (($user->IsInGroup($po->group."=3"))) { return 403 }

	my $nickname = $user->nickname;
	

	# Log the request
	$po->Log( on => "project",
		  type => "discharge_request",
		  msg => "user requested discharge",
		  action_by => $user->nickname,
		  action_by_id => $user->caID
		);

	# Send request to project leader
	my $leader = $po->GetLeader();
	CyberArmy::Message->new( 
		caID => [$leader->caID], 
		user => $user,
		subject => 'Discharge request from project '.$po->name,
		body => 'This is a discharge request from '.$user->showname. ' for the project [b][url=/projects/'.
				$po->group.']'.$po->name.'[/url][/b].');
				

	# Tell the user what happened	
	my $r = CyberArmy::WWW::Request->instance();
	$r->printTemplate(
		'projects/discharge.tmpl', {
			title => "Discharge Request",
			project => $po
		}
	);
	
	return 0;
}

sub AdminProject {
	my ($project) = (shift);
	my $user = CyberArmy::WWW::Request::User->instance;
	return 403 unless ($user);

	my $po = CyberArmy::Projects->new($project,
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,'.
			'proj_status,proj_start,proj_end,proj_modified,proj_access,proj_invite'
	) or return 404;

	return 403 unless (
		($user->IsInGroup($po->group."=3")) || 
		($user->IsInGroup('x'.$po->brigade_parent)) ||
		($user->IsInGroup("proj_coord"))
	);


	my $users = $po->GetUsers();
	my $usertodo = $CyberArmy::Utils::Database::dbh->prepare(
		'SELECT * from user_todo where group_id = ? ORDER BY date'
	);
	$usertodo->execute($po->group);
 	my $todolist= $usertodo->fetchall_arrayref({});
	my $cmdr=$po->GetAdmin();
	#in case the leader is not assigned we get an error here!
	my $leader = $po->GetLeader() ;
	if (!$leader) {
		#return 409; #stay here!
		# 02-07-2008 WE GOT A BUG:
		# If a project hasn't got an assigned leader (or someone like Icydemon fucks it up), the system refuses to
		# display the admin functions since the variable $leader is undefined (as much as I understood by now)
		# my workaround is to assign temporarily a System account leader (SYSTEM) and redirect the admin in this
		# page to change the leader
		# Nevertheless I hope that this wont pose a problem in the the structure of projects and its a temporary fix
		# (till a wizard comes with something better)
		$CyberArmy::Utils::Database::dbh->do("INSERT INTO user_group VALUES('0000001', ? , 3)", undef, $po->group);
		#hopefully this will work fine!
		CyberArmy::WWW::Request->instance->redirectTo('/projects/'.$po->group.'/command/');		
		#It works. more comments on feedback forum!
 	}

	my $r = CyberArmy::WWW::Request->instance();

	my $st = $po->status;
	my $status = qq~ <option value="$st">$CyberArmy::Projects::StatusArray[$st]</option> ~;
	
	my $n = 0;
	foreach my $s (@CyberArmy::Projects::StatusArray) {
		$status .= qq~ <option value="$n">$CyberArmy::Projects::StatusArray[$n]</option>\n ~ unless ($n eq $st);
		$n++;
	}

	my @logs;
	my $id = $po->id;
	my $log_o = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * FROM log_projects WHERE project_id = ? ORDER BY adate DESC limit 15"
	);
	$log_o->execute($id);
	while (my $row = $log_o->fetchrow_hashref()) { push(@logs,$row) }
	my $l = \@logs;

	my $project_coord = ($user->IsInGroup("proj_coord")) ? 1 : 0;
	#sanity check to avoid 500s
	
	$r->content_type('text/html');
	$r->printTemplate(
		'projects/admin.tmpl', {
			title => "Admin Project",
			users => $users,
			leader => $leader->nickname,
			admin => $cmdr->nickname,
			project => $po,
			status => $status,
			proj_coord => $project_coord,
			logs => $l,
			todolist => $todolist
		}
	);
	
	return 0;
}

sub AdminNews {
	my $proj = shift;
	my $user = CyberArmy::WWW::Request::User->instance or return 403;

	# Stupid?
	my $projectobject = CyberArmy::Projects->new($proj,
		"proj_id,proj_brigade_parent,proj_group,proj_q,proj_leader_access")
	or return 404;

	# Wannabe?
	return 403 unless (
		($user->IsInGroup($projectobject->group."=3")) || 
		($user->IsInGroup('x'.$projectobject->brigade_parent)) ||
		($user->IsInGroup("proj_coord")) ||
		($user->CheckGroupList($projectobject->leader_access))
	);

	return 403 unless ($projectobject->q eq "n");

	my $r = CyberArmy::WWW::Request->instance;
	# Get a hash of all the POSTed variables (cleaned)
	my $posted = $r->getParams({from=>'posted',escapehtml=>1});;

	# What do they want to do? add or del?
	if ($posted->{action_type} eq "add") {
		if($posted->{post}){
			# Blind?
			unless (($posted->{title} ne "") && ($posted->{content} ne "")) {
				return 412;
			}
			my $content = $posted->{content};
			#CyberArmy::BBCode::apply($content);
			# Whack 'em in the db.
			$CyberArmy::Utils::Database::dbh->do(
				"INSERT INTO project_news values ('',?,?,?,now(),?)",
				undef,$projectobject->id,$posted->{title},$content,$user->showname
			);
		
			#cMS the project members to inform them of the new news
			#Icydemon: Made a checkbox to enable the NewsAdmin to send or not to send the project news
			#to project members (and its god damn useful imo)
			#Note: Phishy's Comments Rock.
			if ($posted->{sendmessage} eq "yes") {
				my $users = $projectobject->GetUsers();
				my $cmstomembers;
				foreach (@$users){

					my $user = CyberArmy::User->new(
						caID => $_->caID
					) or return 404;

				$cmstomembers .= ';'.$user->nickname;

				}
				CyberArmy::Message->new( 
					nickname => [split /;/ , $cmstomembers], 
					user => $user,
					subject => $projectobject->name.' :: Project News ',
					body => 'A news item has been added for project [b]'.$projectobject->name.'[/b].<br><br><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="80%"><tr ><td width="100%">[b]Posted by:[/b] '.$user->showname.'</td></tr><tr><td>[b]Title:[/b] '.$posted->{title}. '</td></tr><tr bgcolor="#364C7D"><td>'.$content.'</td></tr></table>');
				}	
			}
				
		elsif($posted->{preview}){
			my $r = CyberArmy::WWW::Request->instance();
			$r->content_type('text/html');
			my $content = $posted->{'content'} || '';
			$r->printTemplate(
				'projects/news.tmpl', {
					news => $projectobject->GetNews(),
					project => $projectobject,
					news_title => $posted->{title} ? $posted->{title} : undef,
					content => $content ? $content : undef,
					news_content => $posted->{content} ? CyberArmy::BBCode::Parse($posted->{content}) : undef,
					news_poster => $user->showname			
				}
			);

			return;
	
		}
	}
	elsif ($posted->{action_type} eq "del") {
		# Dumbass?
		return 412 unless ($posted->{delid} ne "");
		# Some security checks...
		my $article = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
				"Select proj_id from project_news where news_id = ?",
				undef,$posted->{delid}
				);
		# if ($user->IsInGroup('butmunchers')) {}
		return 403 unless ($article->{proj_id} == $projectobject->id);
		
		# Delete the article
		$CyberArmy::Utils::Database::dbh->do(
				"Delete from project_news where news_id = ?",
				undef,$posted->{delid}
				);
	}

	# Send 'em back where they started.
	CyberArmy::WWW::Request->instance->redirectTo('/projects/'.$projectobject->group.'/newsadmin');
}

sub NewsForm {
	my ($project) = @_;
	my $user = CyberArmy::WWW::Request::User->instance or return 403;
	
	# No stupid people:
	my $projecto = CyberArmy::Projects->new($project,
		"proj_id,proj_brigade_parent,proj_group,proj_leader_access") 
	or return 404;

	# No pikeys:
	unless (($user->IsInGroup($projecto->group."=3")) || 
		($user->IsInGroup('x'.$projecto->brigade_parent))
		|| ($user->IsInGroup('proj_coord')) ||
		($user->CheckGroupList($projecto->leader_access))	
	) { return 403 }	

	return 403 unless ($projecto->q eq "n");

	my $news = $projecto->GetNews();

	# All good? Print the form:
	my $r = CyberArmy::WWW::Request->instance();

	$r->content_type('text/html');
	$r->printTemplate(
		'projects/news.tmpl', {
			news => $news,
			project => $projecto
		}
	);
	
	return 0;
}	

sub ShowProject {
	my ($pnum) = (shift);
	my $user = CyberArmy::WWW::Request::User->instance;

	# Stupidity checks first, does the project exist at all?
	my $project = CyberArmy::Projects->new($pnum,
		'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
		'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_img,proj_q'
	) or return 404;

	# Is it still linked to a brigade? If not, we dont allow it for now..
	my $bid = $project->brigade_parent;
	my $brigade = CyberArmy::Brigades->new($bid,
			'brigade_name,brigade_description,brigade_parent,brigade_color,'.
			'brigade_hex_color,brigade_recruiting,brigade_joinmode'
		) or exit (404); ## we probably want another error code here?
	my $bcolor = $brigade->hex_color;

	my $ok = 0; #Classified project?
	if ($project->IsClassified()) { # projectmember,brigadeparentxo,invitelist
		if (($user) &&
		    (($user->IsInGroup("x".$brigade->id)) || 
		    ($user->IsInGroup($project->group)) ||
		    ($user->CheckGroupList($project->invite)) || 
		    ($user->IsInGroup('proj_coord')) ||
		    ($user->IsInGroup("b".$project->brigade_parent)))) 
		{
			$ok = 1;
		}
	}
	elsif ($project->access eq "I") { # Internal project projectmember,brigadeparentmember,invitelist,commander
		if (($user) && (($user->IsInGroup($project->group)) ||
		   ($user->IsInBrigade($project->brigade_parent)) ||
                    ($user->IsInGroup('proj_coord')) ||
		   ($user->CheckGroupList($project->invite))))
		{
			$ok = 1;
		}
	} 
	else { 
		$ok = 1;
	}
	
	
	return 403 unless ($ok == 1); 
	
	my $cmdr = $project->GetAdmin() or return 404;

	my $r = CyberArmy::WWW::Request->instance;
	my $includes = $r->dir_config('dinah_includes');

	# This is disgusting code. 

	# Get the commander attributes and make 'em look pretty for the output
	my $cmdr_attr = $cmdr->getAttributes('badge','title_abrv');
	my $cmdcolor = $cmdr->brigade_hex_color if $cmdr->brigade_pos;
	my $cmdrbadges=''; foreach (@{$cmdr_attr->{'badge'}}) {
		$cmdrbadges .= qq~<img border="0" src="$includes/$_" height="15" width="10">~; 
	}
	my $cmdrname = $cmdrbadges." ".join(' ',@{$cmdr_attr->{'title_abrv'}}).
		" <a href=\"/~".$cmdr->nickname."\">".($cmdcolor? "<font color=\"$cmdcolor\">" : "<font color=\"\">").
		$cmdr->showname."</font></a>";

	# Same for the bloody leader...
	my $leader = $project->GetLeader();

	my $ldrname;

	if ($leader) {
		my $leader_attr = $leader->getAttributes('badge','title_abrv');
		my $ldrcolor = $leader->brigade_hex_color if $leader->brigade_pos;
		my $badges=''; 
		foreach (@{$leader_attr->{'badge'}}) {
			$badges .= qq~<img border="0" src="$includes/$_" height="15" width="10">~; 
		}
		$ldrname = $badges." ".join(' ',@{$leader_attr->{'title_abrv'}}).
			" <a href=\"/~".($leader->nickname ? $leader->nickname : "")."\"><font color=\"$ldrcolor\">".
			($leader->showname ? $leader->showname : "")."</font></a>";
	}
	else {
		$ldrname = "[Unassigned]";
	}

	my $projectusers = $project->GetUsers();

	# You guessed it, same for all the users to :/ *sigh*
	my @pretties;
	foreach	(@$projectusers) {
		my $user_attr = $_->getAttributes('badge','title_abrv');
		my $usercolor = $_->brigade_hex_color if $_->brigade_pos;
		my $ubadges=''; foreach (@{$user_attr->{'badge'}}) {
			$ubadges .= qq~<img border="0" src="$includes/$_" height="15" width="10">~; 
		}
		my $userdata = $ubadges." ".join(' ',@{$user_attr->{'title_abrv'}}).
			" <a href=\"/~".($_->nickname ?  $_->nickname : "")."\">".($usercolor ? "<font color=\"$usercolor\">" : "<font color=\"\">").
			($_->showname ? $_->showname : "")."</font></a>";
		push (@pretties, { nick => $_->showname, badges => $userdata });
		undef $usercolor;
	}	
	my $p = \@pretties;

	# Now we've got an array with two hashes, nick and badges, containing all of the shite we need to 
	# give everyone their special rank badges so they feel cool... :D
	
	# </Shit>

	# Define/Get some of the vars required for the vars..
	my $name = $brigade->name;
	my $news = $project->GetNews();
	my $access = $project->AccessName;
	my $status = $project->StatusName;
	my $category = $project->CategoryName;
	my $end_date = $project->end;
	my $desc = $project->desc;

	CyberArmy::BBCode::apply($desc);

	my $project_name = $project->q eq "y" ? "[Queued] ". $project->name : $project->name;

	$end_date =~ s/0000\-00\-00/Ongoing/i;

	# Is our user fella an admin/user... give him the nessecary links..
	my $adminstuff = "";
	my $member = "false"; 
	my $laflag="false";
	if ($user) {
		if(($user->IsInGroup($project->group))) 
		{
			$member = "true";
		}
		if(($leader->showname eq $user->showname) || ($cmdr->showname eq $user->showname)) {
			$laflag="true";
			$member="true";
		}	#The project leader and the project admin have eitherway access to the project thus they are members!
		if (($user->IsInGroup($project->group."=3")) 
			|| ($user->IsInGroup('x'.$bid))
			|| ($user->IsInGroup('proj_coord'))
			|| ($user->CheckGroupList($project->leader_access))
		) { # want to open this to everyone in the grp maybe?
			$adminstuff .= '[<a href="/projects/' .
				$project->group. '/newsadmin">NewsAdmin</a>]';
		}
		if ( ($user->IsInGroup($project->group."=3")) || 
			($user->IsInGroup('x'.$bid))
			|| ($user->IsInGroup('proj_coord'))
		) { # Only for leaders...
			$adminstuff .= ' [<a href="/projects/' .
				$project->group. '/command">Project Admin</a>]';
		}
		
	}
	
	#Count how many articles have been submitted related to this project
	my $articleCount = CyberArmy::Library->countProjectArticles($project->id);
	# Finally we actually print the stuff!
	$r->content_type('text/html');
	$r->printTemplate(
		'projects/desc.tmpl',
		{ 	
			title => $project_name,
			project => $project,
			desc => $desc,
			access => $access,
			status => $status,
			type => $category,
			parent => $bid,
			parentname => $name,
			bcolor => $bcolor,
			admin => $cmdr,
			adminname => $cmdrname,
			leadername => $ldrname,
			users => $p,
			news => $news,
			adminstuff => $adminstuff,
			enddate => $end_date,
			member => $member,
			isAdmin=> ($user && $user->IsInGroup($project->group."=3")) ? "true" : "false",
			articleCount => $articleCount,
			laflag => $laflag
		}
	);
	# long sub, lets exit quick before any other stuff needs printing all fluffily!
	return 0;
}	

####
# sub ProjectArticles
#
# Lists all articles submitted by/for a project
##

sub ProjectArticles {
	my ($pnum) = (shift);
	my $user = CyberArmy::WWW::Request::User->instance;
	#this section is temporarily off limits to non admins/staff
	Apache::exit(404) unless ($user && CyberArmy::Library->isAdmin($user));
	my $r = CyberArmy::WWW::Request->instance();	
	
	my $project = CyberArmy::Projects->new($pnum,
			'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
			'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_img,proj_q'
	) or return 404;

	my $column = 'article.date_submitted'; #default order
	my $orderby = $column.' DESC';
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
		$r->getParams({from=>'posted', escapehtml => 1} );
	
		if($params->{'column'} && $params->{'order'}){ #set user defined order
			if((grep $_ eq $params->{'column'}, ('date', 'author', 'title')) && (grep $_ eq $params->{'order'}, ('asc', 'desc'))){
			
				if($params->{'column'} eq "date")
				{
					$column ="article.date_submitted";
				}
				elsif($params->{'column'} eq "author")
				{
					$column ="users.nickname";			
				}
				elsif($params->{'column'} eq "title")
				{
					$column ="article.title";			
				}
				$orderby = $column.' '.$params->{'order'};
			}
	}
	
	#Count how many articles have been submitted related to this project
	my $articleCount = CyberArmy::Library->countProjectArticles($project->id);
	#get page number
	my $page = $params->{'page'}; my $all = 0;
	if (not $page) { $page = 1; }
	elsif (not ($page =~ /^\d+$/)) { $page = 1; }

	my $pages = $articleCount / 30;
	$pages = int $pages + 1 unless ($pages == int $pages);
	my $articles;
	if($page==1){
		$articles = CyberArmy::Library->getProjectArticles($project->id, $orderby, 30, 0);
	}else{
		$articles = CyberArmy::Library->getProjectArticles($project->id, $orderby, 30,($page-1) * 30);
	}		

	$r->content_type('text/html');
	$r->printTemplate(
		'projects/articles.tmpl', {
			title => $project->name." :: Articles",
			project => $project,
			articles => $articles,
			articleCount => $articleCount,
			order => $params->{'order'} ? $params->{'order'} : 'desc',
			column => $params->{'column'} ? $params->{'column'} : 'date',
			pages => $pages,
			page => $page
		}
	);
	
	return 0;
}



sub lastlogins {
	my ($project) = (shift);
	my $r = CyberArmy::WWW::Request->instance();


	my $user = CyberArmy::WWW::Request::User->instance;
	return 403 unless ($user);

	$project = CyberArmy::Projects->new($project,
		'proj_id,proj_name,proj_brigade_parent,proj_group,proj_desc,proj_type,proj_status,'.
		'proj_start,proj_end,proj_modified,proj_access,proj_invite,proj_img,proj_q'
	) or return 404;

	return 403 unless (
		($user->IsInGroup($project->group."=3")) || 
		($user->IsInGroup('x'.$project->brigade_parent)) ||
		($user->IsInGroup("proj_coord"))
	);
	
	my $projectusers = $project->GetUsers();

	my @members;
	my $includes = $r->dir_config('dinah_includes');
	foreach	(@$projectusers) {
		
		#my $user_attr = $_->getAttributes('badge','title_abrv');
		#my $usercolor = $_->brigade_hex_color if $_->brigade_pos;
		#my $ubadges=''; foreach (@{$user_attr->{'badge'}}) {
		#	$ubadges .= qq~<img border="0" src="$includes/$_" height="15" width="10">~; 
		#}
		#my $userdata = $ubadges." ".join(' ',@{$user_attr->{'title_abrv'}}).
		#	" <a href=\"/~".($_->nickname ?  $_->nickname : "")."\">".($usercolor ? "<font color=\"$usercolor\">" : "<font color=\"\">").
		#	($_->showname ? $_->showname : "")."</font></a>";
		#push (@members, { nick => $_->showname, badges => $userdata });
		#undef $usercolor;
		
		push @members, $_ ;
	}	


	my @memberdiffs;
	foreach (@members) {
		if ($_->HasValidSessionTime) {
			@memberdiffs[$_->caID] = "Logged In Since ".
				gmtime($_->session_time)." UTC";

		} elsif ($_->session_ltime) {
			my $diff = '';
			my $now = time;
			foreach (CyberArmy::Utils::getTimeInterval(
				($now - $_->session_ltime))
			) {
				if ($_->[0]) {
					$_->[1] =~ /^(\w)/;
					$diff .= "$_->[0]$1 "
				}
			}

			$diff .= 'ago' if ($diff);
			$memberdiffs[$_->caID] = "Last Seen ".$diff." UTC";
			undef $diff;
		} else {
			$memberdiffs[$_->caID] = "User Never Logged In";
		}
	}

	$r ->printTemplate('projects/lastlogins.tmpl',{
		user => $user,
		project => $project,
		members => \@members,
		memberdiffs => \@memberdiffs
	});

	return 0;
}

1;
