#!perl -wW
package CyberArmy::WWW::My;

use strict;
use XML::RSS;
use Data::Password;

use CyberArmy::Message;
use CyberArmy::Mail;
use CyberArmy::User;
use CyberArmy::Forum;
use CyberArmy::Template;
use CyberArmy::SkillsDB;
use CyberArmy::Projects;
use CyberArmy::Accomplishments;
use CyberArmy::Remote::Feed;
use CyberArmy::Remote::Missions;
use CyberArmy::User::PGPKey;

$CyberArmy::WWW::My::VERSION = '0.5';

sub handler {
	my $r = shift;

	my (undef,$node) = split(/\//,$r->path_info);
	if (not $node) { &showMy }
	elsif ($node eq 'cyberarmy.rss') { &showRSS }
	elsif ($node eq 'projects.rss') { &showProjectsRSS }
	elsif ($node eq 'profile') { &showProfile }
	elsif ($node eq 'skills') { &showSkills }
	elsif ($node eq 'todo') { &showTodo }
	elsif ($node eq 'messages') { &showMessages }
	elsif ($node eq 'projects') { &listProjects }
	elsif ($node eq 'password') { &recoverPassword }
	elsif ($node eq 'articles') { &articles }
	elsif ($node eq 'accomp') { &accomp }
	elsif ($node eq 'logout') { &logOut }
	
	else { return -1 } ## drop it to the default handler

}

sub _getUserInfo {
	my %info = ();

	if (my $user = CyberArmy::WWW::Request::User->instance) {
		my $getAttr = $user->getAttributes('title_abrv','badge');
		$info{'attributes'} = $getAttr->{'title_abrv'};
		$info{'badges'} =$getAttr->{'badge'};
		$info{'showname'} = $user->showname;
		$info{'nickname'} = $user->nickname;
		$info{'prvemail'} = $user->prvemail;
		$info{'pubemail'} = $user->pubemail;
		$info{'away'} = $user->away;
		$info{'messages'} = $user->getMessages()->CheckForUnreadMsg();
		if (my $brigade_pos = $user->IsInBrigade) {
			$info{'brigade'} = $user->Brigade;
			 if ($brigade_pos > 3) {
				$info{'brigade_queue'} = CyberArmy::Brigades->new($info{'brigade'})->getJoinQueueCount();
				$info{'iscommander'} = 'true';
			}
		}
		$info{'projects'} = $user->getProjects();

		$info{'todo'} = $user->todoList()->get("all",1, 5);
		$info{'todo_size'} = $user->todoList->count();
		$info{'todo_remain'} = ($info{'todo_size'} > 5) ? ($info{'todo_size'} - 5) : undef;
	}

	$info{'missions'} =  CyberArmy::Remote::Missions->get();
	$info{'feeds'} = CyberArmy::Remote::Feed::List->new();
	$info{'project_news'} = CyberArmy::Projects->GetLatest();

	return \%info;
}

sub _getLoggedIPs {
	my $user = CyberArmy::WWW::Request::User->instance;
	return $user ? $user->GetLoggedIPs : undef;
}

sub logOut {

	if (my $user = CyberArmy::WWW::Request::User->instance) {
		$user->logOut;
		my $r = CyberArmy::WWW::Request->instance();
		$r->err_headers_out->add('Refresh' => '3; URL=/');
        $r->printTemplate('my/logout.tmpl',{ userbox => _getUserInfo });
	} else { return 403 }
	
	return 0;

}

sub showMy {
	my $r = CyberArmy::WWW::Request->instance();
	my $params = $r->getParams({ escapehtml => 1 });
	my $headlines_offset;
	if ($params->{headline} && $params->{headline} =~ /^\d+$/ && 
			($params->{headline} % 5 == 0) && $params->{headline} > 0) {
		$headlines_offset = $params->{headline};
	}
	$headlines_offset ||=0;
	my ($projects_offset, $myProjects);
	if ($params->{projects} && $params->{projects} =~ /^\d+$/ && 
			($params->{projects} % 5 == 0) && $params->{projects} > 0) {
		$projects_offset = $params->{projects};
	}
	$projects_offset ||=0;
	
	my $args = {
		offset=> $projects_offset,
		limit => 5,
		select => 'proj_group,news_title,news_poster,news_title,'.
			'DATE_FORMAT(news_date,"%Y-%m-%dT%H:%i:%s") As w3cdate',
	};
	
	$myProjects = CyberArmy::Projects->GetLatest($args);
	my ($news_offset, $myNews, $library_offset, $myArticles,);
	if ($params->{news} && $params->{news} =~ /^\d+$/ && 
			($params->{news} % 10 == 0) && $params->{news} > 0) {
		$news_offset = $params->{news};
	}
	if ($params->{library} && $params->{library} =~ /^\d+$/ && 
			($params->{library} % 10 == 0) && $params->{library} > 0) {
		$library_offset = $params->{library};
	}
	$news_offset ||=0;
	$myNews = CyberArmy::Remote::Feed::List->new({
				offset => $news_offset,
				count => '10',
			   feedtype => 'news'});
	$myArticles = CyberArmy::Remote::Feed::List->new({
				offset => $library_offset,
				count => '10',
			   feedtype => 'library'});
	
	
	my $user = CyberArmy::WWW::Request::User->instance();
	my $forums = $CyberArmy::Utils::Database::dbh->prepare(qq~
		SELECT forum_list.forum,name,smalldesc,access_group,administrator,
		       moderator,guest_post,allow_html,searchable,type,icon,
		       threads_num,posts_num,last_post_id,locked,hex_color,
		       thread,subject,rdate,UNIX_TIMESTAMP(rdate),author,author_rank,author_color,category_name, category
		FROM forum_list LEFT JOIN forum_replies ON last_post_id = mid
		LEFT JOIN forum_categories ON type = category WHERE type != 'orphaned' ORDER by category_name,name
	~) or exit(500);
	$forums->execute; ## And Exec our prepared statement
	my ($forumnames,$previous);
	while ( my $forum = $forums->fetchrow_hashref() ) {
		if ($forum->{access_group} && $user) { ## OoOoO, A Restricted Forum
			next unless $user->CheckGroupList(
				"$forum->{access_group};".
				"$forum->{administrator};".
				"$forum->{moderator}");

			$forum->{access_group} =~ s/;/ /g;
		} elsif ($forum->{access_group} && !$user) { next }

		$forumnames=$forumnames.($previous ? ",'":"'").$forum->{forum}."'";
		$previous = $forum->{forum};
	}
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	$r->content_type('text/html');
	$r->printTemplate(
		'my/my.tmpl',{ 	userbox => _getUserInfo(),
		headlines => CyberArmy::Forum->getHeadlines({
					offset => $headlines_offset,
					count => '5',
					forums =>$forumnames}),
		projects =>  $myProjects,
		newsfeeds => $myNews,
		libraryfeeds => $myArticles,
		loggedips => _getLoggedIPs(),
		headlines_offset => $headlines_offset,
		projects_offset => $projects_offset,
		news_offset => $news_offset,
		library_offset => $library_offset
	});
	
	return 0;
}

sub showRSS {
	my $r = CyberArmy::WWW::Request->instance();

	my $link = $r->getServerLink().'/';

	my $rss = new XML::RSS (version => '1.0');
	$rss->channel(
		title		=> 'CyberArmy Headlines',
		link		=> $link,
		description	=> 'The CyberArmy headlines',
		dc			=> {
			publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
			language	=> 'en-US'
		},
		syn			=> {
			updatePeriod     => 'daily',
			updateFrequency  => 1,
		}
	);

	$rss->image(
		title		=> 'CyberArmy',
		url			=> $link.'includes/header/cyberarmy.gif',
		link		=> $link,
	);

	my $select = "author,subject,DATE_FORMAT(rdate,'%Y-%m-%dT%H:%i:%s') As w3cdate";
	foreach ( @{ CyberArmy::Forum->getHeadlines({
					count => '10',
					select => $select}) } ) {
		$rss->add_item(
			title	=> $_->{'subject'},
			link	=> $link.'forum/'.$_->{'forum'}.
							'/messages/'.$_->{'mid'}.'.html',
			dc		=> {
				creator	=> $_->{'author'},
				subject	=> $_->{'forum'},
				date	=> $_->{'w3cdate'},
			}
		);
	}
	$r->content_type('application/xml');
	$r->print( $rss->as_string );
	
	return 0;
}

sub showProjectsRSS {
	my $r = CyberArmy::WWW::Request->instance();

	my $link = $r->getServerLink().'/';

	my $rss = new XML::RSS (version => '1.0');
	$rss->channel(
		title		=> 'CyberArmy Project News',
		link		=> $link,
		description	=> 'The CyberArmy Project News',
		dc			=> {
			publisher	=> $CyberArmy::Config{'DINAH_ADMIN'},
			language	=> 'en-US'
		},
		syn			=> {
			updatePeriod     => 'daily',
			updateFrequency  => 1,
		}
	);

	my $args = {
		limit => 10,
		select => 'proj_group,news_title,news_poster,news_title,'.
			'DATE_FORMAT(news_date,"%Y-%m-%dT%H:%i:%s") As w3cdate',
	};

	foreach (@{CyberArmy::Projects->GetLatest($args)}) {
		$rss->add_item(
			title	=> $_->{'proj_group'}.': '.$_->{'news_title'},
			link	=> $link.'projects/'.$_->{'proj_group'},
			dc		=> {
				creator	=> $_->{'news_poster'},
				subject	=> $_->{'news_title'},
				date	=> $_->{'w3cdate'}
			}
		);
	}
	$r->content_type('application/xml');
	$r->print( $rss->as_string );
	
	return 0;
}

sub showProfile {

	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;

	my %update = qw( showname showname password passwd private_email prvemail
	public_email pubemail public_pgp_key pgpkey public_plan publicplan
	away away signature signature);

	my $params = $r->getParams({ escapehtml => 1 });

	if ($r->method eq 'POST') {
		$r->checkReferer($r->location. '/profile/') or return 412;
		my $posted = $r->getParams({from => 'posted', escapehtml => 1});
		if ($posted->{'section'}) {
			return 404 unless ($update{$posted->{'section'}} || $posted->{'section'} eq 'login')
		}
		return 404 unless $posted->{'section'};
	    my $message = $posted->{'section'}.' successfully updated';
		## start with updates that require special processing
		if ($posted->{'section'} eq 'password') {
		
			if ($user->Passhash($posted->{'old_password'}) &&
				($posted->{'new_password'} && $posted->{'cfm_password'}) &&
				($posted->{'new_password'} eq $posted->{'cfm_password'})) {

				$Data::Password::MINLEN = 6;
				$Data::Password::MAXLEN = 0;
				my $result;
				if ($result = Data::Password::IsBadPassword($posted->{'cfm_password'})) {
					$result =~ s/^(.)/uc($1)/e;
					$r->printTemplate('my/profile/layout.tmpl', {
						userbox => _getUserInfo(),
						section => 'password',
						view => 'password',
						signature => $user->signature,
						publicplan => $user->publicplan,
						pgpkr => $user->getPGPKeyRing,
						away => $user->away,
						status => 'failed',
						error => $result,
						login=>($r->getParams({from=>'cookies'})) });
					return 0;
				}
				$message = 'You password was successfully updated';
				$posted->{'password'} = $posted->{'cfm_password'};
			} else {
					$r->printTemplate('my/profile/layout.tmpl', {
						userbox => _getUserInfo(),
						section => 'password',
						view => 'password',
						signature => $user->signature,
						publicplan => $user->publicplan,
						pgpkr => $user->getPGPKeyRing,
						away => $user->away,
						login=>($r->getParams({from=>'cookies'})),
						status => 'failed',
						error => "Failed to update password. Ensure your existing password is entered correctly, the new password is entered twice and is 6 or more characters in length." });
					return 0;
			}
		} elsif ($posted->{'section'} eq 'login') {
				$message = 'You login options were successfully updated';
			CyberArmy::WWW::Login->loginOptions();
			
					$r->printTemplate('my/profile/layout.tmpl', {
						userbox => _getUserInfo(),
						section => 'login',
						view => 'login',
						signature => $user->signature,
						publicplan => $user->publicplan,
						pgpkr => $user->getPGPKeyRing,
						away => $user->away,
						message => $message,
						login=> {login_timeout=>$posted->{'timeout'} ? $posted->{'timeout'} : 2, 
							login_mode=>$posted->{'mode'} ? 'secured' : 'normal', 
							login_method=>$posted->{'method'} ? $posted->{'method'} : 'default'} });
					return 0;
		} elsif ($posted->{'section'} eq 'away') {
				$message = 'You away status was successfully updated';
			$posted->{'away'} = $posted->{'away'} 
				&& $posted->{'away'} eq 'yes' ? time : undef;
		} elsif ($posted->{'section'} eq 'signature') {
			$posted->{'action'} ||= 'update';
			if ($posted->{'action'} eq 'Preview') {
				$r->printTemplate('my/profile/layout.tmpl',{
					userbox => _getUserInfo(),
					section => $posted->{'section'},
				    view => $posted->{'section'},
					action => $posted->{'action'},
					signature => $posted->{'signature'},
					publicplan => $user->publicplan,
					pgpkr => $user->getPGPKeyRing,
					away => $user->away,
					login=>($r->getParams({from=>'cookies'}))
				}); 
				return 0;
			}
			$message = 'You signature was successfully updated';
		} elsif ($posted->{'section'} eq 'public_plan') {
			$posted->{'action'} ||= 'update';
			if ($posted->{'action'} eq 'Preview') {
				$r->printTemplate('my/profile/layout.tmpl',{
					userbox => _getUserInfo(),
					section => $posted->{'section'},
					view => $posted->{'section'},
					action => $posted->{'action'},
					publicplan => $posted->{'public_plan'},
					signature => $user->signature,
					pgpkr => $user->getPGPKeyRing,
					away => $user->away,
					login=>($r->getParams({from=>'cookies'}))
				}); return 0;
			}
		    $message = 'You public plan was successfully updated';
		} elsif ($posted->{'section'} eq 'public_pgp_key') {
			my $pgpkr = $user->getPGPKeyRing;
			my $result = '';

			if ($posted->{'public_pgp_key'}) {
				$result = $pgpkr->Parse($posted->{'public_pgp_key'});
			} else { 
				if ($pgpkr->{'has_key'}) { 
					$result = $pgpkr->Delete() ? 'ok' : 'error';
			        $pgpkr = $user->getPGPKeyRing;
				} else { return 412 }
			}
			$r->printTemplate('my/profile/pgp_update.tmpl',{
				userbox => _getUserInfo , 
			    view => $params->{'view'} ? $params->{'view'} : undef,
				section => $params->{'view'},
				signature => $user->signature,
				publicplan => $user->publicplan,
				pgpkr => $pgpkr,
				away => $user->away,
				login=>($r->getParams({from=>'cookies'})),
				nickname => $user->nickname,
			    result => $result,		
			});
			return 0;
		} else { # Allow for users to have a blank public_plan..
			return 412 unless ( ($posted->{$posted->{'section'}}) || ($posted->{'section'} eq 'public_plan') );
		}

		## do the actual update here
		my $success = $user->Update( 
			$update{$posted->{'section'}} => 
				$posted->{$posted->{'section'}} ); 
		
		$r->printTemplate('my/profile/layout.tmpl',{
			userbox => _getUserInfo, 
			section => $posted->{'section'},
			view => $posted->{'section'},
			status => $success ? 'success' : 'failed',
			signature => $user->signature,
			publicplan => $user->publicplan,
			pgpkr => $user->getPGPKeyRing,
			message => $message,
			away => $user->away,
			login=>($r->getParams({from=>'cookies'}))
		});
	} else {
		$r->printTemplate('my/profile/layout.tmpl',{
			userbox => _getUserInfo , 
		    view => $params->{'view'} ? $params->{'view'} : undef,
			section => $params->{'view'},
			signature => $user->signature,
			publicplan => $user->publicplan,
			pgpkr => $user->getPGPKeyRing,
			away => $user->away,
			login=>($r->getParams({from=>'cookies'}))
		})
	}

	return 0;
}

sub showSkills {

	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;

	my $skillsSet = $user->skillsSet();
	
	if ($r->method eq 'POST') {
		$r->checkReferer($r->location. '/skills/') or return 412;
		my $posted = $r->getParams({from => 'posted', escapehtml => 1});
		if ($posted->{'action'}) {
			if ($posted->{'action'} eq 'add') {
				return 412 unless $posted->{'skill'} && $posted->{'level'};
			} elsif ($posted->{'action'} eq 'del') {
				return 412 unless $posted->{'skill'};
			} else { return 404 }
			$skillsSet->set($posted->{'skill'},$posted->{'level'});
			$r->redirectTo($r->uri);
		} else { return 404  }

	} else {
		my $skills = CyberArmy::SkillsDB->listSkills();
		my $user_skills = $skillsSet->get();
		my $skill_levels = CyberArmy::SkillsDB->getskilllevels();

        $r->printTemplate('my/skillsdb.tmpl',{
			userbox => _getUserInfo,
			skills => $skills,
			user_skills => $user_skills,
			skill_levels => $skill_levels
		});
	}
	
	return 0;
}


sub showMessages {

	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;
	
	my $params = $r->method ne 'POST' ? $r->getParams({ escapehtml => 1 }) :
		$r->getParams({from=>'posted', escapehtml => 1, multi => {msg => 1}} );

	if ($params->{'msg'} && ref $params->{'msg'}) {
		$params->{'msg'} = $params->{'msg'}->[0]
			unless $params->{'msg'}->[1];
	}

	my $box = $user->getMessages();
	
	my %folders = (inbox=>1,sent=>1,unread=>1);
	my $cfolders = $box->GetFolders(0);#Get users' customised folders

	$params->{'action'} ||= 'view';
	if ($params->{'action'} eq 'view') {
		my @mail;
		my $exists;
		if ($params->{'msg'}) {
			my $msg = $box->ViewMsg($params->{'msg'}) or exit(404);
			$msg->{'tostring'} =~ s/(\w+)(;|$)/<a href="\/~$1\/">$1<\/a>$2 /g;
			$msg->{'tostring'} .= '<br />';
			$msg->{'body'} =~ s/^(&gt;.*)$/<i>$1<\/i>/mgi;

			$box->SetMsgAsRead( $params->{'msg'} );
			$r->printTemplate('my/messages/view.tmpl',{
				userbox => _getUserInfo,
				message_id => $params->{'msg'},
				message => $msg,
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
			})
		} else { ## folder view
			if($params->{'folder'}){
				#check if folder = a customised folder		
				$exists = $box->GetFolders(2,$params->{'folder'});
				if(!$exists){
				#folder != customised folder, check if = a default folder
				#if not, set to 'inbox'
					$params->{'folder'} = (($params->{'folder'} 
								&& $folders{$params->{'folder'}})?$params->{'folder'}:'inbox' );
				}
			} else {
				$params->{'folder'} = 'inbox';
			}
			my $page = $params->{'page'}; my $all = 0;
			if (not $page) { $page = 1; }
			elsif (not ($page =~ /^\d+$/)) {
				$all = 1 if ($page eq 'All'); $page = 1;
			}
			my $limit = 20; #number of messages per page

			my $pages = $box->MsgCount($params->{'folder'}) / $limit;
			$pages = int $pages + 1 unless ($pages == int $pages);
			my $cdisplay = $all ? 'All' : $page;
	
			my $stickies;
			my $messages;
			if ($all) { 
				$stickies = $box->ViewMsgList(1,$params->{'folder'});
				$messages = $box->ViewMsgList(0,$params->{'folder'});
			} elsif ($page == 1) {
				$stickies = $box->ViewMsgList(1,$params->{'folder'},$limit);
				if (scalar(@$stickies) < $limit) {
					$messages = $box->ViewMsgList(0,$params->{'folder'},$limit - scalar(@$stickies));
				}
			} elsif ($page > 1 && $page <= $pages) {
				$stickies = $box->ViewMsgList(1,$params->{'folder'},$limit,($page-1) * $limit);
				my $numStickies = $box->NumStickies($params->{'folder'});
				if ($numStickies < ($page * $limit)) {
					$messages = $box->ViewMsgList(0,
						$params->{'folder'},$limit - scalar(@$stickies),(($page-1) * $limit)-$numStickies);
				}
			} else { return 404 }

			my $pageslots = 4;
			my @pages;

			if ($page == 1) {
				push @pages, 'All' unless ($all);
			} else { $pageslots--; push @pages, 1 }

			for (my $i = ($page - (int($pageslots / 2))); $i < $page; $i++) {
				if ($i > 1) { $pageslots--;	push @pages, $i;  }
			}
			push @pages, $page;
			for (my $i = $page + 1; $i < $page + 2; $i++) {
				if ($i < $pages) { $pageslots--; push @pages, $i; }
			}		
			if ($pages && ($pages != $page)) {
				$pageslots--; push @pages, $pages;
			}

			$r->printTemplate('my/messages/folder.tmpl',{
				userbox => _getUserInfo(),
				page => $page,pages => \@pages,
				messages => $messages, cdisplay => $cdisplay,
				selected => (($params->{'select'}||'') eq 'all'),
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
				folder => $params->{'folder'},#current folder
				folderid => $exists,#name to use if folder!= one of the defaults
				stickies => (scalar(@$stickies) > 0) ? $stickies : undef,
			});
		}

	} elsif ($params->{'action'} eq 'delete') {
		if ($params->{'msg'}) {
			$params->{'msg'} = [$params->{'msg'}] unless ref $params->{'msg'};

			my $deleted = ( 
				$r->method eq 'POST' && $params->{'confirmed'}
				&& $r->checkReferer($r->location. '/messages/')
				&& ($box->DeleteMsg($params->{'msg'}) != 0E0)
			) ? 1 : 0;

			$r->printTemplate('my/messages/delete.tmpl',{
				userbox => _getUserInfo(),
				confirmed => $params->{'confirmed'},
				msg => $params->{'msg'}, deleted => $deleted,
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
			});
		} else { return 412 }
	} elsif ($params->{'action'} eq 'download') {
		if ($params->{'msg'}) {
			ref $params->{'msg'} or $params->{'msg'} = [$params->{'msg'}];
			$r->header_out('Content-Disposition' => 'attachment; filename='.
				$user->nickname . '@cyberarmy.mbox');
			$r->content_type('application/mbox');
			$r->printTemplate('my/messages/mbox.tmpl',{
				messages => $box->ViewMsgList(0,$params->{'msg'})
			});
		} else { return 412 }	
	} elsif (
		$params->{'action'} eq 'compose' ||
		$params->{'action'} eq 'reply' ||
		$params->{'action'} eq 'replyall' ||
		$params->{'action'} eq 'forward' ) {

		## submit new messages only via POST
		if ($r->method eq 'POST' && $params->{'to'} 
				&& $params->{'subject'} && $params->{'body'} && !$params->{'preview'}) {
			$r->checkReferer($r->location. '/messages/') or return 412;
			$params->{'to'} =~ s/ //g;
			$params->{'body'} =~ s/\r//gs;
			$params->{'body'} =~ s/\n{3,}/\n\n/gs;
			my $failed = CyberArmy::Message->new(
				nickname => [split /;/ , $params->{'to'}], 
				user => $user, body => $params->{'body'},
				subject => $params->{'subject'},
				mid => $params->{'mid'},
				replied => $params->{'replied'} ? 1 : undef,
				forwarded => $params->{'forwarded'} ? 1 : undef
			);

			$r->printTemplate('my/messages/sent.tmpl',{
				userbox => _getUserInfo(),
				error_rcpt => ref$failed ? $$failed : ''
			
			})
		} else {
			my $to = $params->{'to'} || '';
			my $subject = $params->{'subject'} || '';
			my $body = $params->{'body'} || '';

			if ( $params->{'action'} ne 'compose' && $params->{'msg'} &&	
					(my $msg = $box->ViewMsg($params->{'msg'})) ) {
				$msg->{'body'} =~ s/\n/\n&gt;/g;
				$body = "On $msg->{rdate}, $msg->{sender_rank} $msg->{sender_name}"
						." wrote \n&gt;$msg->{body}";

				if ($params->{'action'} =~ /^reply/) {
					if ($params->{'action'} eq 'replyall') {
						my %recpt = map { $_ => 1 } (
							split /;/ ,$msg->{'tostring'}
						), $msg->{'sender_name'};
						delete $recpt{$user->nickname};
						$to = join (';',keys(%recpt));
					} else { $to = $msg->{'sender_name'} }
					$subject = $msg->{'subject'} =~ /^RE:/ ? 
						$msg->{'subject'} : 'RE: '.$msg->{'subject'};
				} elsif ($params->{'action'} eq 'forward') {
					$subject = $msg->{'subject'} =~ /^FWD:/ ? 
						$msg->{'subject'} : 'FWD: '.$msg->{'subject'};
				}
			}
			my $message;
			if($params->{'send'})
			{
				$message = "Required fields missing.";
			}
			my $previewbody = $params->{'body'};
			if($previewbody)
			{
				$previewbody =~ s/\r//gs;
				$previewbody =~ s/\n{3,}/\n\n/gs;
			}
			
			#some messing around to check for extra passed params
			#as the message may have been previewed and hence have
			#different fields being passed - to ensure it is correctly
			#marked as forwarded/replied to.
			$r->printTemplate('my/messages/compose.tmpl',{
				userbox => _getUserInfo, to => $to,
				previewsubject=> $params->{'preview'} ? $params->{'subject'} : undef,
				previewbody=> $previewbody ? $previewbody : undef,
				subject => $subject, body => $body,
				message => $message ? $message : undef,
				mid => ($params->{'msg'} && $params->{'action'} ne 'compose' ) ? $params->{'msg'} : ($params->{'mid'} ? $params->{'mid'} : undef),
				replied => ($params->{'action'} eq 'reply' ||$params->{'action'} eq 'replyall') ? 1 : ($params->{'replied'} ? 1 : undef),
				forwarded => ($params->{'action'} eq 'forward') ? 1 : ($params->{'forwarded'} ? 1 : undef),
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
			
			});
		}
	} elsif (
		$params->{'action'} eq 'open' ||
		$params->{'action'} eq 'close') {

		$r->checkReferer($r->location. '/messages/')
			or return 412;

		if ($params->{'msg'}) {

			$box->SetMsgAsRead($params->{'msg'},
				$params->{'action'} eq 'open' ? 'y' : 'n'
			);
			$r->redirectTo(
				$params->{'action'} eq 'open' ? $r->header_in('Referer') : $r->location.'/messages/'
			);

		} else { return 412 }
	} elsif (
		$params->{'action'} eq 'sticky' ||
		$params->{'action'} eq 'unsticky') {

		$r->checkReferer($r->location. '/messages/')
			or return 412;

		if ($params->{'msg'}) {

			$box->SetMsgAsSticky($params->{'msg'},
				$params->{'action'} eq 'sticky' ? 'y' : 'n'
			);
			$r->redirectTo(
				$params->{'action'} eq 'sticky' ? 
					$r->header_in('Referer') : $r->location. '/messages/'
			);

		} else { return 412 }
	} elsif ($params->{'action'} eq 'search'){

		##Perform search if necessary
		my $search; if ($params->{'terms'}) {
			return 412 unless
				length($params->{'terms'}) > 2 &&
				$r->checkReferer($r->location. '/messages/');

			$params->{'caID'} = $user->caID;
			$search = CyberArmy::Message::Search->new($params)
				or return 500;
		}
		#Get the current year
		my (undef,undef,undef,undef,undef,$year,undef,undef) = gmtime(time());
		$year += 1900;
		$r->printTemplate('my/messages/search.tmpl',{
			userbox => _getUserInfo,
			folders => [keys %folders],#default folders
			cfolders => $cfolders, #customised folders
			thisyear => $year,
			terms => $params->{'terms'} ? $params->{'terms'} : undef,#search terms, if set
			results => $search,
			match => $params->{'match'} ? $params->{'match'} : undef,#match type, if set
			in_msg => (!$params->{'in_msg'} && $params->{'terms'}) ? 'OFF' : $params->{'in_msg'},#do/don't search in msgs, if set
			in_subjs => (!$params->{'in_subjs'} && $params->{'terms'}) ? 'OFF' : $params->{'in_subjs'},#do/don't search in subjects, if set
			in_auth => $params->{'in_auth'} ? $params->{'in_auth'} : undef,#do/don't search in authors, if set
			num => $params->{'num'} ? $params->{'num'} : undef,#num results to display, if set
			date => $params->{'date'} ? $params->{'date'} : undef,
			day_1 => $params->{'day_1'} ? $params->{'day_1'} : undef,
			month_1 => $params->{'month_1'} ? $params->{'month_1'} : undef,
			year_1 => $params->{'year_1'} ? $params->{'year_1'} : undef,
			day_2 => $params->{'day_2'} ? $params->{'day_2'} : undef,
			month_2 => $params->{'month_2'} ? $params->{'month_2'} : undef,
			year_2 => $params->{'year_2'} ? $params->{'year_2'} : undef
			
		});
	
	} elsif($params->{'action'} eq 'move'){
		if($params->{'folder'} && $params->{'confirmed'}){ #folder set, do move
			my $message="Moving message(s) failed.";
			if($box->MoveMessage($params->{'msg'},$params->{'folder'})!=0E0){
				$message="Message(s) successfully moved.";
			}
			$r->printTemplate('my/messages/move.tmpl',{
				userbox => _getUserInfo,
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
				confirm => 1,
				message => $message #success/failure message
			});	
		} else { #folder yet to be set, display menu
			$r->printTemplate('my/messages/move.tmpl',{
				userbox => _getUserInfo,
				folders => [keys %folders],#default folders
				cfolders => $cfolders,#customised folders
				msg => $params->{'msg'},
				confirm => 0
			});	
		}
	} elsif ($params->{'action'} eq 'folders'){
		my $dfolders = $box->GetFolders(1); #get deletable folders (ones that are empty)
		$r->printTemplate('my/messages/edit_folders.tmpl',{
			userbox => _getUserInfo,
			folders => [keys %folders],#default folders
			deletable => $dfolders,#deletable (empty) customised folders
			cfolders => $cfolders #customised folders
		});
	} elsif ($params->{'action'} eq 'del_folder'){
		my $message = "Couldn't delete the selected folder.";
		if($params->{'folder_id'}){
			if($box->DeleteFolder($params->{'folder_id'})!=0E0){
				$message = "Folder successfully deleted.";
			}
		}
		my $dfolders = $box->GetFolders(1); #get deletable folders (ones that are empty)
		$cfolders = $box->GetFolders(0); #get updated list
		
		$r->printTemplate('my/messages/edit_folders.tmpl',{
			userbox => _getUserInfo,
			folders => [keys %folders],#default folders
			deletable => $dfolders,#deletable (empty) customised folders
			cfolders => $cfolders,#customised folders
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'rename_folder'){
		my $message="Couldn't rename the selected folder.";
		if($params->{'old_name'} && $params->{'new_name'}){
			if(!$folders{$params->{'new_name'}}){#can't use the default folder names
				if($box->RenameFolder($params->{'old_name'},$params->{'new_name'})!=0E0){
					$message = "Folder successfully renamed.";
				}
			}
			else{
				$message.=" Folder name reserved.";
			}
		}
		my $dfolders = $box->GetFolders(1); #get deletable folders (ones that are empty)
		$cfolders = $box->GetFolders(0); #get updated list
		
		$r->printTemplate('my/messages/edit_folders.tmpl',{
			userbox => _getUserInfo,
			folders => [keys %folders],#default folders
			deletable => $dfolders,#deletable (empty) customised folders
			cfolders => $cfolders,#customised folders
			message => $message#success/failure message
		});
	} elsif ($params->{'action'} eq 'add_folder'){
		my $message="Couldn't create folder.";
		if($params->{'folder_name'}){
			if(!$folders{$params->{'folder_name'}}){#can't use the default folder names
				if($box->CreateFolder($params->{'folder_name'},$user->caID)!=0E0){
					$message="Folder successfully created.";
				}
			}
			else{
				$message.=" Folder name reserved.";
			}
		}
		my $dfolders = $box->GetFolders(1); #get deletable folders (ones that are empty)

		$cfolders = $box->GetFolders(0); #get updated list
		
		$r->printTemplate('my/messages/edit_folders.tmpl',{
			userbox => _getUserInfo,
			folders => [keys %folders],#default folders
			deletable => $dfolders,#deletable (empty) customised folders
			cfolders => $cfolders,#customised folders
			message => $message#success/failure message
		});
	} else { return 404 }

	return 0;
}

sub listProjects {
	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;
		
	$r->printTemplate('my/projects/list.tmpl',{
		userbox => _getUserInfo,
		user => $user
	});

	return 0;
}

sub showTodo {

	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;
	
	my $todoList = $user->todoList();
	if ($r->method eq 'POST') {
		$r->checkReferer($r->location. '/todo/') or return 412;
		my $posted = $r->getParams({from => 'posted', escapehtml => 1});
		if ($posted->{'action'}) {
			if ($posted->{'action'} eq 'add') {
				if ($posted->{'task'}) {
				        #delete leading white space
					$posted->{'task'} =~ s/^\s+//;
					#delete trailing white space
					$posted->{'task'} =~ s/\s+$//;
					$posted->{'task'} = substr($posted->{'task'}, 0, 4096)
						if (length($posted->{'task'}) > 4096);
					$todoList->add($posted->{'task'});
				} else { return 412 }
			} elsif ($posted->{'action'} eq 'del') {
				if ($posted->{'id'}) {
					$todoList->del($posted->{'id'});
				} else { return 412 }
			} else { return 404 }
			$r->redirectTo( $r->uri );
		} else { return 404 }
	} else {
		my $projects = $user->getProjects();
		my $todolist =$todoList->get();
		my $projTodolist =$todoList->get("project");
		
		$r->printTemplate('my/todo.tmpl',{
			userbox => _getUserInfo, 
			todoes => scalar(@$todolist)>0 ? $todolist : undef,#normal user ToDos
			projectTodoes => scalar(@$projTodolist)>0 ? $projTodolist : undef, #all project ToDos for this user
			projects => $projects ? $projects : undef
		});
	}
	
	return 0;
}

sub recoverPassword {
	my $r = CyberArmy::WWW::Request->instance();
	$r->redirectTo($r->uri .'/') unless ($r->uri =~ m{/$});
	my $params = $r->getParams({ escapehtml => 1});
	
	if ($params->{'confirm'}) {
		my $db = CyberArmy::Database->instance();
		my $recover = $db->selectrow_arrayref(
			'SELECT confirm,caID,retrieved FROM script_password WHERE confirm = ?',
		undef,$params->{'confirm'});
		
		if ($recover) {
			my $user = CyberArmy::User->new(caID => $recover->[1])
				or return 500;

			if ($recover->[2] == 0) {
				if ($user->Update( passwd => 
						( my $password = CyberArmy::Utils::RandomAlphaNum(8) ))) {
	
					$user->Log( type => 'passreset', logby => '/my/password',
						msg => 'From '.$r->connection->remote_ip);
	
					CyberArmy::Template->instance->process(
						'my/password/password_email.tmpl',
						{ password => $password }, \my $body
					);
	
					CyberArmy::Mail::Send(
						to => [$user->prvemail,$user->showname], body=> $body,
						subject => 'Password Reset ('. $user->nickname. ')'
					);

					$db->do(
						'UPDATE script_password SET retrieved=1 WHERE confirm=?',
						undef,$recover->[0]
					) or return 500;

					$r->printTemplate('my/password/reset_ok.tmpl',
							{userbox => _getUserInfo });
				} else {
					return 500;
				}
			} else {
				$r->printTemplate('my/password/already_sent.tmpl',{
					userbox => _getUserInfo
				});
			}
		} else {
			$r->printTemplate('my/password/conf_invalid.tmpl',{
				userbox => _getUserInfo
			});
		}
	} else {
		if ($r->method eq 'POST') {
			my $posted = $r->getParams({from=>'posted', escapehtml => 1});
			$posted->{'nickname'} && $posted->{'private_email'} or return 412;

			my $user = CyberArmy::User->new(nickname=>$posted->{'nickname'});
			if ($user && ($user->prvemail eq $posted->{'private_email'})) {
				my $remote_ip = $r->connection->remote_ip;
				my $confirm = CyberArmy::Utils::RandomAlphaNum(24);
				my $db = CyberArmy::Database->instance();
				if ($db->selectrow_array(
						'SELECT caID from script_password '.
						'WHERE caID = ? OR remote_ip = ?',
						undef,$user->caID,$remote_ip)) {

					$r->printTemplate(
						'my/password/already_reset.tmpl',
							{ userbox => _getUserInfo});

				} else {
					$db->do(
						'INSERT INTO script_password '.
						'VALUES (?,?,?,NOW(),0)',
						undef,$confirm,$user->caID,$remote_ip
					) or return 500;

					CyberArmy::Template->instance->process(
						'my/password/confirmation_email.tmpl',
						{ confirmation => $confirm, 
							ip => $remote_ip, 
							server => $r->getServerLink()
						}, \my $body
					);

					CyberArmy::Mail::Send(
						to => [$user->prvemail,$user->showname], body=> $body,
						subject => 'Password Reset ('. $user->nickname. ')'
					);

					$r->printTemplate(
						'my/password/confirm.tmpl',{ userbox => _getUserInfo});
				}
			} else {
				$r->printTemplate(
					'my/password/invalid.tmpl',{ userbox => _getUserInfo });
			}
		} else {
			$r->printTemplate(
				'my/password/reset.tmpl',{ userbox => _getUserInfo });
		}
	}
	
	return 0;	
}




sub articles {


	
	my $r = CyberArmy::WWW::Request->instance();
	#actions from url (GET)
	my $actionParams = $r->getParams({ escapehtml => 1 });

	#params passed by a form (POST)
	my $params = $r->getParams({from=>'posted', escapehtml => 1});

	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;
	
	my $categoryList = CyberArmy::Library->getSubCategories(0, 0);
	my $languageList = CyberArmy::Library->listLanguages();
	
		
	if($actionParams->{'action'} && $actionParams->{'action'} eq 'create'){
		if ($params->{'preview'}) {	
		    $r->printTemplate('my/articles/create-article.tmpl',{ 
			userbox => _getUserInfo ,
			categories => $categoryList,
			languages => $languageList,
			title => $params->{'title'},
			preview_summary => $params->{'summary'},
			summary => $params->{'summary'},
			body => $params->{'body'},
			category => $params->{'category'},
			language => $params->{'language'},
			article_proj => $params->{'project'},
			preview => 1
			});
		}
		elsif ($params->{'submit'}) {
			#remove line feeds
			$params->{'body'} =~ s/\r//gs;
			#Reduce line breaks
			$params->{'body'} =~ s/\n{3,}/\n\n/gs;

			if($params->{'title'} && 
				$params->{'summary'} &&
				$params->{'body'} && 
				$params->{'category'} && 
				$params->{'license'}) {

				#check category is valid
				my $category = CyberArmy::Library->getCategory($params->{'category'});
				if($category=='0E0'){
					#category does not exist
					$r->printTemplate('my/articles/create-article.tmpl',{ 
						userbox => _getUserInfo ,
						message => "Category does not exist, it is possible the category was deleted and is no longer available. Please try again.",
						categories => $categoryList,
						languages => $languageList,
						title => $params->{'title'},
						preview_summary => substr($params->{'summary'}, 0, 255),
						summary => $params->{'summary'},
						body => $params->{'body'},
						article_proj => $params->{'project'},
						language => $params->{'language'},
						preview => 1
					});
					return 1;
				}
				#Do initial save of article to db
				my $article_id = CyberArmy::Library->add(
					$user,
					$params->{'title'},
					$params->{'summary'},
					$params->{'body'},
					$params->{'category'},
					$params->{'project'},
					$params->{'language'}
				) or return 404;

				CyberArmy::Library->submit_article(
					$user,
					$article_id
				) or return 404;
				$r->printTemplate('my/articles/submitted.tmpl',{ 
					userbox => _getUserInfo ,
					title => "Article Submitted",
					message => "Your article has been submitted and will be reviewed by an editor. The article is available for you to view in '<a href=\"/my/articles/\">Your Articles</a>'."});
			} else {
				my $message ="Mandatory Field(s) Missing - Please try again.";
				if($params->{'title'} && 
					$params->{'summary'} &&
					$params->{'body'} && 
					$params->{'category'})
					{
						$message = "You must agree to the CyberArmy Library Contributor Agreement in order to submit articles.";
					}
			       $r->printTemplate('my/articles/create-article.tmpl',{ 
					userbox => _getUserInfo ,
					message => $message,
					categories => $categoryList,
					languages => $languageList,
					title => $params->{'title'},
					preview_summary => substr($params->{'summary'}, 0, 255),
					summary => $params->{'summary'},
					body => $params->{'body'},
					category => $params->{'category'},
					language => $params->{'language'},
					preview => 1
					});

			}

		}
		elsif ($params->{'draft'}) {

			#remove line feeds
			$params->{'body'} =~ s/\r//gs;
			#Reduce line breaks
			$params->{'body'} =~ s/\n{3,}/\n\n/gs;

			if($params->{'title'} && 
				$params->{'summary'} &&
				$params->{'body'} && 
				$params->{'category'} && 
				$params->{'license'}) {

				my $category = CyberArmy::Library->getCategory($params->{'category'});
				if($category=='0E0'){
					#category does not exist
					$r->printTemplate('my/articles/create-article.tmpl',{ 
						userbox => _getUserInfo ,
						message => "Category does not exist, it is possible the category was deleted and is no longer available. Please try again.",
						categories => $categoryList,
						languages => $languageList,
						title => $params->{'title'},
						preview_summary => substr($params->{'summary'}, 0, 255),
						summary => $params->{'summary'},
						body => $params->{'body'},
						language => $params->{'language'},
						article_proj => $params->{'project'},
						preview => 1
					});
					return 1;
				}

				CyberArmy::Library->add(
					$user,
					$params->{'title'},
					$params->{'summary'},
					$params->{'body'},
					$params->{'category'},
					$params->{'project'},
					$params->{'language'}
				) or return 404;


				$r->printTemplate('my/articles/submitted.tmpl',{ 
					userbox => _getUserInfo ,
					title => "Article Saved",
					message => "Your article has been saved and is available in '<a href=\"/my/articles/\">Your Articles</a>'."});
			} else {
				my $message ="Mandatory Field(s) Missing - Please try again.";
				if($params->{'title'} && 
					$params->{'summary'} &&
					$params->{'body'} && 
					$params->{'category'})
					{
						$message = "You must agree to the CyberArmy Library Licensing Agreement in order to submit articles.";
					}

			       $r->printTemplate('my/articles/create-article.tmpl',{ 
					userbox => _getUserInfo ,
					message => $message,
					categories => $categoryList,
					languages => $languageList,
					title => $params->{'title'},
					preview_summary => substr($params->{'summary'}, 0, 255),
					summary => $params->{'summary'},
					body => $params->{'body'},
					category => $params->{'category'},
					language => $params->{'language'},
					preview => 1
					});

			}
		}
		else {

		   $r->printTemplate('my/articles/create-article.tmpl',{ userbox => _getUserInfo,
					categories => $categoryList,
					languages => $languageList});
		}
	} 
	elsif($actionParams->{'action'} && $actionParams->{'action'} eq 'delete'){
		if ($actionParams->{'id'}) {	
			my $article = CyberArmy::Library->getArticle($actionParams->{'id'});
			if($article!=0E0){
		   		$r->printTemplate('my/articles/delete_draft.tmpl',{ 
					userbox => _getUserInfo ,
		   		article => $article});
					}
			else { return 404 }
					
		}
		else { return 404 }
	
	} 
	elsif($actionParams->{'action'} && $actionParams->{'action'} eq 'confirm_del'){
		if ($actionParams->{'id'}) {	
			CyberArmy::Library->deleteDraft($actionParams->{'id'}, $user);
			
			my $draft_articles = CyberArmy::Library->getArticles(
						$user,
						'draft'
						);
			my $review_articles = CyberArmy::Library->getArticles(
						$user,
						'review'
						);
			my $live_articles = CyberArmy::Library->getArticles(
						$user,
						'live'
						);
		   my $pubonly_articles = CyberArmy::Library->getArticles(
					$user,
					'publication-only'
					);
			$r->printTemplate('my/articles/list.tmpl',{ 
				userbox => _getUserInfo,
				draft_articles=> $draft_articles,
				review_articles => $review_articles,
				live_articles=> $live_articles,
			    pubonly_articles => $pubonly_articles});
					
		}
		else {
			return 404;
		}
	
	} else {
		my $draft_articles = CyberArmy::Library->getArticles(
					$user,
					'draft'
					);
		my $review_articles = CyberArmy::Library->getArticles(
					$user,
					'review'
					);
		my $live_articles = CyberArmy::Library->getArticles(
					$user,
					'live'
					);
		my $pubonly_articles = CyberArmy::Library->getArticles(
					$user,
					'publication-only'
					);
		$r->printTemplate('my/articles/list.tmpl',{ 
			userbox => _getUserInfo,
			draft_articles=> $draft_articles,
			review_articles => $review_articles,
			live_articles=> $live_articles,
			pubonly_articles => $pubonly_articles});
	
	}
	
	return 0;

}

sub accomp {
    my $r = CyberArmy::WWW::Request->instance();
    #actions from url (GET)
    my $actionParams = $r->getParams({ escapehtml => 1 });
    my $posted = $r->getParams({from => 'posted', escapehtml => 1, multi => { categs => 1 } } );

	my $user = CyberArmy::WWW::Request::User
		->instance() or return 403;

	my $staff_access = ($user->CheckGroupList('staff'))?1:0;
	return 403 if !$staff_access;

	### [More information] button
	###### Permission: can_see_acl on accomp
    if (defined $actionParams->{'info'}) { 
        my($queryid) = $actionParams->{'info'};
		return 403 if ($queryid =~ /\D/ || !CyberArmy::Accomplishments->canSee($queryid, $user));

        my $acc_details = CyberArmy::Accomplishments->getAccomplishmentDetails($queryid);
        $r->printTemplate('my/accomplishments/info.tmpl', { userbox => _getUserInfo, Accomplishment => $acc_details, staff_access => $staff_access } );

        return 0;
    }

	### Delete Accomplishment
	###### Permission: staff
    if (defined $actionParams->{'action'} && $actionParams->{'action'} eq 'deleAccomp') {
        my $acc_id = $actionParams->{'acc_id'};
		return 403 if ($acc_id =~ /\D/ || !$user->CheckGroupList('staff'));
        if (!$actionParams->{'confirmed'} == 1) {
            my $acc_details = CyberArmy::Accomplishments->getAccomplishmentDetails($acc_id);
            $r->printTemplate('my/accomplishments/delete_feature.tmpl', { 
                userbox => _getUserInfo, 
                acc_id => $acc_id, 
                Achievement => $acc_details
            });
            return 0;
        }
        else {
            CyberArmy::Accomplishments->deleteAccomplishmentCategories($acc_id);
        }
    }

	### Create New Accomplishment
	###### Permission: staff
    if (defined $actionParams->{'action'} && $actionParams->{'action'} eq 'createAccomp' && !$posted->{'newtitle'}) {
		return 403 if (!$user->CheckGroupList('staff'));

		# No need to filter, staff can see all groups
        my $categories = CyberArmy::Accomplishments->getCategories($user->caID);
        $r->printTemplate('my/accomplishments/edit_accomp.tmpl', { userbox => _getUserInfo, Categories => $categories } );
        
        return 0;
    }
	
	### Award Accomplishments
	###### Permission: can_award_acl or staff
    if (defined $actionParams->{'action'} && $actionParams->{'action'} eq 'awardAccomp') {
		my($note, $adduser, $acc_id, $categ, @categoryList, $accomplishmentsList, $completedList);

		my $categories = CyberArmy::Accomplishments->getCategories(
			$user->caID
		);

		foreach $categ (@$categories) {
			next if (!CyberArmy::Accomplishments->canSeeWithDetails($categ, $user));

			push(@categoryList, ({
				category => $categ->{id},
				category_name => $categ->{name},
				category_displayname => $categ->{displayname},
				category_desc => $categ->{short_desc}
			}));
		}

		$categ = $actionParams->{'categ'};
		$acc_id = $actionParams->{'accomp'};
		$adduser = $actionParams->{'adduser'};
		$note = $actionParams->{'note'};

		# If they have asked us to grant it to a user 
		if ($adduser && $acc_id) {
			return 403 if ($acc_id =~ /\D/ || !CyberArmy::Accomplishments->canAward($acc_id, $user));
			my $adduser_obj = CyberArmy::User->new(nickname => $adduser);
	        return 500 if !$adduser_obj;

			CyberArmy::Accomplishments->awardOnce(
	            $adduser_obj->caID,
	            $acc_id,
				$note,
				$user->caID
	        );
		}

		# get accomps in categ for display, filter so only accessible ones are seen
		if ($categ) {
			my $details = CyberArmy::Accomplishments->getCategoryByName($categ);
			return 403 if (!$details->[0] || !CyberArmy::Accomplishments->canSeeWithDetails($details->[0], $user));
			
			my($cid) = CyberArmy::Accomplishments->getCategoryID($categ);
			return 403 if !$cid;
			
			$accomplishmentsList = CyberArmy::Accomplishments->getAccomplishmentsFromCategory($cid);
			$accomplishmentsList = CyberArmy::Accomplishments->filterAccomplishments($accomplishmentsList, $user);

			# If they've selected an accomp, get the list of users who have it.
			if ($acc_id) {
				return 403 if ($acc_id =~ /\D/ || !CyberArmy::Accomplishments->canAward($acc_id, $user));
				$completedList = CyberArmy::Accomplishments->getWhoEarnedAccomp($acc_id);
			}
		}

        $r->printTemplate('my/accomplishments/award_accomp.tmpl', {
			userbox => _getUserInfo,
			categ => $categ,
			acc_id => $acc_id,
			Categories => \@categoryList, 
			AccompList => $accomplishmentsList,
			CompletedList => $completedList
		}
		);
        
        return 0;
    }

	### Edit / Create Accomplishment
	###### Permission: staff
    if (defined $actionParams->{'action'} && ($actionParams->{'action'} eq 'editAccomp' || $actionParams->{'action'} eq 'createAccomp')) {
        my $acc_id = $actionParams->{'acc_id'};
		return 403 if ($acc_id =~ /\D/ || !$user->CheckGroupList('staff'));

        my($newtitle, $newimageurl_before, $newimageurl_after, $newshort_desc, $newcan_see, $newcan_award, $newidempotent, $newlong_desc) = (
            $posted->{'newtitle'}, $posted->{'newimageurl_before'}, $posted->{'newimageurl_after'}, $posted->{'newshort_desc'},
            $posted->{'newcan_see'}, $posted->{'newcan_award'}, $posted->{'newidempotent'}, $posted->{'newlong_desc'}
        );

        if (!$acc_id && $newtitle && ($#{$posted->{'categs'}} > -1)) { # new creation
             CyberArmy::Accomplishments->createAccomplishment(
                title => $newtitle,
                imageurl_before => $newimageurl_before,
                imageurl_after => $newimageurl_after,
                short_desc => $newshort_desc,
                can_see_acl => $newcan_see,
                can_award_acl => $newcan_award,
                idempotent => $newidempotent,
                long_desc => $newlong_desc
            );
            $acc_id = CyberArmy::Accomplishments->getAccomplishmentIdByTitle($newtitle);
            return 500 if !$acc_id;
        
            CyberArmy::Accomplishments->deleteAccomplishmentCategories($acc_id);

            my($addcateg);
            foreach $addcateg (@{$posted->{'categs'}}) {
                CyberArmy::Accomplishments->linkAccomplishmentToCategory($acc_id, $addcateg);
            }
        }
        else {   
            return 500 if !$acc_id;

            if ($newtitle && ($#{$posted->{'categs'}} > -1)) {
                CyberArmy::Accomplishments->setAccomplishmentDetails($acc_id,
                    title => $newtitle,
                    imageurl_before => $newimageurl_before,
                    imageurl_after => $newimageurl_after,
                    short_desc => $newshort_desc,
                    can_see_acl => $newcan_see,
                    can_award_acl => $newcan_award,
                    idempotent => $newidempotent,
                    long_desc => $newlong_desc
                );

                CyberArmy::Accomplishments->deleteAccomplishmentCategories($acc_id);

                my($addcateg);
                foreach $addcateg (@{$posted->{'categs'}}) {
                    CyberArmy::Accomplishments->linkAccomplishmentToCategory($acc_id, $addcateg);
                }
            }
            else {
                my $categories = CyberArmy::Accomplishments->getCategories($user->caID);
                my $acc_details = CyberArmy::Accomplishments->getAccomplishmentDetails($acc_id);
                $r->printTemplate('my/accomplishments/edit_accomp.tmpl', { userbox => _getUserInfo, Accomplishment => $acc_details, Categories => $categories } );

                return 0;
            }
        }
    }

	### Edit Category
	###### Permission: staff
    if (defined $actionParams->{'action'} && $actionParams->{'action'} eq 'editGroup') {
		return 403 if (!$user->CheckGroupList('staff'));

        my $categ = $actionParams->{'categ'};
        return 500 if !$categ;

        my($newcateg, $newdisplayname, $newsdesc, $newcan_see, $newcan_award) = (
            $posted->{'newcateg'}, $posted->{'newdisplayname'}, 
            $posted->{'newsdesc'}, $posted->{'newcan_see'}, $posted->{'newcan_award'}
        );

        if ($newcateg && $newdisplayname) {
            $categ = $posted->{'categ'};
            ## Security check here
            ######
            CyberArmy::Accomplishments->updateCategoryByName($categ,
                name => $newcateg,
                displayname => $newdisplayname,
                short_desc => $newsdesc,
                can_see_acl => $newcan_see,
                can_award_acl => $newcan_award
            );
        }
        else {
            my $categoryList = CyberArmy::Accomplishments->getCategoryByName($categ);
            $r->printTemplate('my/accomplishments/edit_group.tmpl', { userbox => _getUserInfo, Category => $categoryList } );

            return 0;
        }
    }

	### Create Category
	###### Permission: staff
    if (defined $actionParams->{'action'} && $actionParams->{'action'} eq 'createGroup') { 
		return 403 if (!$user->CheckGroupList('staff'));

        my $categ = $actionParams->{'categ'};
        my $displayname = $actionParams->{'displayname'};
        my $can_see = $actionParams->{'can_see'};

        if ($categ && $displayname) {
            CyberArmy::Accomplishments->createCategory(
                name => $categ,
                displayname => $displayname,
                short_desc => '',
                can_see_acl => $can_see,
                can_award_acl => ''
            );
        }
        else {
            $r->printTemplate('my/accomplishments/create_group.tmpl', { userbox => _getUserInfo });
            return 0;
        }
    }


    ### If we get here, display a list for the user
	###############################################

	my $categories = CyberArmy::Accomplishments->getCategories(
	    $user->caID
	);

    my $categ;
    my $accomplishmentsEarned;
    my $accomplishmentsNotEarned;
    my @categoryList;
    my %accomplishmentsEarnedCateg;
    my %accomplishmentsNotEarnedCateg;

    foreach $categ (@$categories) {
		next if (!CyberArmy::Accomplishments->canSeeWithDetails($categ, $user));

        my $accomplishmentsEarned = CyberArmy::Accomplishments->getAccomplishmentsEarnedFromCategory(
            $user->caID,
            $categ->{id}
        );

        my $accomplishmentsNotEarned = CyberArmy::Accomplishments->getAccomplishmentsNotEarnedFromCategory(
            $user->caID,
            $categ->{id}
        );

		$accomplishmentsNotEarned = CyberArmy::Accomplishments->filterAccomplishments($accomplishmentsNotEarned, $user);

        push(@categoryList, ({
            category => $categ->{id},
            category_name => $categ->{name},
            category_displayname => $categ->{displayname},
            category_desc => $categ->{short_desc},
            AccomplishmentsEarned => $accomplishmentsEarned,
            AccomplishmentsNotEarned => $accomplishmentsNotEarned
        }));
    }

    $r->printTemplate('my/accomplishments/list.tmpl', { userbox => _getUserInfo, Categories => \@categoryList, staff_access => $staff_access } );

	return 0;
}

1;
