#!perl -wW
package CyberArmy::User::PGPKey;

use strict;
use IPC::Open3;
use File::Temp qw(tempfile);

use CyberArmy::User;
use CyberArmy::Database;

$__PACKAGE__::VERSION = '0.2';

our $AUTOLOAD;

sub CyberArmy::User::getPGPKeyRing {
	my $self = { 'caID' => shift->caID };
	bless $self , __PACKAGE__;
	_load($self);
	return $self;
}

sub GetExpireWithin {
	my $class = shift;
	my $time = shift;

	my $result;
	my $db = CyberArmy::Database->instance();
	my $sth = $db->prepare(
		'SELECT caID,uid,fingerprint,created,expires,pgpkey,warned '.
		'FROM pgpkey WHERE expires > 0 AND expires < ? AND NOT warned'
	);
	$sth->execute(time() + $time);
	my $rows = $sth->fetchall_hashref('caID');
	$sth->finish();

	my @objs;
	foreach my $row (keys %$rows) {
		my $obj = {};
		foreach (keys %{$rows->{$row}}) {
			$obj->{$_} = $rows->{$row}->{$_};
		}
		$obj->{has_key} = 1;
		bless($obj, $class);
		push(@objs, $obj);
	}

	return @objs;
}

sub hasValidKey {
	my $self = shift;
	$self->{has_key} 
		&& (!$self->{expires} || ($self->{expires} > time()));
}



sub Delete {
	my $self = shift;
	my $db = CyberArmy::Database->instance();
	my $ok = $db->do(
		'DELETE FROM pgpkey WHERE caID = ?',
		undef, $self->{caID});

	$self->{has_key} = 0;
	return $ok;
}

sub Warn {
	my $self = shift;
	my $db = CyberArmy::Database->instance();
	$db->do(
		'UPDATE pgpkey SET warned = 1 WHERE caID = ?', 
		undef, $self->{caID}
	) && ($self->{warned} = 1);
}

sub DESTROY {
	my $self = shift;
	_unlink($self);
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/.*://;
	return defined($self->{$AUTOLOAD}) 
		? $self->{$AUTOLOAD} : undef;
}

sub _unlink {
	my $self = shift;

	if (defined($self->{keyring})) {
		unlink($self->{keyring});
		## gpg always makes backups of keystores :\
		unlink($self->{keyring}.'~');
		$self->{keyring} = undef;
	}
}

sub _gpgCommandLine {
	my $self = shift;
	return ($ENV{GPG_PATH}, '--quiet', '--no-secmem-warning',
		'--no-default-keyring', '--keyring', $self->{keyring}, @_);
}

sub _gpgReadWrite {
	my $self = shift;
	open3(*W, *R, undef, _gpgCommandLine($self, @_)) || die($_);
	return (*R, *W);
}

sub _gpgRead {
	my $self = shift;
	open3(undef, *R, undef, _gpgCommandLine($self, @_)) || die($_);
	return *R;
}

sub Parse {
	my ($self, $key) = @_;

        (undef, $self->{keyring}) = tempfile();

	my ($r, $w) = _gpgReadWrite($self, '--import', '--allow-non-selfsigned-ui',
		'--status-fd', '1');
	print $w $key;
	close($w);

	my ($error, $fingerprint, $done) = (undef, undef, undef);
	while (<$r>) {
		next if ($_ !~ /^\[GNUPG:\] (.*)$/);

		my @args = split(' ', $1);
		if ($args[0] eq 'IMPORT_OK') {
			if ($args[1] & 16) {
				$error = 'private_key';
				last;
			} else {
				$fingerprint = $args[2];
			}
		} elsif ($args[0] eq 'IMPORT_RES') {
			$done = 1;
			if ($args[1] > 1) {
				$error = 'too_many';
			} elsif ($args[1] < 1) {
				$error = 'no_key';
			}
			last;
		}
	}
	close($r);

	$error ||= 'misc_error' if (!$done || !$fingerprint);

	if (defined($error)) {
		_unlink();
		return $error;
	}

	$r = _gpgRead($self, '--no-auto-check-trustdb', '--with-key-data',
		'--fixed-list-mode', '--list-keys', $fingerprint);
	my ($created, $expires, $uid) = (undef, undef, undef);
	while (<$r>) {
		my @args = split(':', $_);
		if ($args[0] eq 'pub') {
			if ($args[1] eq 'r') {
				$error = 'revoked';
				last;
			} elsif ($args[1] eq 'e') {
				$error = 'expired';
				last;
			}

			($created, $expires) = ($args[5], $args[6]);
		} elsif ($args[0] eq 'uid') {
			$uid = $args[9];
		}
	}
	close($r);

	if (defined($error)) {
		_unlink();
		return $error;
	}

	$r = _gpgRead($self, '--armor', '--no-emit-version',
		'--export', $fingerprint);
	my @keylines = <$r>;
	close($r);

	splice(@keylines, 1, 0, 'Version: '. __PACKAGE__ 
		. ' '. $__PACKAGE__::VERSION . "\n");
	my $newkey = join('', @keylines);

	$self->{uid} = defined($uid) ? $uid : $fingerprint;
	$self->{fingerprint} = $fingerprint;
	$self->{created} = $created;
	$self->{expires} = $expires;
	$self->{pgpkey} = $newkey;
	$self->{warned} = 0;

	_store($self);
	$self->{has_key} = 1;

	return 'ok';
}

sub _load {
	my ($self) = @_;

	my $result;
	my $db = CyberArmy::Database->instance();
	if (my $row = $db->selectrow_hashref(
		'SELECT uid,fingerprint,created,expires,pgpkey,warned '.
		'FROM pgpkey WHERE caID = ?',
		undef, $self->{caID}))
	{
                foreach (keys %$row) {
			$self->{$_} = $row->{$_};
		}
		$self->{has_key} = 1;
	} else {
		$self->{has_key} = 0;
	}
}

sub _store {
	my ($self) = @_;

	my $db = CyberArmy::Database->instance();
	$db->do('REPLACE INTO pgpkey (caID,uid,fingerprint,created,expires,pgpkey,warned) VALUES (?,?,?,?,?,?,?)',
		undef, $self->{caID}, $self->{uid}, $self->{fingerprint},
		$self->{created}, $self->{expires}, $self->{pgpkey}, $self->{warned});
}

1;
