#!/usr/bin/perl -W
package CyberArmy::Library;

use strict;
use CyberArmy::User;
use CyberArmy::Utils;
use Encode ();

$CyberArmy::Library::VERSION = '0.4_5';

our $AUTOLOAD;

sub DESTROY {
	## Since we are using AUTOLOAD(), we define a DESTROY sub
	## so AUTOLOAD() won't bother looking for it
}

####
#sub add
#
#Create an article
##

sub add {
	my ($class, $user, $title, $summary, $body, $category, $project, $language) = @_;
	if ( $title && $summary && $body && $category){
		if(!$project){
			$project = 'NULL';
		}
		#TODO: check category exists
		my $db = CyberArmy::Database->instance;
		$db->do(
			'INSERT INTO article (author_caID,date_submitted,status,title,summary,body,admin_thread,public_thread,project_id,language_id, category) VALUES (?,NOW(),"draft",?,?,?,"","",?,?,?)',
			undef, $user->caID, $title, $summary, $body, $project, $language, $category);

		my $insertid = $db->{'mysql_insertid'};

		return $insertid
		
	} else { return undef }

}

####
# sub submit_article
#
# Submit an article to be reviewed & released.
# The article will have its status changed to 'review' and a 
# post will be made to the article admin forum.
##
sub submit_article {
	my ($class, $user, $article_id) = @_;
	#get article from db
	
        my $db = CyberArmy::Database->instance;
	
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	
	#post to admin forum
	
	my $rank = "Author";
	my $subject = '['.$article->{'category_title'}.'] '.$article->{'title'};
	my $body = 'View and edit the article here: [url=/library/article/'.$article->{'id'}.']'.$article->{'title'}.'[/url]<br/><br/>'
		.'<hr width="90%"/><center><h3>'.$article->{'title'}.'</h3></center><table width="100%">'
		.'<tr><td valign="top">[b]Category[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'category_title'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Summary[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'summary'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Body[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'body'}.'</td></tr></table></center></td></tr></table>';
		
	my $reportid = CyberArmy::Library->postToAdmin($subject, $body, $user, $rank);
	
	
	
	#update article to 'review' and add admin_thread
	$db->do('UPDATE article SET status = "review", date_submitted=NOW(), admin_thread=? WHERE id = ? ',
		undef, $reportid, $article_id);

	#log
	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'submitted article',
			msg => 'Title: '.$article->{'title'}.' Author: '.$user->showname,
			action_by => $user->showname,
			action_by_caID => $user->caID
		);

}

####
# sub update_article
#
# Update article (e.g. user edited a draft, editor edited submitted article)
##
sub update_article {
	my ($class, $article_id, $title, $summary, $body, $category, $project, $language) = @_;
		$title = Encode::decode_utf8($title);
		$summary = Encode::decode_utf8($summary);
		$body = Encode::decode_utf8($body);
		$category = Encode::decode_utf8($category);
		$language = Encode::decode_utf8($language);
		
        my $db = CyberArmy::Database->instance;
	
	$db->do('UPDATE article SET title = ?, summary=?, body = ?, category=?, project_id=?, language_id=? WHERE id = ?',
			undef, $title, $summary, $body, $category, $project, $language, $article_id);
			


}

####
# sub accept_article
#
# Accept an article under review.
# The article will have its status changed to 'live' and a 
# post will be made to the article admin thread. The thread is
# then moved to the garbage bin. The article is posted to the
# public forum.
##
sub accept_article {

	my ($class, $user, $article_id, $noCMS) = @_;
	my $db = CyberArmy::Database->instance;
	#get article from db
	
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	my $category = CyberArmy::Library->getCategory($article->{'category_id'}) or return undef;
	my $imported_info='';
	#post to public forum
	if($article->{'imported_from'} && ($article->{'imported_from'} eq 'zzine'))
	{
		$imported_info = 'This article was imported from [url=http://www.zzine.org]zZine[/url]. <small>(original author: '.$article->{'original_author'}.')</small>';
	}
	elsif($article->{'imported_from'} && ($article->{'imported_from'} eq 'cau'))
	{
		$imported_info = 'This article was imported from the CyberArmy University site. <small>(original author: '.$article->{'original_author'}.')</small>';
	}

	my $rank = "Author";
	my $subject = '['.$article->{'category_title'}.'] '.$article->{'title'};
	my $body = 'View and vote on the article here: [url=/library/article/'.$article_id.']'.$article->{'title'}.'[/url]<br/><br/>'
		.'<hr width="90%"/><center><h3>'.$article->{'title'}.'</h3></center><table width="100%">'
		.'<tr><td valign="top">[b]Category[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'category_title'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Summary[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'summary'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Body[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'body'}.'</td></tr></table></center></td></tr></table>'
		.'<br/><br/>'.$imported_info;
	my $author = new CyberArmy::User('caID' => $article->{'author_caID'});
	my $reportid = CyberArmy::Library->postToPublic($subject, $body, $author, $rank);
	
	
	#post to admin thread indicating made live, if it exists

	$rank = "Editor";
	$subject = 'Article set to \'live\' -nt-';
	$body = '[nt]';
	CyberArmy::Library->postToAdmin($subject, $body, $user, $rank, $article->{'admin_thread'});
	
	
	#move admin thread to garbage, if it exists
	CyberArmy::Library->movePost($article->{'admin_thread'}, 'art_garb');
	#update article to 'live' and add public_thread
	$db->do('UPDATE article SET status = "live", date_released=NOW(), public_thread=? WHERE id = ?',
		undef, $reportid, $article_id);
		
	if(!$noCMS)
	{
	#inform the user
	
	#send cMS from the system account (to show it is automated)
	my $system = new CyberArmy::User('nickname' => 'system');
	if(!$system) {
		#if the system account doesn't exist (someone deleted it on their version! :o)
		#send it from the user who set it to live.
		$system = $user;
	}
	CyberArmy::Message->new(
			caID => [$author->caID],
			user => $system,
			subject => 'Article Accepted/Published ',
			body => 'An article you submitted has been accepted and it is now publically available.'
			.'<br><br>[url=/library/article/'.$article_id.'][b]'.$article->{'title'}.'[/b][/url] submitted on : '.$article->{'date_submitted'}
			.'<br><br><br>[small]Please note the article may have been modified by a CyberArmy Editor prior to publication.[/small]'
			.'<br><br>[small]Do not respond to this cMS, this is an automated message.[/small]'
		);
	}
	#log

	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'accepted article',
			msg => 'Title: '.$article->{'title'}.' Author: '.($author ? $author->showname : 'N/A'),
			action_by => $user->showname,
			action_by_caID => $user->caID
		);	

}

####
# sub movePost
#
# Calls relevant methods to move a post to a diff forum given the message id and forum id.
# Does not attempt to handle errors, if the post doesnt exist, nothing can be done
# 
# TODO: in future, return an indicator of error so a new thread can be made if this is 
# related to admin threads, as we want to log activity if possible.
##
sub movePost {
	my ($class, $thread_id, $forum_id) = @_;

	my $message = new CyberArmy::Forum::Message ( id => $thread_id,
		select => 'pid,mid,forum,thread,subject,author_rank,author,rdate,trail'
	);
	if($message) {
		$message->moveTo($forum_id);
	}

}

####
# sub edit_article
#
# Edit an article
# The article will be updated and post will be made to the 
# article admin thread.
##
sub edit_article {
	my ($class, $user, $article_id, $title, $summary, $body, $category, $project, $language) = @_;

	#update article 
	CyberArmy::Library->update_article($article_id, $title, $summary, $body, $category, $project, $language);
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	$category = CyberArmy::Library->getCategory($article->{'category_id'}) or return undef;
		
	#post to admin thread
	my $rank = "Editor";
	my $subject = 'Article Edited';
	$body = 'View and vote on the article here: [url=/library/article/'.$article_id.']'.$article->{'title'}.'[/url]<br/><br/>'
		.'<hr width="90%"/><center><h3>'.$article->{'title'}.'</h3></center><table width="100%">'
		.'<tr><td valign="top">[b]Category[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'category_title'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Summary[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'summary'}.'</td></tr></table></center></td></tr>'
		.'<tr><td valign="top">[b]Body[/b]</td><td width="100%"><center><table border="1" cellpadding="3" cellspacing="0" style="border-collapse: collapse" bordercolor="#000000" width="95%" bgcolor="#364C7D"><tr><td width="100%">'
		.$article->{'body'}.'</td></tr></table></center></td></tr></table>';
	CyberArmy::Library->postToAdmin($subject, $body, $user, $rank, $article->{'admin_thread'});
	
	
	
	#log
	my $author = new CyberArmy::User('caID' => $article->{'author_caID'});
	
	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'edited article',
			msg => 'Title: '.$article->{'title'}.' Author: '.$author->showname,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);

}

####
# sub delete_article
#
# Delete an article
# The article will have its status changed to 'deleted' and a
# post will be made to the article admin thread. The admin
# thread will be moved to the garbage bin (if not already there).
# Any public thread will be deleted.
##
sub delete_article {
	my ($class, $user, $article_id) = @_;
	
	my $db = CyberArmy::Database->instance;
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	
	#post to admin thread, if exists
	my $rank = "Editor";
	my $subject = 'Article deleted -nt-';
	my $body = '[nt]';
	CyberArmy::Library->postToAdmin($subject, $body, $user, $rank, $article->{'admin_thread'});
	
	
	#move admin thread to garbage if not already there
	CyberArmy::Library->movePost($article->{'admin_thread'}, 'art_garb');
	#update article status to 'deleted'
	$db->do('UPDATE article SET status = "deleted" WHERE id = ?',
		undef,  $article_id);
	
	#delete public thread, if exists
	CyberArmy::Library->deletePublicPost($article->{'public_thread'});
	
	#inform the user
	my $author = new CyberArmy::User('caID' => $article->{'author_caID'});
	
	#send cMS from the system account (to show it is automated)
	my $system = new CyberArmy::User('nickname' => 'system');
	if(!$system) {
		#if the system account doesn't exist (someone deleted it on their version! :o)
		#send it from the user who set it to live.
		$system = $user;
	}
	CyberArmy::Message->new(
			caID => [$author->caID],
			user => $system,
			subject => 'Article Deleted',
			body => 'An article you submitted has not been accepted and will not be made publically available. To query this decision, please contact a library admin or use the [url=/contact]Feedback[/url] system.'
			.'<br><br>[b]'.$article->{'title'}.'[/b] submitted on : '.$article->{'date_submitted'}
			.'<br><br><br>[small]Do not respond to this cMS, this is an automated message.[/small]'
	);
	#log
	
	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'deleted article',
			msg => 'Title: '.$article->{'title'}.' Author: '.$author->showname,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
}


####
# sub setPublicationOnly
#
# Sets an article to appear in publications only
# The article will have its status changed to 'publication-only' and a
# post will be made to the article admin thread. The admin
# thread will be moved to the garbage bin (if not already there).
# Any public thread will be deleted.
##
sub setPublicationOnly {
	my ($class, $user, $article_id) = @_;
	
	my $db = CyberArmy::Database->instance;
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	
	#post to admin thread, if exists
	my $rank = "Editor";
	my $subject = 'Article set to publication-only -nt-';
	my $body = '[nt]';
	CyberArmy::Library->postToAdmin($subject, $body, $user, $rank, $article->{'admin_thread'});
	
	
	#move admin thread to garbage if not already there
	if($article->{'admin_thread'}) {
		CyberArmy::Library->movePost($article->{'admin_thread'}, 'art_garb');
	}
	
	#update article status to 'publication-only'
	$db->do('UPDATE article SET status = "publication-only" WHERE id = ?',
		undef,  $article_id);
	
	#delete public thread, if exists
	if($article->{'public_thread'}) {
		CyberArmy::Library->deletePublicPost($article->{'public_thread'});
	}
	
	#inform the user
	my $author = new CyberArmy::User('caID' => $article->{'author_caID'});
	
	#send cMS from the system account (to show it is automated)
	my $system = new CyberArmy::User('nickname' => 'system');
	if(!$system) {
		#if the system account doesn't exist (someone deleted it on their version! :o)
		#send it from the user who set it to live.
		$system = $user;
	}
	CyberArmy::Message->new(
			caID => [$author->caID],
			user => $system,
			subject => 'Article set to publication-only',
			body => 'An article you submitted has been accepted to appear in publications only. This means the article will not be released publically on its own but may appear in one or more publication editions. To query this decision, please contact a library admin or use the [url=/contact]Feedback[/url] system.'
			.'<br><br>[b]'.$article->{'title'}.'[/b] submitted on : '.$article->{'date_submitted'}
			.'<br><br><br>[small]Do not respond to this cMS, this is an automated message.[/small]'
	);
	#log
	
	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'article set to \'publication-only\'',
			msg => 'Title: '.$article->{'title'}.' Author: '.$author->showname,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
}


####
# sub review_article
#
# Set an article back to 'review'
# The article will have its status changed to 'review' and a
# post will be made to the article admin thread. The admin
# thread will be moved back to the admin forum.
# Any public thread will be deleted.
##
sub review_article {
	my ($class, $user, $article_id) = @_;
	
	my $db = CyberArmy::Database->instance;
	my $article = CyberArmy::Library->getArticle($article_id) or return undef;
	
	#post to admin thread, if exists
	my $rank = "Editor";
	my $subject = 'Article returned to \'review\' status -nt-';
	my $body = '[nt]';
	CyberArmy::Library->postToAdmin($subject, $body, $user, $rank, $article->{'admin_thread'});
	#move admin thread to admin forum if not already there
	CyberArmy::Library->movePost($article->{'admin_thread'}, 'art_admin');
	#update article status to 'review'
	$db->do('UPDATE article SET status = "review" WHERE id = ?',
		undef, $article_id);
	#delete public thread, if exists
	if($article->{'public_thread'}){
		CyberArmy::Library->deletePublicPost($article->{'public_thread'});
	}
	#log
	my $author = new CyberArmy::User('caID' => $article->{'author_caID'});
	
	CyberArmy::Library->Log(
			id => $article->{'id'},
			type => 'set article to review',
			msg => 'Article returned to \'review\'. Title: '.$article->{'title'}.' Author: '.$author->showname,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
}


####
# sub CategoryName
#
# Given the id of a category, returns the full details.
##
sub CategoryName {
	my $result = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
		"SELECT * FROM article_category where id = ?",
		undef,$_[0]->type
	);
	return $result->{category_name};
}



sub AUTOLOAD {
system("echo autoload: $AUTOLOAD > /tmp/blah1");
#	$AUTOLOAD =~ s/.*://;
#	$_[0]->Field('article_'.$AUTOLOAD); 
}


####
# sub Move
#
# Move an article to a category
##
sub move {
	my $id = pop;
	my $new_cat = pop;
	my $db = CyberArmy::Database->instance();

	## update the category an article is associated with 

	$db->do( 
		'UPDATE article SET category = ? WHERE id = ?',
	undef,$new_cat,$id);

}



####
# sub getCategories
#
# Obtain list of categories, ordered by titles
##
sub getCategories {
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From article_category order by title asc"
			);
	$categories->execute();
	my $categoryList = $categories->fetchall_arrayref({});
	return $categoryList;
}





####
# sub getArticles
# 
# Obtain list of articles of specific status, by a specific user, ordered by titles
##
sub getArticles {
	my ($class, $user, $status) = @_;
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT article.id,article.title,
		date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
		date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
		article.admin_thread, article.public_thread,
		article.project_id, projects.proj_name , projects.proj_group ,
		article_category.title AS category_title , article.language_id, 
		library_language.language
		FROM article
		LEFT JOIN article_category ON article.category = article_category.id 
		LEFT JOIN projects ON projects.proj_id=article.project_id
	    LEFT JOIN library_language ON library_language.id=article.language_id
		WHERE article.author_caID = ? AND article.status=? 
		ORDER BY article.date_submitted DESC
		"
		);
	$articles->execute($user->caID, $status);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
}


####
# sub createCategory
#
# Create a category
#
# TODO: currently all categories are 'parent' categories, expand this.
##
sub createCategory {
	my ($class, $category_name, $parent, $description, $user) = @_;
	$category_name = Encode::decode_utf8($category_name);
	my $db = CyberArmy::Database->instance();
	$db->do(
		'INSERT INTO article_category (parent_id, title,description) VALUES (?,?,?)'
	,undef,$parent, $category_name, $description);
	
	#log
	
	my $insertid = $db->{'mysql_insertid'};
	CyberArmy::Library->Log(
			id => $insertid,
			type => 'created category',
			msg => 'Title: '.$category_name,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
	
}

####
# sub postToAdmin
#
# Posts a message to the Library Administration Forum
##
sub postToAdmin {
	my ($class, $subject, $body, $user, $rank, $pid) = @_;
		my $forum = CyberArmy::Forum->new( id=> "art_admin" );
		return(CyberArmy::Library->post($subject, $body, $user, $forum, $rank, $pid));

}

####
# sub postToPublic
#
# Posts a message to the Article Discussion Forum
##
sub postToPublic {
	my ($class, $subject, $body, $user, $rank, $pid) = @_;
		my $forum = CyberArmy::Forum->new( id=> "art_public" );
		return(CyberArmy::Library->post($subject, $body, $user, $forum, $rank, $pid));

}

####
# sub post
#
# Simply posts a message
##
sub post {
	my ($class, $subject, $body, $user, $forum, $rank, $pid) = @_;
	if($pid) {
		#parent thread may be on a different forum now, so make sure it goes there
		my $message = new CyberArmy::Forum::Message ( id => $pid,
				select => 'pid,mid,forum,thread,subject,author_rank,author,rdate,trail'
		);
		$forum = CyberArmy::Forum->new( id=> $message->{'forum'});
	}
	if($forum) {
		if(!$user)
		{
			$user = new CyberArmy::User('nickname' => 'system');
		}
			return ($forum->PostMessage( author => $user->showname,
					author_caID => $user->caID,
					author_rank => $rank,
					subject => $subject,
					body => $body,
					pid => $pid ? $pid : undef
			));
	}
	

}



####
# sub deleteCategory
#
# Deletes category if the category has no articles in it
##
sub deleteCategory{
	my ($class,$id, $user) = @_;
	my $category = CyberArmy::Library->getCategory($id);
	if($category!=0E0){
		
		CyberArmy::Database->instance()->do('
			DELETE FROM article_category WHERE (
			id = ? AND article_category.id NOT IN (SELECT distinct article.category FROM article WHERE status!="draft"))',
		undef, $id);

		CyberArmy::Library->Log(
				id => $category->{'id'},
				type => 'deleted category',
				msg => 'Title: '.$category->{'title'},
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}

}

####
# sub editCategory
#
# Edits (rename / alter description) a category
##
sub editCategory{ 
	my ($class,$id,$newname,$newdesc, $user) = @_;
	my $db = CyberArmy::Database->instance();
	my $category = CyberArmy::Library->getCategory($id);
		
	if($category!=0E0){
		$db->do(
		'UPDATE article_category SET title = ?, description=? WHERE id = ?'
			,undef,$newname,$newdesc, $id);
	
		#log

		CyberArmy::Library->Log(
				id => $category->{'id'},
				type => 'edited category',
				msg => 'Old name: '.$category->{'title'}.', new title: '.$newname,
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}
	
}

####
# sub listCategories
#
# Lists all non-archived categories
##
sub listCategories {
	#Delete category, if empty
		
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From article_category WHERE archived='0' order by title asc"
				);
	$categories->execute();
	my $categoryList = $categories->fetchall_arrayref({});
	return $categoryList

}

####
# sub listDeletableCategories
#
# Lists all deletable categories, i.e. categories with no articles
##
sub listDeletableCategories {
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From article_category WHERE (
		id NOT IN (SELECT distinct article.category FROM article WHERE status!='draft') 
		AND id NOT IN(SELECT distinct article_category.parent_id FROM article_category))
		order by title asc"
				);
	$categories->execute();
	my $categoryList = $categories->fetchall_arrayref({});
	return $categoryList;

}

####
# sub countCategoryArticles
#
# Counts the number of 'live' articles in a category
##
sub countCategoryArticles {
	my ($class,$category_id) = @_;
	my $countArticles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT COUNT(*) FROM article ".
		"WHERE category = ? AND status='live'"
	); $countArticles->execute($category_id) or return undef;
	my $count = $countArticles->fetchrow_array();
	return $count;

}

####
# sub countLinguaArticles
#
# Counts the number of 'live' articles in a language
##
sub countLinguaArticles {
	my ($class,$lingua_id) = @_;
	my $countArticles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT COUNT(*) FROM article ".
		"WHERE language_id = ? AND status='live'"
	); $countArticles->execute($lingua_id) or return undef;
	my $count = $countArticles->fetchrow_array();
	return $count;

}

####
# sub listCatWithUnreviewed
#
# Lists all categories with articles that need to be reviewed
##
sub listCatWithUnreviewed {
		
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From article_category WHERE (
		id IN (SELECT distinct article.category FROM article WHERE article.status='review'))
		order by title asc"
				);
	$categories->execute();
	my $categoryList = $categories->fetchall_arrayref({});
	return $categoryList

}

####
# sub getCatUnreviewedArticles
#
# Lists all articles in a particular category that are set to 'review'
##
sub getCatUnreviewedArticles {
	my ($class,$category_id) = @_;
		
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT article.id,article.title,
				date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
				date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
				users.caID,users.showname,users.nickname,
				article.admin_thread, article.public_thread,
				article.project_id, projects.proj_name , projects.proj_group  
				FROM article 
				LEFT JOIN users ON article.author_caID=users.caID 
				LEFT JOIN projects ON projects.proj_id=article.project_id
				WHERE status='review' AND category = ?
				ORDER BY article.date_submitted desc"
				);
	$articles->execute($category_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList

}

####
# sub getArticle
#
# Returns a specific article
##
sub getArticle {
	my ($class,$article_id,$user_caID) = @_;
	my $article = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
			"SELECT article.id,article.title,article.summary, article.status,
			article.body, article.admin_thread, article.public_thread, 
			article.total_rating, article.num_votes,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article_category.id AS category_id,
			article_category.title AS category_title, article.author_caID, 
			users.showname, users.nickname, article.project_id, 
			projects.proj_name , projects.proj_group, article.language_id, 
			library_language.language, imported_from, original_author
			FROM article 
			LEFT JOIN article_category ON article.category = article_category.id 
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN library_language ON library_language.id=article.language_id
			LEFT JOIN users ON article.author_caID = users.caID
			WHERE article.id=?"
			,undef,$article_id
			);
    if ($article != '0E0')
    {
		my $user_vote = undef;
		if($user_caID){
		$user_vote=$CyberArmy::Utils::Database::dbh->selectrow_hashref(
				"SELECT vote FROM article_votes
				 WHERE article_id=? AND user_caID=?"
				,undef,$article_id, $user_caID
				);
		}
		$article->{'uservote'} = $user_vote->{'vote'};
    }
	return $article
	
}

####
# sub castVote
#
# Submit user vote on an article
##

sub castVote {
	my ($class, $user, $vote, $article_id) = @_;
	if ( $user && $vote && $article_id){
		#TODO: check category exists
		my $user_vote = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
					"SELECT vote FROM article_votes
					 WHERE article_id=? AND user_caID=?"
					,undef,$article_id, $user->caID
			);
		return undef if ($user_vote && $user_vote!=0);
		
		my $db = CyberArmy::Database->instance;
		$db->do(
			"REPLACE INTO article_votes (user_caID, article_id, vote) 
			VALUES (?,?,?)",
			undef, $user->caID, $article_id, $vote);
		
		my $article = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
				"SELECT total_rating, num_votes
				FROM article 
				WHERE article.id=?"
				,undef,$article_id
		);	
		$db->do('UPDATE article SET total_rating = ?, num_votes=? WHERE id = ?',
			undef, $article->{'total_rating'}+$vote , $article->{'num_votes'}+1, $article_id,);
			

		
	} else { return undef }

}

####
# sub getCategory
#
# Returns a specific category
##
sub getCategory {
	my ($class,$category_id) = @_;
	my $category = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
			"SELECT * FROM article_category WHERE id=?"
			,undef,$category_id
			);
	$category->{'resources'} = CyberArmy::Library->listCategoryResources($category_id);
	return $category
	
}

####
# sub getLingua
#
# Returns a specific language
##
sub getLingua {
	my ($class,$lingua_id) = @_;
	my $lingua = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
			"SELECT * FROM library_language WHERE id=?"
			,undef,$lingua_id
			);
	return $lingua
	
}

####
# sub getCategoryArticles
#
# Returns 'live' articles for a specific category
##
sub getCategoryArticles {
	my ($class,$category_id, $orderby, $limit, $slimit) = @_;
	$orderby = $orderby ? "ORDER BY ".$orderby : "";
	my $limit_sql = '';
	if($limit)
	{
		$slimit= $slimit || 0 ;
		$limit_sql=' LIMIT '.$slimit.', '.$limit;
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title, article.summary, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname , article.project_id, projects.proj_name , projects.proj_group,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN article_category ON article_category.id = article.category
			WHERE article.category=? AND article.status='live' ".$orderby.$limit_sql
			);
	$articles->execute($category_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
	
}

####
# sub getLinguaArticles
#
# Returns 'live' articles for a specific lingua
##
sub getLinguaArticles {
	my ($class,$lingua_id, $orderby, $limit, $slimit) = @_;
	$orderby = $orderby ? "ORDER BY ".$orderby : "";
	my $limit_sql = '';
	if($limit)
	{
		$slimit= $slimit || 0 ;
		$limit_sql=' LIMIT '.$slimit.', '.$limit;
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title, article.summary, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname , article.project_id, projects.proj_name , projects.proj_group,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN article_category ON article_category.id = article.category
			LEFT JOIN library_language ON library_language.id = article.language_id
			WHERE article.language_id=? AND article.status='live' ".$orderby.$limit_sql
			);
	$articles->execute($lingua_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
	
}

####
# sub getProjectArticles
#
# Returns 'live' articles for a specific project
##
sub getProjectArticles {
	my ($class,$project_id, $orderby, $limit, $slimit) = @_;
	$orderby = $orderby ? "ORDER BY ".$orderby : "";
	my $limit_sql = '';
	if($limit)
	{
		$slimit= $slimit || 0 ;
		$limit_sql=' LIMIT '.$slimit.', '.$limit;
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title,article.summary, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname , article.project_id, projects.proj_name , projects.proj_group,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN article_category ON article_category.id = article.category
			WHERE article.project_id=? AND article.status='live' ".$orderby.$limit_sql
			);
	$articles->execute($project_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
	
}

####
# sub getUserArticles
#
# Returns 'live' articles submitted by a specific user
##
sub getUserArticles {
	my ($class,$user_id, $orderby, $limit, $slimit) = @_;
	$orderby = $orderby ? "ORDER BY ".$orderby : "";
	my $limit_sql = '';
	if($limit)
	{
		$slimit= $slimit || 0 ;
		$limit_sql=' LIMIT '.$slimit.', '.$limit;
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title,article.summary, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname ,article.project_id, projects.proj_name , projects.proj_group,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN article_category ON article_category.id = article.category
			WHERE article.author_caID=? AND article.status='live' ".$orderby.$limit_sql
			);
	$articles->execute($user_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
	
}


####
# sub isAdmin
#
# Used to determine if a user is an article administrator
# To update the groups that should have this access, only this 
# function requires changing.
##
sub isAdmin {
	my ($class,$user) = @_;
	if($user && ($user->IsInGroup('commander')||
		$user->IsInGroup('article_admin') ||
		$user->IsInGroup('staff'))) {
		return 1;
	}
		


}

####
# sub isEditor
#
# Used to determine if a user is an article editor
# To update the groups that should have this access, only this 
# function requires changing.
##
sub isEditor {
	my ($class,$user) = @_;
	if($user && $user->IsInGroup('article_editor')) {
		return 1;
	}
		


}

####
# sub isPublicationAdmin
#
# Used to determine if a user is an admin for one or more publications.
##
sub isPublicationAdmin {
	my ($class,$user) = @_;
	if($user){
		my $publicationList = CyberArmy::Library->listPublications();
		foreach my $publication (@$publicationList){	
			if ($user->CheckGroupList("$publication->{administrator}")){
				return 1;
			}
		}
	}

}

####
# sub deleteDraft
#
# Deletes a draft article. When a user deletes a draft, no logging is required.
##
sub deleteDraft {
	my ($class,$article_id,$user) = @_;
	
	my $db = CyberArmy::Database->instance();
	$db->do(
	"DELETE FROM article WHERE id = ? AND author_caID =? AND status='draft'"
	,undef,$article_id,$user->caID);
		


}


####
# sub Log
# 
# Write log entry for a library activity
##
sub Log {
	my $self = shift; my %tags = @_;
	
	return undef unless $tags{type};
	$tags{action_by} ||= '[LOG-DAEMON]';

	$CyberArmy::Utils::Database::dbh->do(
		'INSERT INTO log_library (item_id, type, msg, '.
		'action_by, action_by_id, adate) VALUES (?, ?, ?,'.
		'?, ?, NOW())',undef,
	$tags{id},$tags{type},$tags{msg},$tags{action_by},
	$tags{action_by_caID}) or return undef;
		
}

####
# sub getLogs
#
# Obtains last 200 library logs
##
sub getLogs {
	my $logs = $CyberArmy::Utils::Database::dbh->prepare(
            'SELECT item_id, type, msg, '.
	     'action_by, action_by_id, adate ' .
            'FROM log_library ORDER BY adate DESC limit 200');
        $logs->execute();
	 my $logList = $logs->fetchall_arrayref({});
	return $logList

}

####
# sub countProjectArticles
#
# Counts the number of 'live' articles related to a project
##
sub countProjectArticles {
	my ($class,$project_id) = @_;
	my $countArticles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT COUNT(*) FROM article ".
		"WHERE project_id = ? AND status='live'"
	); $countArticles->execute($project_id) or return undef;
	my $count = $countArticles->fetchrow_array();
	return $count;

}

####
# sub countUserArticles
#
# Counts the number of 'live' articles submitted by a user
##
sub countUserArticles {
	my ($class,$user_id) = @_;
	my $countArticles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT COUNT(*) FROM article ".
		"WHERE author_caID = ? AND status='live'"
	); $countArticles->execute($user_id) or return undef;
	my $count = $countArticles->fetchrow_array();
	return $count;

}


####
# sub deletePublicPost
#
# Delete A Post #
##
sub deletePublicPost {
	my ($class,$pid) = @_;

	my $post = new CyberArmy::Forum::Message ( id => $pid,
				select => 'pid,mid,forum,thread,subject,author_rank,author,rdate,trail'
		);
		
	if($post){
		my $forum = $post->GetForum('forum,administrator,moderator,name');

		#we presume this is a top level thread, as that should be
		#the only thing we delete using this method
		my ($rows,$tpthread) = (0,0); ## Set Scope...
		
		my $kill = $CyberArmy::Utils::Database::dbh->prepare(q~
			DELETE FROM forum_replies
			WHERE thread = ?
		~); $kill->execute($post->{thread});
		$kill->finish;
		$rows = $kill->rows; $tpthread++;
		$CyberArmy::Utils::Database::dbh->do(q~
			INSERT INTO log_forum 
			(forum, action, thread_id, subject, author, rdate, logby, logby_id, ldate, reason) 
			VALUES (?,'delete',?,?,?,?,?,?,NOW(),?)~,undef,
		$forum->{forum}, $post->{mid}, $post->{subject},
			$post->{author_rank}.' '.$post->{author},
			$post->{rdate}, 'System', '0000001', 'Article Removed');	


		my $last_post = $CyberArmy::Utils::Database::dbh
		->selectrow_hashref(q~
			SELECT MAX(mid) as max_mid FROM forum_replies WHERE forum = ?
		~,undef,$forum->{forum});

		 $CyberArmy::Utils::Database::dbh->do(qq~
			UPDATE forum_list
			SET last_modified = NOW(),
			threads_num = threads_num - $tpthread,
			posts_num = posts_num - $rows,
			 last_post_id = ?
			WHERE forum = ?
		~,undef,$last_post->{max_mid},$forum->{forum});
	}
		
	
}



####
# sub getCategories2
#
#
##
sub getSubCategories {
	my ($class,$parent_id, $depth) = @_;
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From article_category WHERE parent_id = ? order by title asc"
			);
	$categories->execute($parent_id);
	my $categoryList = $categories->fetchall_arrayref({});
	foreach my $category (@$categoryList){		
		my $subCategories = CyberArmy::Library->getSubCategories($category->{id}, $depth+1);
		$category->{sub_categories} = $subCategories;
		$category->{depth} = $depth;
		my $articleCount = CyberArmy::Library->countCategoryArticles($category->{id});
		$category->{articleCount} = $articleCount;
	}
	return $categoryList;
}

####
# sub getLanguages
#
#
##
sub getLanguages {
	my $languages = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From library_language order by id asc"
			);
	$languages->execute();
	my $languageList = $languages->fetchall_arrayref({});
	foreach my $language (@$languageList){		
		my $articleCount = CyberArmy::Library->countLinguaArticles($language->{id});
		$language->{articleCount} = $articleCount;
	}
	return $languageList;
}


####
# sub getParentCategories
#
#
##
sub getParentCategories {
	my ($class,$parent_id, $depth) = @_;
	my $categories = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT * From article_category WHERE id = ? order by title asc"
			);
	$categories->execute($parent_id);
	my $categoryList = $categories->fetchall_arrayref({});
	foreach my $category (@$categoryList){		
		my $parentCategories = CyberArmy::Library->getParentCategories($category->{parent_id}, $depth+1);
		$category->{parent_categories} = $parentCategories;
		$category->{depth} = $depth;
	}
	return $categoryList;
}

sub GetLatest {
	my ($class,$limit,$category_id) = @_;
	my $articles;
	if($category_id){
		my $articles = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT article.total_rating, article.num_votes,
				article.id,article.title,article.summary,
				date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
				users.showname , users.nickname , article_category.title AS category_title
				FROM article 
				LEFT JOIN users ON article.author_caID = users.caID
				LEFT JOIN article_category ON article_category.id = article.category
				LEFT JOIN library_language ON library_language.id = article.language_id
				WHERE article.category=? AND article.status='live' ORDER BY article.date_released DESC LIMIT ".$limit
				);
		$articles->execute($category_id);
		my $articleList = $articles->fetchall_arrayref({});
		return $articleList;	
	}	
	else {
		my $articles = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT article.total_rating, article.num_votes,
				article.id, article.title,article.summary,
				date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
				users.showname , users.nickname , article_category.title AS category_title
				FROM article 
				LEFT JOIN users ON article.author_caID = users.caID
				LEFT JOIN article_category ON article_category.id = article.category
				LEFT JOIN library_language ON library_language.id = article.language_id
				WHERE article.status='live' ORDER BY article.date_released DESC LIMIT ".$limit
				);
		$articles->execute();	
		my $articleList = $articles->fetchall_arrayref({});
		return $articleList;
	}
}


####
# sub createPublication
#
# Create a publication
#
##
sub createPublication {
	my ($class, $name, $description, $user) = @_;
	my $db = CyberArmy::Database->instance();
	$db->do(
		'INSERT INTO publication (title,description, administrator) VALUES (?,?, ?)'
	,undef,$name, $description, "");
	
	#log
	
	my $insertid = $db->{'mysql_insertid'};
	CyberArmy::Library->Log(
			id => $insertid,
			type => 'created publication',
			msg => 'Title: '.$name,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
	
}

####
# sub createEdition
#
# Create an edition
#
##
sub createEdition {
	my ($class, $publication_id, $name, $description, $user) = @_;
	my $db = CyberArmy::Database->instance();
	$db->do(
		'INSERT INTO publication_edition (publication_id, title,description) VALUES (?,?, ?)'
	,undef,$publication_id, $name, $description);
	
	#log
	
	my $insertid = $db->{'mysql_insertid'};
	CyberArmy::Library->Log(
			id => $insertid,
			type => 'created publication edition',
			msg => 'Title: '.$name,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
	
}

####
# sub getPublication
#
# Gets publication details and all editions within it
##
sub getPublication {
	my ($class,$publication_id) = @_;
	
	my $publication = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
				"SELECT * From publication WHERE id = ?"
			,undef,$publication_id
			);
	my $editions = CyberArmy::Library->listPublicationEditions($publication->{id});
	$publication->{editions} = $editions;
	return $publication

}

####
# sub listPublications
#
# Lists all publications
##
sub listPublications {
		
	my $publications = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From publication order by title asc"
				);
	$publications->execute();
	my $publicationList = $publications->fetchall_arrayref({});
	foreach my $publication (@$publicationList){		
		my $editions = CyberArmy::Library->listPublicationEditions($publication->{id});
		$publication->{editions} = $editions;
	}
	return $publicationList

}

####
# sub listDeletablePublications
#
# Lists all deletable publications, i.e. publications with no editions
##
sub listDeletablePublications {
	my $publications = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From publication WHERE (
		publication.id NOT IN (SELECT distinct publication_edition.publication_id FROM publication_edition))
		order by title asc"
				);
	$publications->execute();
	my $publicationList = $publications->fetchall_arrayref({});
	return $publicationList

}

####
# sub deletePublication
#
# Deletes publication if the publication does not have any editions
##
sub deletePublication {
	my ($class,$id, $user) = @_;
	my $publication = CyberArmy::Library->getPublication($id);
	if($publication!=0E0){
		
		CyberArmy::Database->instance()->do('
			DELETE FROM publication WHERE (
			id = ? AND publication.id NOT IN (SELECT distinct publication_edition.publication_id FROM publication_edition))',
		undef, $id);

		CyberArmy::Library->Log(
				id => $publication->{'id'},
				type => 'deleted publication',
				msg => 'Title: '.$publication->{'title'},
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}

}

####
# sub listPublicPublications
#
# Lists all public publications
##
sub listPublicPublications {
		
	my $publications = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT distinct publication.id, publication.title, publication.administrator From publication 
			 inner join publication_edition ON publication_edition.publication_id = publication.id 
			WHERE publication_edition.status='published'
			order by title asc"
				);
	$publications->execute();
	my $publicationList = $publications->fetchall_arrayref({});
	foreach my $publication (@$publicationList){		
		my $editions = CyberArmy::Library->listPublicPublicationEditions($publication->{id});
		$publication->{editions} = $editions;
	}
	return $publicationList

}



####
# sub getEdition
#
# Gets edition details and all articles within it
##
sub getEdition {
	my ($class,$edition_id,$user) = @_;
	
	my $edition = $CyberArmy::Utils::Database::dbh->selectrow_hashref(
				"select a.*,b.administrator from publication_edition a, "
				."publication b WHERE a.id=? AND a.publication_id = b.id"
			,undef,$edition_id
			);
	if($edition!='0E0'){
		my $admin=undef;
		if($user){
			if(CyberArmy::Library->isAdmin($user) || $user->CheckGroupList("$edition->{administrator}")){
				$admin='true';			
			}
		}
		my $articles = CyberArmy::Library->getEditionArticles($edition->{id},$admin);
		$edition->{articles} = $articles;
	}
	return $edition

}

####
# sub listPublicationEditions
#
# Lists all editions for a publication
##
sub listPublicationEditions {
	my ($class,$publication_id) = @_;
		
	my $editions = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From publication_edition WHERE publication_id = ? order by id desc"
				);
	$editions->execute($publication_id);
	my $editionList = $editions->fetchall_arrayref({});
	return $editionList

}

####
# sub listPublicPublicationEditions
#
# Lists all released editions for a publication
##
sub listPublicPublicationEditions {
	my ($class,$publication_id) = @_;
		
	my $editions = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT * From publication_edition 
				WHERE publication_id = ? AND status='published'
				order by id desc"
				);
	$editions->execute($publication_id);
	my $editionList = $editions->fetchall_arrayref({});
	return $editionList

}

####
# sub listEditions
#
# Lists all editions
##
sub listEditions {
	my ($class) = @_;
		
	my $editions = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT publication_edition.id, publication_edition.title, publication.administrator From publication_edition 
			LEFT JOIN publication ON publication.id = publication_edition.publication_id
			order by id desc"
				);
	$editions->execute();
	my $editionList = $editions->fetchall_arrayref({});
	return $editionList

}

####
# sub listDeletableEditions
#
# Lists all deletable editions, i.e. editions with no articles
##
sub listDeletableEditions {
	my ($class, $publication_id) = @_;
	my $editions = $CyberArmy::Utils::Database::dbh->prepare(
				"SELECT publication_edition.id, publication_edition.title, publication.administrator From publication_edition 
			LEFT JOIN publication ON publication.id = publication_edition.publication_id WHERE (
		publication_edition.id NOT IN (SELECT distinct edition_article.edition_id FROM edition_article))
		AND publication_edition.publication_id =? order by title asc "
				);
	$editions->execute($publication_id);
	my $editionList = $editions->fetchall_arrayref({});
	return $editionList

}


####
# sub deleteEdition
#
# Deletes edition if the edition does not have any articles
##
sub deleteEdition {
	my ($class,$id, $user) = @_;
	my $edition = CyberArmy::Library->getEdition($id);
	if($edition!=0E0){
		
		CyberArmy::Database->instance()->do('
			DELETE FROM publication_edition WHERE (
			publication_edition.id = ? AND publication_edition.id NOT IN (SELECT distinct edition_article.edition_id FROM edition_article))',
		undef, $id);

		CyberArmy::Library->Log(
				id => $edition->{'id'},
				type => 'deleted edition',
				msg => 'Title: '.$edition->{'title'},
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}

}


####
# sub addArticleToEdition
#
# Adds an article to an edition
#
##
sub addArticleToEdition {
	my ($class, $edition_id, $article_id, $weight, $user) = @_;
	my $db = CyberArmy::Database->instance();
	$db->do(
		'INSERT INTO edition_article (edition_id, article_id,weight) VALUES (?,?, ?)'
	,undef,$edition_id, $article_id, $weight);
	
	#log
	
	my $insertid = $db->{'mysql_insertid'};
	CyberArmy::Library->Log(
			id => $insertid,
			type => 'added article to edition',
			msg => 'Article: '.$article_id.', edition: '.$edition_id,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
	
}

####
# sub addRemoveFromEdition
#
# Removes an article to an edition
#
##
sub removeFromEdition {
	my ($class, $edition_id, $article_id, $user) = @_;
	my $db = CyberArmy::Database->instance();
	$db->do(
		'DELETE FROM edition_article WHERE edition_id = ? AND  article_id = ?'
	,undef,$edition_id, $article_id);
	
	#log
	
	CyberArmy::Library->Log(
			id => $article_id,
			type => 'removed article from edition',
			msg => 'Article: '.$article_id.', edition: '.$edition_id,
			action_by => $user->showname,
			action_by_caID => $user->caID
	);
	
}


####
# sub editPublication
#
# Edits (rename / alter description) a publication
##
sub editPublication{ 
	my ($class,$id,$newname,$newdesc, $admin, $user) = @_;
	my $db = CyberArmy::Database->instance();
	my $publication = CyberArmy::Library->getPublication($id);
		
	if($publication!=0E0){
		$db->do(
		'UPDATE publication SET title = ?, description=?, administrator= ? WHERE id = ?'
			,undef,$newname,$newdesc, $admin, $id);
	
		#log

		CyberArmy::Library->Log(
				id => $publication->{'id'},
				type => 'edited publication',
				msg => 'Old title: '.$publication->{'title'}.', new title: '.$newname,
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}
	
}

####
# sub getEditionArticles
#
# Returns articles within an edition
##
sub getEditionArticles {
	my ($class,$edition_id, $admin) = @_;
	my $status = '';
	if(!$admin){
		#commented out as admins should see what a normal user will see
		#$status="AND (status='live' OR status='publication-only')";
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title,article.summary, article.status, edition_article.weight, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname ,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN edition_article ON edition_article.article_id=article.id
			LEFT JOIN article_category ON article_category.id = article.category
			WHERE edition_article.edition_id=? ".$status." ORDER BY edition_article.weight DESC"
			);
	$articles->execute($edition_id);
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList	
}



####
# sub editEdition
#
# Edits (rename / alter description) an edition
##
sub editEdition{ 
	my ($class,$id,$newname,$newdesc,$user) = @_;
	my $db = CyberArmy::Database->instance();
	my $edition = CyberArmy::Library->getEdition($id);
		
	if($edition!=0E0){
		$db->do(
		'UPDATE publication_edition SET title = ?, description=? WHERE id = ?'
			,undef,$newname,$newdesc,$id);	
		#log
		CyberArmy::Library->Log(
				id => $edition->{'id'},
				type => 'edited edition',
				msg => 'Old title: '.$edition->{'title'}.', new title: '.$newname,
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}	
}


####
# sub editEditionStatus
#
# Edits the status of an edition
##
sub editEditionStatus{ 
	my ($class,$id,$newstatus,$user) = @_;
	my $db = CyberArmy::Database->instance();
	my $edition = CyberArmy::Library->getEdition($id);
		
	if($edition!=0E0){
		$db->do(
		'UPDATE publication_edition SET status=? WHERE id = ?'
			,undef,$newstatus, $id);
		#log
		CyberArmy::Library->Log(
				id => $edition->{'id'},
				type => 'edited edition status',
				msg => 'Old status: '.$edition->{'status'}.', new status: '.$newstatus,
				action_by => $user->showname,
				action_by_caID => $user->caID
		);
	}
}


####
# sub isInEdition
#
#
##
sub isInEdition{
	my ($class,$article_id, $edition_id) = @_;
	my $count =  CyberArmy::Database->instance()->selectrow_arrayref(
		'SELECT COUNT(*) FROM edition_article WHERE article_id = ? AND edition_id = ?',
	undef,,$article_id, $edition_id)->[0];
	return $count	
}

####
# sub countPublicationArticles
#
# Counts the number of 'publication-only' articles
##
sub countPublicationArticles {
	my $countArticles = $CyberArmy::Utils::Database::dbh->prepare(
		"SELECT COUNT(*) FROM article ".
		"WHERE status='publication-only'"
	); $countArticles->execute() or return undef;
	my $count = $countArticles->fetchrow_array();
	return $count;

}



####
# sub getPublicationArticles
#
# Returns 'publication-only' articles
##
sub getPublicationArticles {
	my ($class, $orderby, $limit, $slimit) = @_;
	$orderby = $orderby ? "ORDER BY ".$orderby : "";
	my $limit_sql = '';
	if($limit)
	{
		$slimit= $slimit || 0 ;
		$limit_sql=' LIMIT '.$slimit.', '.$limit;
	}
	my $articles = $CyberArmy::Utils::Database::dbh->prepare(
			"SELECT article.id,article.title,article.summary, 
			article.body, article.admin_thread, article.public_thread,
			date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
			date_format(article.date_released, '%d-%b-%Y %T') AS date_released,
			article.author_caID, 
			users.showname , users.nickname ,article.project_id, projects.proj_name , projects.proj_group,
			article_category.title AS category_title, article_category.id AS category_id
			FROM article 
			LEFT JOIN users ON article.author_caID = users.caID
			LEFT JOIN projects ON projects.proj_id=article.project_id
			LEFT JOIN article_category ON article_category.id = article.category
			WHERE article.status='publication-only' ".$orderby.$limit_sql
			);
	$articles->execute();
	my $articleList = $articles->fetchall_arrayref({});
	return $articleList
	
}



####
# sub listLanguages
#
# Returns the list of languages
##
sub listLanguages{
	my $getlanguages= $CyberArmy::Utils::Database::dbh->prepare('SELECT * FROM library_language');
	$getlanguages->execute() or return undef;
	my $languages = $getlanguages->fetchall_arrayref({});

	return $languages;
}

####
# sub addLanguage
#
# Adds a new language to the library language list. 
# Returns the new id.
##
sub addLanguage{
	my ($class, $language) = @_;
	$language = Encode::decode_utf8($language);
	$CyberArmy::Utils::Database::dbh->do(
		'INSERT INTO library_language (language) VALUES (?)'
	,undef,$language); 
	$CyberArmy::Utils::Database::dbh->{'mysql_insertid'};
}

####
# sub delLanguage
#
# Deletes a language from the library language list. 
# Currently not used in order to protect articles from having incorrect language association
# (in future functionality can be added to default the articles to english if no language association
# and/or only allow deleting languages that are not yet used).
##
sub delLanguage{ shift;
	$CyberArmy::Utils::Database::dbh->do(
		'DELETE FROM library_language WHERE id = ?',
	undef,@_);
}




####
# sub listCategoryResources
#
# Lists resources relating to a category
#
# Parameters:
#	category		- The category to get resources for
##
sub listCategoryResources{
	my ($class, $category) = @_;
	my $getresources= $CyberArmy::Utils::Database::dbh->prepare('SELECT * FROM category_resource WHERE category=?');
	$getresources->execute($category) or return undef;
	
	my $resources = $getresources->fetchall_arrayref({});
	return $resources;
}


####
# sub addResource
#
# Adds a resource to a category
#
# Parameters:
#   url				- The url for the resource
#	category		- The category the resource is for
#   title 			- The title for the resource
#   description		- The description (optional)
##
sub addResource{
	my ($class, $url,$category,$title,$description) = @_;
	$CyberArmy::Utils::Database::dbh->do(
		'INSERT INTO category_resource (url,category,title,description,date_added) VALUES (?,?,?,?,NOW())'
	,undef,$url,$category,$title,$description); 
	$CyberArmy::Utils::Database::dbh->{'mysql_insertid'};
}


####
# sub delResource
#
# Deletes a resource from a category. 
# 
# Parameters:
#	resource		- The id of the resource to delete
#	category		- The category to get resources for
##
sub delResource{ 
	my ($class, $resource,$category) = @_;
	$CyberArmy::Utils::Database::dbh->do(
		'DELETE FROM category_resource WHERE id = ? AND category=?',
	undef,$resource,$category);
}



package CyberArmy::Library::Search;

use CyberArmy::Database;
use base qw(CyberArmy::Database::FetchIter);

sub new {
	my $class = shift;
	my $args = shift;

	my $db = CyberArmy::Database->instance();

	## Sanity Check
	return undef unless $args->{'terms'};

	my @opts;
	
	my @fields; ## Specify what fields are to be searched
	if ($args->{'in_title'}) { push @fields, 'article.title' }
	if ($args->{'in_summ'}) { push @fields, 'article.summary' }
	if ($args->{'in_body'}) { push @fields,'article.body' }
	if ($args->{'in_author'}) { push @fields,'users.nickname'}
	return undef unless ($#fields > -1);

	my $opts = join (' AND ',@opts); $opts && ($opts .= ' AND ');

	my $sqlquery .= "SELECT article.id,article.title,
		date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
		date_format(article.date_submitted, '%d-%b-%Y %T') AS date_submitted,
				users.caID,users.showname,users.nickname,
		article_category.title AS category_title , category as category_id, article.language_id, 
		library_language.language
		FROM article
		LEFT JOIN users ON article.author_caID = users.caID
		LEFT JOIN article_category ON article.category = article_category.id 
	    LEFT JOIN library_language ON library_language.id=article.language_id
	     WHERE article.status!='draft' AND article.status!='review' AND article.status!='deleted' AND ".$opts." (";

	## Terms search	
	if ($args->{'match'} eq 'All') {
		my @terms = split / /,$args->{'terms'};
		my $j; foreach (@fields) {
			$j++; $sqlquery .= ' OR ' if ($j > 1);
			my $i; foreach my $term (@terms) { 
				$i++; my $term = $db->quote($term); $term =~ s/'(.*)'/'%$1%'/;
				$sqlquery .= "($_ like $term)";
				$sqlquery .= ' AND ' if ($i <= $#terms);
			}
		}

	} elsif ($args->{'match'} eq 'Any') {
		my @terms = split / /,$args->{'terms'};
		my $j; foreach (@fields) {
			$j++; $sqlquery .= ' OR ' if ($j > 1);
			my $i; foreach my $term (@terms) {
				$i++; my $term = $db->quote($term); $term =~ s/'(.*)'/'%$1%'/;
				$sqlquery .= "($_ like $term)";
				$sqlquery .= ' OR ' if ($i <= $#terms);
			}
		}

	} elsif ($args->{'match'} eq 'Exact') {
		my $i; foreach (@fields) { 
			$i++; my $term = $db->quote($args->{'terms'});
			$term =~ s/'(.*)'/'%$1%'/;
			$sqlquery .= "($_ like $term)";
			$sqlquery .= ' OR ' if ($i <= $#fields);
		}

	} else { return undef  }

	$sqlquery .= ')';

	##Date parameters
	if ($args->{'date'}) {
		if ($args->{'date'} eq 'before') {
			my $date = $args->{'year_1'}.'-'.
				$args->{'month_1'}.'-'.$args->{'day_1'};
			$sqlquery .= ' AND (date_submitted < '.$db->quote($date).')';
		} elsif ($args->{'date'} eq 'after') {
			my $date = $args->{'year_1'}.'-'.
				$args->{'month_1'}.'-'.$args->{'day_1'};
			$sqlquery .= ' AND (date_submitted > '.$db->quote($date).')';		
		} elsif ($args->{'date'} eq 'between') {
			my $date = $args->{'year_1'}.'-'.
				$args->{'month_1'}.'-'.$args->{'day_1'};
			my $date2 = $args->{'year_2'}.'-'.
				$args->{'month_2'}.'-'.$args->{'day_2'};
			$sqlquery .= ' AND (date_submitted > '.$db->quote($date).
						') AND (date_submitted < '.$db->quote($date2).') ';		
		}
	}
	
	if($args->{'category'})
	{
		$sqlquery .= ' AND category = '.$db->quote($args->{'category'}).' ';
	}

	## Final touch
	$sqlquery .= ' ORDER BY date_submitted DESC';

	## Limits?
	if ($args->{'num'} =~ /^\d+$/ && $args->{'num'} <= 100) {
		$sqlquery .= ' LIMIT 0,'.$args->{'num'};
	}
	
	## Finaly execute the query
	my $self = $db->prepare($sqlquery);

	$self->execute() ? 
		bless ($self,$class) : undef;
}
1;
