#!/usr/bin/perl -wW

## configure/setup dinah

use strict;
use File::Find ();
use File::Spec ();

eval { require Module::Build };
die "install Module::Build first\n" if $@;

eval { require CyberArmy };
if($@) {
	die "forgot to run make install first? ($@)\n";
} else { 
	require CyberArmy::Database;
	require CyberArmy::Template;
}

$ENV{'HOME'} = $CyberArmy::Config{'DINAH_BASE'};

my $build = Module::Build->new(
	dist_name => 'dinah', dist_version=>'1'
);

my $tt2 = CyberArmy::Template->instance;

my $conf_dir = File::Spec->catfile($CyberArmy::Config{'DINAH_BASE'},'conf');
my %conf = (
	httpd => File::Spec->catfile($conf_dir,'httpd.conf'),
	mycnf => File::Spec->catfile($conf_dir,'my.cnf'),
	mysql => File::Spec->catfile($conf_dir,'scheme.sql'),
	mysql_bootstrap => File::Spec->catfile($conf_dir,'bootstrap.sql'),
	crontab => File::Spec->catfile($conf_dir,'dinah.cron')
); main->$_ foreach (sort keys %conf);

print "config completed\n";

sub httpd {
	if (-f $conf{'httpd'}) { 
		return 0 unless 
			$build->y_n('Overwrite current httpd.conf?','n')
	}
	
	my %httpd_config = ## get compile time values
		map { /^\s+-D (\w+)="(.*)"$/ ? ($1,$2) : () }
	( `$CyberArmy::Config{HTTPD} -V` );

	my %httpd_modules = ## what modules are compiled in?
		map { /^\s+(.*)\.c$/ ? ($1,1) : () }
	( `$CyberArmy::Config{HTTPD} -l` );

	my %modules; if ($httpd_modules{'mod_so'}) {
		my $apxs = File::Spec->catfile(
			(File::Spec->splitpath($CyberArmy::Config{HTTPD}))[0,1],'apxs');
		$apxs = $build->prompt('Where is your apxs?', $apxs)
		while (
			not -f $apxs or (
				## the chosen apxs should match our apache
				$_ = `$apxs -q SBINDIR TARGET` && s/\s/\//g
					&& $_ ne $CyberArmy::Config{'HTTPD'}
			)
		);

		my %libexec_dirs;
		if (open (CONF,File::Spec->catfile(
				$httpd_config{'HTTPD_ROOT'},
				$httpd_config{'SERVER_CONFIG_FILE'}))
		) { ## try to guess some more libexec dirs
			while(<CONF>) { 
				/LoadModule\s+\w+\s+(.*)$/
					&& File::Spec->file_name_is_absolute($1)
					&& $libexec_dirs{$1}++;
			}
			close (CONF);
		}

		$libexec_dirs{$_}++ while ( $_ = $build->prompt(
			scalar keys %libexec_dirs ? ('Any other places to look in?')
			: ('Where is your apache LIBEXECDIR',`$apxs -q LIBEXECDIR`))
		);

		File::Find::find(
			## some distros uses libexec subs
			## to place some of their modules
			sub { 
				$modules{$1} = $File::Find::name 
					if (-f && /(\w+)\.so$/) 
			}, (keys %libexec_dirs)
		);
	}

	$CyberArmy::Config{'USER'} = $build->prompt(
		'User to run as',$> ? getpwuid($>) : 'nobody');
	$CyberArmy::Config{'PORT'} = $build->prompt(
		'Port to bind to',$> ? 8080 : 80);

	$CyberArmy::Config{'PIDLOG'} = $build->prompt(
		'Pid file',$> ? File::Spec->catfile(File::Spec->tmpdir()
		,'dinah.pid') : $httpd_config{'DEFAULT_PIDLOG'});
	$CyberArmy::Config{'ERROR_LOG'} = $build->prompt(
		'Error file',$> ? File::Spec->catfile(File::Spec->tmpdir()
		,'dinah-error.log') : $httpd_config{'DEFAULT_ERRORLOG'});

	$CyberArmy::Config{'PARALLEL'} = $build->prompt(
		'How many instances do you want to run?',1);

	my @gpgspec = split(/ /, `whereis gpg`);
	my $gpg;
	do { $gpg = $build->prompt('Where is your gpg?', $gpgspec[1]); } while (not -f $gpg);

	warn $tt2->process(
		File::Spec->catfile('conf','httpd.conf.tmpl'),{
			dinah => \%CyberArmy::Config,
			httpd => \%httpd_config,
			httpmodule => \%httpd_modules,
			modules => \%modules,
			gpgpath => $gpg
		},$conf{'httpd'}) ? "Written $conf{httpd}\n" : $tt2->error();
}

sub mycnf {
	if (-f $conf{'mycnf'}) { 
		return 0 unless 
			$build->y_n('Overwrite current my.cnf?','n')
	}
	
	warn $tt2->process(
		File::Spec->catfile('conf','my.cnf.tmpl'),{
			host => $build->prompt('MySQL hostname?','localhost'),
			user => $build->prompt('MySQL username?','cyberarmy'),
			pass => $build->prompt('MySQL password?','')
		},$conf{'mycnf'}) ? "Written $conf{mycnf}\n" : $tt2->error();
}

sub mysql {
	return 0 unless 
		$build->y_n('Sync your database schema?','n');

	eval { require MySQL::Diff }; if ($@) { 
		warn "You need to install MySQL::Diff first, skipping!\n";
	} else {
		## make sure we can connect to the mysql server
		CyberArmy::Database->instance;

		unlink (my $cnf = File::Spec
			->catfile($ENV{'HOME'},'.my.cnf'));
		## hacky, but needed
		link ($conf{'mycnf'},$cnf)
			or warn "Can't create $cnf";

		## silence uneeded verbose output
		open(OLDERR,'>&STDERR') && close(STDERR);

		## generate the diff
		my $diffs = MySQL::Diff::diff_dbs({tolerant=>1},
			MySQL::Database->new(db => 'cyberarmy' ),
			MySQL::Database->new(file => $conf{'mysql'})
		);

		## restore STDERR
		open (STDERR,'>&OLDERR') && close(OLDERR);
	
		if ($diffs) {
			print $diffs;
			if ($build->y_n('shall we proceed?','n')) {
				open  MYSQL, '|mysql -B -D cyberarmy';
				print MYSQL $diffs;
				close MYSQL;
			}
		} else { print "already in sync, skipping!\n" }
		
		unlink $cnf; ## cleanup
	}
}

sub mysql_bootstrap {
	my $db = CyberArmy::Database->instance;

	return 0 unless
		$build->y_n('Bootstrap your database?',
			$db->selectrow_array('SELECT count(*) from users') ? 'n' : 'y');

	open (CONF, $conf{'mysql_bootstrap'});
	/^\w/ and $db->do($_) while( <CONF> );
	close (CONF);
}

sub crontab {
	## FIXME: need to automate this
	warn "FIXME: You need to manually install $conf{'crontab'} at the moment";
}

